/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#ifndef __CNIC_GLOBAL_H__
#define __CNIC_GLOBAL_H__

#include <vmkapi.h>


/*
 * \brief Binary IPv4 address in network byte order.
 */
typedef struct vmk_SocketIPAddressAddr {
   vmk_uint32 s_addr;
} vmk_SocketIPAddressAddr;

/*
 * \brief An IPv4-style socket address.
 */
typedef struct vmk_SocketIPAddress {
   /* \brief Address length. Should be sizeof(vmk_SocketIPAddress) */
   vmk_uint8                 sin_len;

   /* \brief Address family. Should be VMK_SOCKET_AF_INET */
   vmk_uint8                 sin_family;

   /* \brief IP port in network byte order */
   vmk_uint16                sin_port;

   /* \brief Binary IP address in network byte order */
   vmk_SocketIPAddressAddr   sin_addr;

   /* \brief Padding. This area should be zeroed. */
   vmk_uint8                 sin_zero[8];
} VMK_ATTRIBUTE_PACKED vmk_SocketIPAddress;

/*
 * \brief Binary IPv6 address in network byte order.
 */
typedef struct vmk_SocketIPv6AddressAddr {
   union {
      vmk_uint8   __u6_addr8[16];
      vmk_uint16  __u6_addr16[8];
      vmk_uint32  __u6_addr32[4];
   } __u6_addr;
} VMK_ATTRIBUTE_PACKED vmk_SocketIPv6AddressAddr;


/*
 * \brief An IPv6-style socket address.
 */
typedef struct vmk_SocketIPv6Address {
   /* \brief Address length. Should be sizeof(vmk_SocketIPv6Address) */
   vmk_uint8                 sin6_len;

   /* \brief Address family. Should be VMK_SOCKET_AF_INET6 */
   vmk_uint8                 sin6_family;

   /* \brief IP port in network byte order */
   vmk_uint16                sin6_port;

   /* \brief Flow Info in network byte order */
   vmk_uint32                sin6_flowinfo;

   /* \brief Binary IP address in network byte order */
   vmk_SocketIPv6AddressAddr sin6_addr;

   /* \brief Scope ID in network byte order */
   vmk_uint32                sin6_scopeid;
} VMK_ATTRIBUTE_PACKED vmk_SocketIPv6Address;


/* GPL'ed defines */

#define	PCI_COMMAND		0x04    /* 16 bits */
#define	PCI_COMMAND_MASTER	0x4     /* Enable bus mastering */

typedef int pci_power_t;

#define PCI_D0          ((pci_power_t ) 0)
#define PCI_D1          ((pci_power_t ) 1)
#define PCI_D2          ((pci_power_t ) 2)
#define PCI_D3hot       ((pci_power_t ) 3)
#define PCI_D3cold      ((pci_power_t ) 4)
#define PCI_UNKNOWN     ((pci_power_t ) 5)
#define PCI_POWER_ERROR ((pci_power_t ) -1)


#define PCI_CAP_ID_PM		0x01    /* Power Management */
#define PCI_PM_CTRL		4	/* PM control and status register */
#define PCI_PM_CTRL_STATE_MASK	0x0003	/* Current power state (D0 to D3) */
#define PCI_PM_CTRL_PME_STATUS	0x8000	/* PME pin status */
#define PCI_PM_CTRL_PME_ENABLE	0x0100	/* PME pin enable */

enum {
	NETIF_MSG_DRV           = 0x0001,
	NETIF_MSG_PROBE         = 0x0002,
	NETIF_MSG_LINK          = 0x0004,
	NETIF_MSG_TIMER         = 0x0008,
	NETIF_MSG_IFDOWN        = 0x0010,
	NETIF_MSG_IFUP          = 0x0020,
	NETIF_MSG_RX_ERR        = 0x0040,
	NETIF_MSG_TX_ERR        = 0x0080,
	NETIF_MSG_TX_QUEUED     = 0x0100,
	NETIF_MSG_INTR          = 0x0200,
	NETIF_MSG_TX_DONE       = 0x0400,
	NETIF_MSG_RX_STATUS     = 0x0800,
	NETIF_MSG_PKTDATA       = 0x1000,
	NETIF_MSG_HW            = 0x2000,
	NETIF_MSG_WOL           = 0x4000,
};

#define SUPPORTED_10baseT_Half		(1 << 0)
#define SUPPORTED_10baseT_Full		(1 << 1)
#define SUPPORTED_100baseT_Half		(1 << 2)
#define SUPPORTED_100baseT_Full		(1 << 3)
#define SUPPORTED_1000baseT_Half	(1 << 4)
#define SUPPORTED_1000baseT_Full	(1 << 5)
#define SUPPORTED_Autoneg		(1 << 6)
#define SUPPORTED_TP			(1 << 7)
#define SUPPORTED_AUI			(1 << 8)
#define SUPPORTED_MII			(1 << 9)
#define SUPPORTED_FIBRE			(1 << 10)
#define SUPPORTED_BNC			(1 << 11)
#define SUPPORTED_10000baseT_Full	(1 << 12)
#define SUPPORTED_Pause			(1 << 13)
#define SUPPORTED_Asym_Pause		(1 << 14)
#define SUPPORTED_2500baseX_Full	(1 << 15)
#define SUPPORTED_Backplane		(1 << 16)
#define SUPPORTED_1000baseKX_Full	(1 << 17)
#define SUPPORTED_10000baseKX4_Full	(1 << 18)
#define SUPPORTED_10000baseKR_Full	(1 << 19)
#define SUPPORTED_10000baseR_FEC	(1 << 20)
#define SUPPORTED_20000baseMLD2_Full	(1 << 21)
#define SUPPORTED_20000baseKR2_Full	(1 << 22)
#define SUPPORTED_40000baseKR4_Full	(1 << 23)
#define SUPPORTED_40000baseCR4_Full	(1 << 24)
#define SUPPORTED_40000baseSR4_Full	(1 << 25)
#define SUPPORTED_40000baseLR4_Full	(1 << 26)

#define ADVERTISED_10baseT_Half		(1 << 0)
#define ADVERTISED_10baseT_Full		(1 << 1)
#define ADVERTISED_100baseT_Half	(1 << 2)
#define ADVERTISED_100baseT_Full	(1 << 3)
#define ADVERTISED_1000baseT_Half	(1 << 4)
#define ADVERTISED_1000baseT_Full	(1 << 5)
#define ADVERTISED_Autoneg		(1 << 6)
#define ADVERTISED_TP			(1 << 7)
#define ADVERTISED_AUI			(1 << 8)
#define ADVERTISED_MII			(1 << 9)
#define ADVERTISED_FIBRE		(1 << 10)
#define ADVERTISED_BNC			(1 << 11)
#define ADVERTISED_10000baseT_Full	(1 << 12)
#define ADVERTISED_Pause		(1 << 13)
#define ADVERTISED_Asym_Pause		(1 << 14)
#define ADVERTISED_2500baseX_Full	(1 << 15)
#define ADVERTISED_Backplane		(1 << 16)
#define ADVERTISED_1000baseKX_Full	(1 << 17)
#define ADVERTISED_10000baseKX4_Full	(1 << 18)
#define ADVERTISED_10000baseKR_Full	(1 << 19)
#define ADVERTISED_10000baseR_FEC	(1 << 20)
#define ADVERTISED_20000baseMLD2_Full	(1 << 21)
#define ADVERTISED_20000baseKR2_Full	(1 << 22)
#define ADVERTISED_40000baseKR4_Full	(1 << 23)
#define ADVERTISED_40000baseCR4_Full	(1 << 24)
#define ADVERTISED_40000baseSR4_Full	(1 << 25)
#define ADVERTISED_40000baseLR4_Full	(1 << 26)
#define SUPPORTED_25000baseKR_Full    (1<<27)
#define SUPPORTED_50000baseKR2_Full   (1<<28)
#define SUPPORTED_100000baseKR4_Full  (1<<29)
#define SUPPORTED_100000baseCR4_Full  (1<<30)

/* The following are all involved in forcing a particular link
 * mode for the device for setting things.  When getting the
 * devices settings, these indicate the current mode and whether
 * it was forced up into this mode or autonegotiated.
 */

/* The forced speed, 10Mb, 100Mb, gigabit, 2.5Gb, 10GbE. */
#define SPEED_10		10
#define SPEED_100		100
#define SPEED_1000		1000
#define SPEED_2500		2500
#define SPEED_10000		10000
#define SPEED_UNKNOWN		-1

/* Duplex, half or full. */
//#define DUPLEX_HALF		0x00
//#define DUPLEX_FULL		0x01
//#define DUPLEX_UNKNOWN		0xff

/* Which connector port. */
#define PORT_TP			0x00
#define PORT_AUI		0x01
#define PORT_MII		0x02
#define PORT_FIBRE		0x03
#define PORT_BNC		0x04
#define PORT_DA			0x05
#define PORT_NONE		0xef
#define PORT_OTHER		0xff

#define min_t(type, x, y) ({			\
	type __min1 = (x);			\
	type __min2 = (y);			\
	__min1 < __min2 ? __min1: __min2; })

#define max_t(type, x, y) ({			\
	type __max1 = (x);			\
	type __max2 = (y);			\
	__max1 > __max2 ? __max1: __max2; })

#ifndef DIV_ROUND_UP
#define DIV_ROUND_UP(n,d) (((n) + (d) - 1) / (d))
#endif
#define ADDR (*(volatile long *) addr)

/*
 * deal with unrepresentable constant logarithms
 */
extern __attribute__((const, noreturn))
int ____ilog2_NaN(void);

/**
 * fls - find last bit set
 * @x: the word to search
 *
 * Finds last bit set in the specified word
 *
 * RETURN VALUE:
 * The last set bit in specified word
 */
static __inline__ int fls(int x)
{
	int r;

	__asm__("bsrl %1,%0\n\t"
		"cmovzl %2,%0"
		: "=&r" (r) : "rm" (x), "rm" (-1));
	return r+1;
}

/*
 * __fls: find last bit set.
 * @word: The word to search
 *
 * Undefined if no zero exists, so code should check against ~0UL first.
 */
static __inline__ unsigned long __fls(unsigned long word)
{
	__asm__("bsrq %1,%0"
		:"=r" (word)
		:"rm" (word));
	return word;
}


/**
 * fls64 - find last bit set in 64 bit word
 * @x: the word to search
 *
 * This is defined the same way as fls.
 */
static __inline__ int fls64(vmk_uint64 x)
{
	if (x == 0)
		return 0;
	return __fls(x) + 1;
}



/*
 * non-constant log of base 2 calculators
 * - the arch may override these in asm/bitops.h if they can be implemented
 *   more efficiently than using fls() and fls64()
 * - the arch is not required to handle n==0 if implementing the fallback
 */
static inline __attribute__((const))
int __ilog2_u32(vmk_uint32 n)
{
	return fls(n) - 1;
}

static inline __attribute__((const))
int __ilog2_u64(vmk_uint64 n)
{
	return fls64(n) - 1;
}


/**
 * ilog2 - log of base 2 of 32-bit or a 64-bit unsigned value
 * @n - parameter
 *
 * constant-capable log of base 2 calculation
 * - this can be used to initialise global variables from constant data, hence
 *   the massive ternary operator construction
 *
 * selects the appropriately-sized optimised version depending on sizeof(n)
 */
#define ilog2(n)				\
(						\
	__builtin_constant_p(n) ? (		\
		(n) < 1 ? ____ilog2_NaN() :	\
		(n) & (1ULL << 63) ? 63 :	\
		(n) & (1ULL << 62) ? 62 :	\
		(n) & (1ULL << 61) ? 61 :	\
		(n) & (1ULL << 60) ? 60 :	\
		(n) & (1ULL << 59) ? 59 :	\
		(n) & (1ULL << 58) ? 58 :	\
		(n) & (1ULL << 57) ? 57 :	\
		(n) & (1ULL << 56) ? 56 :	\
		(n) & (1ULL << 55) ? 55 :	\
		(n) & (1ULL << 54) ? 54 :	\
		(n) & (1ULL << 53) ? 53 :	\
		(n) & (1ULL << 52) ? 52 :	\
		(n) & (1ULL << 51) ? 51 :	\
		(n) & (1ULL << 50) ? 50 :	\
		(n) & (1ULL << 49) ? 49 :	\
		(n) & (1ULL << 48) ? 48 :	\
		(n) & (1ULL << 47) ? 47 :	\
		(n) & (1ULL << 46) ? 46 :	\
		(n) & (1ULL << 45) ? 45 :	\
		(n) & (1ULL << 44) ? 44 :	\
		(n) & (1ULL << 43) ? 43 :	\
		(n) & (1ULL << 42) ? 42 :	\
		(n) & (1ULL << 41) ? 41 :	\
		(n) & (1ULL << 40) ? 40 :	\
		(n) & (1ULL << 39) ? 39 :	\
		(n) & (1ULL << 38) ? 38 :	\
		(n) & (1ULL << 37) ? 37 :	\
		(n) & (1ULL << 36) ? 36 :	\
		(n) & (1ULL << 35) ? 35 :	\
		(n) & (1ULL << 34) ? 34 :	\
		(n) & (1ULL << 33) ? 33 :	\
		(n) & (1ULL << 32) ? 32 :	\
		(n) & (1ULL << 31) ? 31 :	\
		(n) & (1ULL << 30) ? 30 :	\
		(n) & (1ULL << 29) ? 29 :	\
		(n) & (1ULL << 28) ? 28 :	\
		(n) & (1ULL << 27) ? 27 :	\
		(n) & (1ULL << 26) ? 26 :	\
		(n) & (1ULL << 25) ? 25 :	\
		(n) & (1ULL << 24) ? 24 :	\
		(n) & (1ULL << 23) ? 23 :	\
		(n) & (1ULL << 22) ? 22 :	\
		(n) & (1ULL << 21) ? 21 :	\
		(n) & (1ULL << 20) ? 20 :	\
		(n) & (1ULL << 19) ? 19 :	\
		(n) & (1ULL << 18) ? 18 :	\
		(n) & (1ULL << 17) ? 17 :	\
		(n) & (1ULL << 16) ? 16 :	\
		(n) & (1ULL << 15) ? 15 :	\
		(n) & (1ULL << 14) ? 14 :	\
		(n) & (1ULL << 13) ? 13 :	\
		(n) & (1ULL << 12) ? 12 :	\
		(n) & (1ULL << 11) ? 11 :	\
		(n) & (1ULL << 10) ? 10 :	\
		(n) & (1ULL <<  9) ?  9 :	\
		(n) & (1ULL <<  8) ?  8 :	\
		(n) & (1ULL <<  7) ?  7 :	\
		(n) & (1ULL <<  6) ?  6 :	\
		(n) & (1ULL <<  5) ?  5 :	\
		(n) & (1ULL <<  4) ?  4 :	\
		(n) & (1ULL <<  3) ?  3 :	\
		(n) & (1ULL <<  2) ?  2 :	\
		(n) & (1ULL <<  1) ?  1 :	\
		(n) & (1ULL <<  0) ?  0 :	\
		____ilog2_NaN()			\
				   ) :		\
	(sizeof(n) <= 4) ?			\
	__ilog2_u32(n) :			\
	__ilog2_u64(n)				\
 )

/**
 *  constant_test_bit - determine whether a bit is set
 *  @nr: bit number to test
 *  @addr: addr to test
 *
 *  Determines the state of the specified bit.
 *  This is used when @nr is known to be constant at compile-time.
 *  Use test_bit() instead of using this directly.
 *
 *  RETURN VALUE:
 *  0 if the bit was 0 and NON-ZERO otherwise
 */
static __inline__ int constant_test_bit(int nr, const volatile void * addr)
{
	return ((1UL << (nr & 31)) & (((const volatile unsigned int *) addr)[nr >> 5])) != 0;
}

/**
 *  variable_test_bit - determine whether a bit is set
 *  @nr: bit number to test
 *  @addr: addr to test
 *
 *  Determines the state of the specified bit.
 *  This is used when @nr is a variable.
 *  Use test_bit() instead of using this directly.
 *
 *  RETURN VALUE:
 *  0 if the bit was 0 and NON-ZERO otherwise
 */
static __inline__ int variable_test_bit(int nr, volatile const void * addr)
{
	int oldbit;

	__asm__ __volatile__(
		"btl %2,%1\n\tsbbl %0,%0"
		:"=r" (oldbit)
		:"m" (ADDR),"dIr" (nr));
	return oldbit;
}
/**
 *  test_bit - Determine if bit at given position is set
 *  @nr: number of bit to be tested
 *  @addr: pointer to byte to test
 *
 *  It tests if the bit at position nr in *addr is 0 or not.
 *  If the bit number is a constant an optimized bit extract is done.
 *  Note that the return value need not be 1 (just non-zero) if the bit was 1.
 *
 *  SYNOPSIS:
 *  #define test_bit(nr,addr)
 *
 *  RETURN VALUE:
 *  0 if the bit was 0 and NON-ZERO otherwise
 */
#define test_bit(nr,addr) \
(__builtin_constant_p(nr) ? \
 constant_test_bit((nr),(addr)) : \
 variable_test_bit((nr),(addr)))

unsigned long find_first_zero_bit(const unsigned long *addr, unsigned long size);
__inline__ void set_bit(int nr, volatile void * addr);
__inline__ void clear_bit(int nr, volatile void * addr);
__inline__ int test_and_set_bit(int nr, volatile void * addr);
__inline__ int test_and_clear_bit(int nr, volatile void * addr);
//void int_to_scsilun(unsigned int lun, vmk_uint8	*scsilun);
vmk_ByteCount strlen(const char *s);

/* Hash table defines */
struct hlist_node {
	struct hlist_node *next, **pprev;
};
struct hlist_head {
	struct hlist_node *first;
};

#define DECLARE_HASHTABLE(name, bits)                                           \
	struct hlist_head name[1 << (bits)]

#define GOLDEN_RATIO_PRIME_32 0x9e370001UL

static inline vmk_uint32 hash_32(vmk_uint32 val, unsigned int bits)
{
	/* On some cpus multiply is faster, on others gcc will do shifts */
	vmk_uint32 hash = val * GOLDEN_RATIO_PRIME_32;

	/* High bits are more random, so use them. */
	return hash >> (32 - bits);
}

#define hash_long(val, bits) hash_32(val, bits)

#define hash_min(val, bits)                                                     \
	(sizeof(val) <= 4 ? hash_32(val, bits) : hash_long(val, bits))

#define HASH_SIZE(name) (ARRAY_SIZE(name))
#define HASH_BITS(name) ilog2(HASH_SIZE(name))

#define hlist_entry(ptr, type, member) VMK_LIST_ENTRY(ptr,type,member)

#define hlist_entry_safe(ptr, type, member) \
		({ typeof(ptr) ____ptr = (ptr); \
		____ptr ? hlist_entry(____ptr, type, member) : NULL; \
		})

#define hlist_for_each_entry(pos, head, member)                         \
	for (pos = hlist_entry_safe((head)->first, typeof(*(pos)), member);\
		pos;                                                       \
		pos = hlist_entry_safe((pos)->member.next, typeof(*(pos)), member))

#define hash_for_each_possible(name, obj, member, key)                  \
	hlist_for_each_entry(obj, &name[hash_min(key, HASH_BITS(name))], member)

static inline int hlist_empty(const struct hlist_head *h)
{
	return !h->first;
}

static inline vmk_Bool __hash_empty(struct hlist_head *ht, unsigned int sz)
{
	unsigned int i;

	for (i = 0; i < sz; i++)
		if (!hlist_empty(&ht[i]))
			return VMK_FALSE;

	return VMK_TRUE;
}

#define hash_empty(hashtable) __hash_empty(hashtable, HASH_SIZE(hashtable))

#define INIT_HLIST_HEAD(ptr) ((ptr)->first = NULL)

static inline void __hash_init(struct hlist_head *ht, unsigned int sz)
{
	unsigned int i;

	for (i = 0; i < sz; i++)
		INIT_HLIST_HEAD(&ht[i]);
}

#define hash_init(hashtable) __hash_init(hashtable, HASH_SIZE(hashtable))

static inline void hlist_add_head(struct hlist_node *n, struct hlist_head *h)
{
	struct hlist_node *first = h->first;
	n->next = first;
	if (first)
		first->pprev = &n->next;
	h->first = n;
	n->pprev = &h->first;
}

#define hash_add(hashtable, node, key)                                          \
	hlist_add_head(node, &hashtable[hash_min(key, HASH_BITS(hashtable))])

static inline void __hlist_del(struct hlist_node *n)
{
	struct hlist_node *next = n->next;
	struct hlist_node **pprev = n->pprev;
	*pprev = next;
	if (next)
		next->pprev = pprev;
}

#define LIST_POISON1  ((void *) 0x00100100)
#define LIST_POISON2  ((void *) 0x00200200)

static inline void hlist_del(struct hlist_node *n)
{
	__hlist_del(n);
	n->next = LIST_POISON1;
	n->pprev = LIST_POISON2;
}

/*
 * 24 bit port ID type definition.
 */
typedef union {
	vmk_uint32 b24 : 24;

	struct {
		vmk_uint8 d_id[3];
		vmk_uint8 rsvd_1;
	} r;

	struct {
		vmk_uint8 al_pa;
		vmk_uint8 area;
		vmk_uint8 domain;
		vmk_uint8 rsvd_1;
	} b;
} port_id_t;
#define INVALID_PORT_ID	0xFFFFFF

/* Copyright (c) 2003 Silicon Graphics, Inc.  All rights reserved. */
static inline vmk_uint64 wwn_to_u64(vmk_uint8 *wwn)
{
	return (vmk_uint64)wwn[0] << 56 | (vmk_uint64)wwn[1] << 48 |
	    (vmk_uint64)wwn[2] << 40 | (vmk_uint64)wwn[3] << 32 |
	    (vmk_uint64)wwn[4] << 24 | (vmk_uint64)wwn[5] << 16 |
	    (vmk_uint64)wwn[6] <<  8 | (vmk_uint64)wwn[7];
}

#define container_of(ptr, type, member) ({                      \
	const typeof( ((type *)0)->member ) *__mptr = (ptr);    \
	(type *)( (char *)__mptr - vmk_offsetof(type,member) );})

struct ifla_vf_info {
	vmk_uint32 vf;
	vmk_uint8 mac[32];
	vmk_uint32 vlan;
	vmk_uint32 qos;
	vmk_uint32 spoofchk;
	vmk_uint32 linkstate;
	vmk_uint32 min_tx_rate;
	vmk_uint32 max_tx_rate;
};

struct file {
	void *private_data;
};

/* SV: From qed_gpl_2.h */

/* GPL'ed defines */

#define BIT(x) (1 << (x))

/* SV: XXX: Borrowed from if_ether.h --start-- */
/*
 *	IEEE 802.3 Ethernet magic constants.  The frame sizes omit the preamble
 *	and FCS/CRC (frame check sequence).
 */

//#define ETH_ALEN	6		/* Octets in one ethernet addr	 */
#define ETH_HLEN	14		/* Total octets in header.	 */
#define ETH_ZLEN	60		/* Min. octets in frame sans FCS */
#define ETH_DATA_LEN	1500		/* Max. octets in payload	 */
#define ETH_FRAME_LEN	1514		/* Max. octets in frame sans FCS */

#define ETH_HEADER_TYPE_DIX               0
#define ETH_HEADER_TYPE_802_1PQ           1
#define ETH_HEADER_TYPE_802_3             2
#define ETH_HEADER_TYPE_802_1PQ_802_3     3

#define MAX_ADDR_LEN 32
//#define VLAN_VID_MASK 0xfff

/*
 *	These are the defined Ethernet Protocol ID's.
 */
#define ETH_P_LOOP	0x0060		/* Ethernet Loopback packet	*/
#define ETH_P_PUP	0x0200		/* Xerox PUP packet		*/
#define ETH_P_PUPAT	0x0201		/* Xerox PUP Addr Trans packet	*/
#define ETH_P_IP	0x0800		/* Internet Protocol packet	*/
#define ETH_P_X25	0x0805		/* CCITT X.25			*/
#define ETH_P_ARP	0x0806		/* Address Resolution packet	*/
#define	ETH_P_BPQ	0x08FF		/* G8BPQ AX.25 Ethernet Packet	[ NOT AN OFFICIALLY REGISTERED ID ] */
#define ETH_P_IEEEPUP	0x0a00		/* Xerox IEEE802.3 PUP packet */
#define ETH_P_IEEEPUPAT	0x0a01		/* Xerox IEEE802.3 PUP Addr Trans packet */
#define ETH_P_DEC       0x6000          /* DEC Assigned proto           */
#define ETH_P_DNA_DL    0x6001          /* DEC DNA Dump/Load            */
#define ETH_P_DNA_RC    0x6002          /* DEC DNA Remote Console       */
#define ETH_P_DNA_RT    0x6003          /* DEC DNA Routing              */
#define ETH_P_LAT       0x6004          /* DEC LAT                      */
#define ETH_P_DIAG      0x6005          /* DEC Diagnostics              */
#define ETH_P_CUST      0x6006          /* DEC Customer use             */
#define ETH_P_SCA       0x6007          /* DEC Systems Comms Arch       */
#define ETH_P_RARP      0x8035		/* Reverse Addr Res packet	*/
#define ETH_P_ATALK	0x809B		/* Appletalk DDP		*/
#define ETH_P_AARP	0x80F3		/* Appletalk AARP		*/
#define ETH_P_8021Q	0x8100          /* 802.1Q VLAN Extended Header  */
#define ETH_P_IPX	0x8137		/* IPX over DIX			*/
#define ETH_P_IPV6	0x86DD		/* IPv6 over bluebook		*/
#define ETH_P_SLOW	0x8809		/* Slow Protocol. See 802.3ad 43B */
#define ETH_P_WCCP	0x883E		/* Web-cache coordination protocol
					 * defined in draft-wilson-wrec-wccp-v2-00.txt */
#define ETH_P_PPP_DISC	0x8863		/* PPPoE discovery messages     */
#define ETH_P_PPP_SES	0x8864		/* PPPoE session messages	*/
#define ETH_P_MPLS_UC	0x8847		/* MPLS Unicast traffic		*/
#define ETH_P_MPLS_MC	0x8848		/* MPLS Multicast traffic	*/
#define ETH_P_ATMMPOA	0x884c		/* MultiProtocol Over ATM	*/
#define ETH_P_ATMFATE	0x8884		/* Frame-based ATM Transport
					 * over Ethernet
					 */
#define ETH_P_AOE	0x88A2		/* ATA over Ethernet		*/
#define ETH_P_TIPC	0x88CA		/* TIPC 			*/

#define ETH_P_VMWARE    0x8922
#define ETH_P_AKIMBI    0x88DE
#define ETH_P_IPV6_NBO  0xDD86
#define ETH_P_AKIMBI_NBO 0xDE88

#define ETH_P_FCOE      0x8906          /* FCOE ether type */
#define ETH_P_FIP       0x8914          /* FIP ether type */

/*
 *	Non DIX types. Won't clash for 1500 types.
 */
#define ETH_P_802_3	0x0001		/* Dummy type for 802.3 frames  */
#define ETH_P_AX25	0x0002		/* Dummy protocol id for AX.25  */
#define ETH_P_ALL	0x0003		/* Every packet (be careful!!!) */
#define ETH_P_802_2	0x0004		/* 802.2 frames 		*/
#define ETH_P_SNAP	0x0005		/* Internal only		*/
#define ETH_P_DDCMP     0x0006          /* DEC DDCMP: Internal only     */
#define ETH_P_WAN_PPP   0x0007          /* Dummy type for WAN PPP frames*/
#define ETH_P_PPP_MP    0x0008          /* Dummy type for PPP MP frames */
#define ETH_P_LOCALTALK 0x0009		/* Localtalk pseudo type 	*/
#define ETH_P_PPPTALK	0x0010		/* Dummy type for Atalk over PPP*/
#define ETH_P_TR_802_2	0x0011		/* 802.2 frames 		*/
#define ETH_P_MOBITEX	0x0015		/* Mobitex (kaz@cafe.net)	*/
#define ETH_P_CONTROL	0x0016		/* Card specific control frames */
#define ETH_P_IRDA	0x0017		/* Linux-IrDA			*/
#define ETH_P_ECONET	0x0018		/* Acorn Econet			*/
#define ETH_P_HDLC	0x0019		/* HDLC frames			*/
#define ETH_P_ARCNET	0x001A		/* 1A for ArcNet :-)            */

/*
 *	This is an Ethernet frame header.
 */
struct ethhdr_llc {
	unsigned char	h_dest[VMK_ETH_ADDR_LENGTH];
	unsigned char	h_source[VMK_ETH_ADDR_LENGTH];
	unsigned short  h_llc_len;
	unsigned char   h_llc_dsap;
	unsigned char   h_llc_ssap;
	unsigned char   h_llc_control;
	unsigned char   h_llc_snap_org[3];
	/*__be16*/vmk_uint16	h_llc_snap_type;
} __attribute__((packed));

struct ethhdr {
	unsigned char          h_dest[VMK_ETH_ADDR_LENGTH];	/* destination eth addr	*/
	unsigned char          h_source[VMK_ETH_ADDR_LENGTH];	/* source ether addr	*/
	/*__be16*/vmk_uint16	h_proto;		/* packet type ID field	*/
} __attribute__((packed));

struct fence_header {
	unsigned int ver:2,
				 frag:1,
				 fragId:5,
				 fid:24;
	unsigned char origDstMac[VMK_ETH_ADDR_LENGTH];
} __attribute__((packed));;
/* SV: XXX: Borrowed from if_ether.h --end-- */


#ifndef WARN_ON
#define WARN_ON(x)
#endif

#ifndef IS_ERR
#define MAX_ERRNO 4095
#define IS_ERR(x) ((unsigned long)x >= (unsigned long)-MAX_ERRNO)
#endif

#ifndef PTR_ERR
#define PTR_ERR(x) (long)(x)
#endif

#ifndef ERR_PTR
#define ERR_PTR(x) ((void *)x)
#endif

#ifndef min
#define min(x, y) (((x) < (y)) ? (x) : (y))
#endif

#define __constant_htonl(x) vmk_CPUToBE32(x)
#define htonl(x)    vmk_CPUToBE32(x)
#define htons(x)    vmk_CPUToBE16(x)
#define ntohl(x)    vmk_BE32ToCPU(x)
#define ntohs(x)    vmk_BE16ToCPU(x)


#endif
