/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

/*
 * qfle3f_vmkmgmt.c: QLogic 10 Gigabit ESX Ethernet FCoE offload driver.
 *
 */

#include <vmkapi.h>
#include "ql_fcoe_vmkmgmt.h"
#include "qedfc_basetypes.h"
#include <qfle3f_version.h>
#include <qfle3f_vmk.h>
#include "qfle3f.h"

#define QED_PT_CMD_TOV      (66)

#define CMD_IOCTL_COMP_STATUS(Cmnd) ((Cmnd)->status.plugin)
#define CMD_IOCTL_SCSI_STATUS(Cmnd) ((Cmnd)->lbc)
#define CMD_IOCTL_ACTUAL_SNSLEN(Cmnd)       ((Cmnd)->lba)

#define SS_MASK             0xfff   /* Reserved bits BIT_12-BIT_15*/
#define SS_RESIDUAL_UNDER       BIT_11
#define SS_RESIDUAL_OVER        BIT_10
#define SS_SENSE_LEN_VALID      BIT_9
#define SS_RESPONSE_INFO_LEN_VALID  BIT_8
#define SS_CHECK_CONDITION      BIT_1

/*
 * Status entry completion status
 */
#define CS_COMPLETE     0x0 /* No errors */
#define CS_INCOMPLETE       0x1 /* Incomplete transfer of cmd. */
#define CS_DMA          0x2 /* A DMA direction error. */
#define CS_TRANSPORT        0x3 /* Transport error. */
#define CS_RESET        0x4 /* SCSI bus reset occurred */
#define CS_ABORTED      0x5 /* System aborted command. */
#define CS_TIMEOUT      0x6 /* Timeout error. */
#define CS_DATA_OVERRUN     0x7 /* Data overrun. */
#if (VMWARE_ESX_DDK_VERSION == 60000)
#define CS_DIF_ERROR        0xC /* DIF error detected  */
#endif

#define CS_DATA_UNDERRUN    0x15    /* Data Underrun. */
#define CS_QUEUE_FULL       0x1C    /* Queue Full. */
#define CS_PORT_UNAVAILABLE 0x28    /* Port unavailable */
                    /* (selection timeout) */
#define CS_PORT_LOGGED_OUT  0x29    /* Port Logged Out */
#define CS_PORT_CONFIG_CHG  0x2A    /* Port Configuration Changed */
#define CS_PORT_BUSY        0x2B    /* Port Busy */
#define CS_COMPLETE_CHKCOND 0x30    /* Error? */
#define CS_BAD_PAYLOAD      0x80    /* Driver defined */
#define CS_UNKNOWN      0x81    /* Driver defined */
#define CS_RETRY        0x82    /* Driver defined */
#define CS_LOOP_DOWN_ABORT  0x83    /* Driver defined */

#define STATUS_MASK     0xfe
#define IOCTL_INVALID_STATUS 0xffff

vmk_MgmtApiSignature ql_fcoe_sig;
vmk_MgmtHandle qed_fcapi_vmkmgmt_api_handle;

static VMK_ReturnStatus qedfc_find_qfle3fHBA(UINT64 host_no,
		struct qfle3fHBA **find_hba, int host) {
	struct qfle3fHBA *hba;
	vmk_ListLinks *current, *nextPtr;

	if (vmk_ListIsEmpty(&qfle3fDriverInfo.qfle3fHostList)) {
		vmk_WarningMessage("%s: %d: List is empty\n",
				__func__, __LINE__);
		return 0;
	}

	VMK_LIST_FORALL_SAFE(&qfle3fDriverInfo.qfle3fHostList, current,
			nextPtr) {
		hba = VMK_LIST_ENTRY(current, struct qfle3fHBA, link);
		if (host) {
			if (hba && hba->host_no == host_no) {
				qfle3f_log(hba, LOG_APP, "hba = %p host_no = 0x%x\n",
					hba, hba->host_no);
				*find_hba = hba;
				return VMK_OK;
			}
		} else {
			if (hba && hba->instance == host_no) {
				qfle3f_log(hba, LOG_APP, "hba = %p instance_no = 0x%x\n",
						hba, hba->instance);
				*find_hba = hba;
				return VMK_OK;
			}
		}
	}

	vmk_WarningMessage("%s: %d hba is not found for the host = 0x%lx\n",
		__func__, __LINE__, host_no);
	return VMK_FAILURE;
}

int ql_fcoe_get_hba_cnt(UINT64 api_cookie, UINT64 instance_id,
	struct ql_fcoe_hba_cnt_cb *get_hba_cnt)
{
	get_hba_cnt->hba_cnt = qfle3fDriverInfo.host_count;
	get_hba_cnt->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_fcoe_get_hostno(UINT64 api_cookie, UINT64 instance_id,
	struct ql_fcoe_hostno_cb *get_host_no)
{
	struct qfle3fHBA *hba = NULL;

	if (qedfc_find_qfle3fHBA(get_host_no->inst, &hba, 0) != VMK_OK) {
		get_host_no->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	qfle3f_log(hba, LOG_APP, "Enter for inst = 0x%lx\n",
		get_host_no->inst);

	get_host_no->host_no = hba->host_no;
	get_host_no->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_fcoe_get_pci_info(UINT64 api_cookie, UINT64 instance_id,
	struct ql_fcoe_pci_info_cb *get_pci_info)
{
	struct qfle3fHBA *hba = NULL;
	struct ql_fcoe_pci_info *ql_fcoe_pci_info = &get_pci_info->pci;
	vmk_PCIDeviceID *ql_fcoe_pci_dev = NULL;
	enum ql_fcoe_pci_bus_speed pcie_speed;
	enum ql_fcoe_pcie_link_width pcie_width;
	vmk_uint32 val = 0;
	struct cnic_dev *cnic = NULL;

	if (qedfc_find_qfle3fHBA(get_pci_info->host_no, &hba, 1) != VMK_OK) {
		get_pci_info->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	vmk_Memset(ql_fcoe_pci_info, 0, sizeof(struct ql_fcoe_pci_info));

	qfle3f_log(hba, LOG_APP, "Enter for host = 0x%x\n",
		get_pci_info->host_no);

	cnic = hba->cnic;

	ql_fcoe_pci_dev = &hba->pdev_id;

	ql_fcoe_pci_info->vendor_id = ql_fcoe_pci_dev->vendorID;
	ql_fcoe_pci_info->device_id = ql_fcoe_pci_dev->deviceID;
	ql_fcoe_pci_info->subvendor_id = ql_fcoe_pci_dev->subVendorID;
	ql_fcoe_pci_info->subsystem_id = ql_fcoe_pci_dev->subDeviceID;
//	ql_fcoe_pci_info->domain_num = ;
	ql_fcoe_pci_info->bus_num = hba->vmk_pci_addr.bus;
	ql_fcoe_pci_info->device_num = hba->vmk_pci_addr.dev;
	ql_fcoe_pci_info->function_num = hba->vmk_pci_addr.fn;
	ql_fcoe_pci_info->revision = ql_fcoe_pci_dev->progIFRevID;

	if (cnic->pf_id & 0x1)
		ql_fcoe_pci_info->port_num = (cnic->pf_num & 0x3) + 1;
	else
		ql_fcoe_pci_info->port_num = (cnic->pf_num & 0x1) + 1;

	 vmk_PCIReadConfig(vmk_ModuleCurrentID, hba->pdev,
		VMK_PCI_CONFIG_ACCESS_32, QL_FCOE_PCICFG_LINK_CONTROL, &val);

	 pcie_width = (enum ql_fcoe_pcie_link_width) ((val & QL_FCOE_PCICFG_LINK_WIDTH) >>
		QL_FCOE_PCICFG_LINK_WIDTH_SHIFT);
	 pcie_speed = (val & QL_FCOE_PCICFG_LINK_SPEED) >> QL_FCOE_PCICFG_LINK_SPEED_SHIFT;

	 switch (pcie_speed) {
		 case 3:
			 ql_fcoe_pci_info->pcie_speed = QL_FCOE_PCIE_SPEED_8_0GT;
			 break;
		case 2:
			 ql_fcoe_pci_info->pcie_speed = QL_FCOE_PCIE_SPEED_5_0GT;
			 break;
		default:
			 ql_fcoe_pci_info->pcie_speed = QL_FCOE_PCIE_SPEED_2_5GT;
			 break;
	 }

	 ql_fcoe_pci_info->pcie_width = pcie_width;

	 get_pci_info->status = EXT_STATUS_OK;

	return VMK_OK;
}

#define QFC_PORTSPEED_UNKNOWN        0 /* Unknown - transceiver incapable of reporting */
#define QFC_PORTSPEED_1GBIT      1
#define QFC_PORTSPEED_2GBIT      2
#define QFC_PORTSPEED_4GBIT      4
#define QFC_PORTSPEED_10GBIT     8
#define QFC_PORTSPEED_8GBIT      0x10
#define QFC_PORTSPEED_16GBIT     0x20
#define QFC_PORTSPEED_20GBIT         0x80
#define QFC_PORTSPEED_25GBIT     0x800
#define QFC_PORTSPEED_40GBIT         0x100
#define QFC_PORTSPEED_50GBIT         0x200
#define QFC_PORTSPEED_100GBIT    0x400
#define QFC_PORTSPEED_NOT_NEGOTIATED (1 << 15) /* Speed not established */

#define SPEED_40000     40000
#define SPEED_25000     25000
#define SPEED_50000     50000
#define SPEED_100000    100000
#define SPEED_1000 		1000
#define SPEED_10000 	10000

#define ELINK_SUPPORTED_10baseT_Half        (1<<0)
#define ELINK_SUPPORTED_10baseT_Full        (1<<1)
#define ELINK_SUPPORTED_100baseT_Half       (1<<2)
#define ELINK_SUPPORTED_100baseT_Full       (1<<3)
#define ELINK_SUPPORTED_1000baseT_Full  (1<<4)
#define ELINK_SUPPORTED_2500baseX_Full  (1<<5)
#define ELINK_SUPPORTED_10000baseT_Full     (1<<6)
#define ELINK_SUPPORTED_TP          (1<<7)
#define ELINK_SUPPORTED_FIBRE           (1<<8)
#define ELINK_SUPPORTED_Autoneg         (1<<9)
#define ELINK_SUPPORTED_Pause           (1<<10)
#define ELINK_SUPPORTED_Asym_Pause      (1<<11)
#define ELINK_SUPPORTED_1000baseKX_Full     (1<<17)
#define ELINK_SUPPORTED_10000baseKR_Full    (1<<19)
#define ELINK_SUPPORTED_20000baseMLD2_Full  (1<<21)
#define ELINK_SUPPORTED_20000baseKR2_Full   (1<<22)

int ql_fcoe_get_port_info(UINT64 api_cookie, UINT64 instance_id,
	struct ql_fcoe_port_info_cb *get_port_info)
{
	struct qfle3fHBA *hba = NULL;
	struct ql_fcoe_port_info *ql_fcoe_port_info = &get_port_info->port;
	struct qfle3f_rport *target;
	int cnt, i;
	vmk_UplinkSharedData *sharedData;
	struct cnic_dev *cnic;
	vmk_uint16 supported_speeds;
	struct ql_fcoe_fabric *Fabric;

	if (qedfc_find_qfle3fHBA(get_port_info->host_no, &hba, 1) != VMK_OK) {
		get_port_info->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	vmk_Memset(ql_fcoe_port_info, 0, sizeof(struct ql_fcoe_port_info));

	qfle3f_log(hba, LOG_APP, "Enter for host = 0x%x\n",
		get_port_info->host_no);
	cnic = hba->cnic;

	vmk_Memcpy(ql_fcoe_port_info->manufacturer, "QLogic Corporation", QLEVFC_SIZE_64);
	vmk_StringFormat(ql_fcoe_port_info->model_name, QLEVFC_SIZE_32, NULL, "ISP%04x",
		hba->pdev_id.deviceID);
	vmk_Memcpy(ql_fcoe_port_info->model_desc, "QLogic 10G FCoE Adapter",
		QLEVFC_SIZE_128);
	vmk_Memcpy(ql_fcoe_port_info->driver_version, QFLE3F_VERSION, QLEVFC_SIZE_32);
	vmk_Strncpy(ql_fcoe_port_info->driver_date, DRV_MODULE_RELDATE, QLEVFC_SIZE_32);
	vmk_Strncpy(ql_fcoe_port_info->driver_name, QFLE3F_DRIVER_NAME, QLEVFC_SIZE_16);
	ql_fcoe_port_info->driver_loaded = 1;
	vmk_Memcpy(ql_fcoe_port_info->mac, cnic->fip_mac,
		QLEVFC_SIZE_6);

	sharedData = cnic->uplinkSharedData;
	vmk_NameCopy((vmk_Name *)ql_fcoe_port_info->fw_version,
		&sharedData->driverInfo.firmwareVersion);
	ql_fcoe_port_info->wwnn = hba->wwnn;
	ql_fcoe_port_info->wwpn = hba->wwpn;

	vmk_Memset(&supported_speeds, 0, sizeof(vmk_uint16));
	if (cnic->link_supported_speeds & ELINK_SUPPORTED_1000baseT_Full)
		supported_speeds |= QFC_PORTSPEED_1GBIT;
	if (cnic->link_supported_speeds & ELINK_SUPPORTED_10000baseT_Full)
		supported_speeds |= QFC_PORTSPEED_10GBIT;
	if (cnic->link_supported_speeds & ELINK_SUPPORTED_2500baseX_Full)
		supported_speeds |= QFC_PORTSPEED_25GBIT;
	ql_fcoe_port_info->port_sup_speed = supported_speeds;

	for(i=0; (i < MAX_NUM_FABRICS + MAX_NUM_NPIV_LOGIN); i++) {
		Fabric = hba->qlfcoeAdapter->FabricArray[i];
		if (Fabric) {
			if (Fabric->Flags & FABRIC_FLAGS_LOGGED_IN) {
				ql_fcoe_port_info->portid = (vmk_uint32)Fabric->fc_id[0] << 16 |
					Fabric->fc_id[1] << 8 | Fabric->fc_id[2];

				switch (sharedData->link.speed) {
					case SPEED_1000:
						ql_fcoe_port_info->port_speed = QFC_PORTSPEED_1GBIT;
						break;
					case SPEED_10000:
						ql_fcoe_port_info->port_speed = QFC_PORTSPEED_10GBIT;
						break;
					case SPEED_40000:
						ql_fcoe_port_info->port_speed = QFC_PORTSPEED_40GBIT;
						break;
					case SPEED_25000:
						ql_fcoe_port_info->port_speed = QFC_PORTSPEED_25GBIT;
						break;
					case SPEED_100000:
						ql_fcoe_port_info->port_speed = QFC_PORTSPEED_100GBIT;
						break;
					default:
						qfle3f_log(hba, LOG_APP, "Unknown speed (0x%x)\n",
								sharedData->link.speed);
				}

				cnt = 0;
				vmk_SemaLock(&hba->hbaMutex);
				for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
					target = hba->targetOffloadList[i];
					if (target)
						cnt++;
				}
				vmk_SemaUnlock(&hba->hbaMutex);
				ql_fcoe_port_info->tgt_cnt = cnt;

				cnt = 0;
				for (i=0; (i < MAX_NUM_FABRICS + MAX_NUM_NPIV_LOGIN); i++) {
					struct ql_fcoe_fabric *Fabric = hba->qlfcoeAdapter->FabricArray[i];
					if (Fabric)
						cnt++;
				}
				ql_fcoe_port_info->fcf_cnt = cnt;

				ql_fcoe_port_info->link_state = VMK_FC_PORTSTATE_ONLINE;
			} else {
				ql_fcoe_port_info->link_state = VMK_FC_PORTSTATE_LINKDOWN;
			}
			break;
		} else {
			ql_fcoe_port_info->link_state = VMK_FC_PORTSTATE_LINKDOWN;
		}
	}

	ql_fcoe_port_info->mtu = hba->cnic->mtu;
	ql_fcoe_port_info->vlan = hba->vlan_id;

	if (hba->type == HBA_PHYSICAL) {
		ql_fcoe_port_info->npiv_count = hba->vportsInUse;
		ql_fcoe_port_info->npiv_index = 0;
		ql_fcoe_port_info->port_type = 3;
	} else if (hba->type == HBA_VIRTUAL) {
		ql_fcoe_port_info->npiv_count = 0;
		ql_fcoe_port_info->npiv_index = hba->vp_id;
		ql_fcoe_port_info->port_type = 7;
	}

	vmk_Strncpy(ql_fcoe_port_info->hba_name, hba->vmhbaName,
			QLEVFC_SIZE_32);

	get_port_info->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_fcoe_get_port_stats(UINT64 api_cookie, UINT64 instance_id,
	struct ql_fcoe_port_stats_cb *get_port_stat)
{
	struct qfle3fHBA *hba = NULL;
	struct qfle3fHBA *phba = NULL;
	struct ql_fcoe_port_stats *port_stats = &get_port_stat->stats;
	struct fcoe_statistics_params *fw_stats;
	VMK_ReturnStatus status;

	if (qedfc_find_qfle3fHBA(get_port_stat->host_no, &hba, 1) != VMK_OK) {
		get_port_stat->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	qfle3f_log(hba, LOG_APP, "Enter for host = 0x%x\n",
		get_port_stat->host_no);

	phba = qfle3f_find_hba_for_cnic(hba->cnic);
	if (!phba){
		get_port_stat->status = EXT_STATUS_NO_MEMORY;
		qfle3f_warning(hba, "Not able to find parent hba \n");
		return VMK_OK;
	}

	vmk_Memset(port_stats, 0, sizeof(struct ql_fcoe_port_stats));
	fw_stats = (struct fcoe_statistics_params *)phba->statsBuffer;
	if (!fw_stats) {
		get_port_stat->status = EXT_STATUS_NO_MEMORY;
		qfle3f_log(hba, LOG_APP, "No stats buffer allocated in driver");
		return VMK_OK;
	}

	/* Port stats is just for physical port, even for virtual it
	 * send to corresponding physical port.
	 */
	if (qfle3f_sendStatRequest(phba)) {
		get_port_stat->status = EXT_STATUS_UNKNOWN;
		qfle3f_log(hba, LOG_APP, "failed to read stats info");
		return VMK_OK;
	}

	status = qfle3f_sleepWithCondition(phba, phba, (vmk_WorldEventID)&(phba->statRequestDone),
		VMK_LOCK_INVALID, 2, "Waiting from stats RAMROD",
		checkStatReqComplFlag);
	if (status != VMK_OK) {
		get_port_stat->status = EXT_STATUS_NO_MEMORY;
		qfle3f_warning(phba, "failed to read stats info \n");
		return VMK_OK;
	}

	port_stats->rx_fcoe_frames = fw_stats->rx_stat0.fcoe_rx_pkt_cnt;
	port_stats->rx_fcoe_bytes = fw_stats->rx_stat0.fcoe_rx_byte_cnt;
	port_stats->rx_fcp_frames = fw_stats->rx_stat2.fcp_rx_pkt_cnt;
	port_stats->tx_fcoe_frames = fw_stats->tx_stat.fcoe_tx_pkt_cnt;
	port_stats->tx_fcoe_bytes = fw_stats->tx_stat.fcoe_tx_byte_cnt;
	port_stats->tx_fcp_frames = fw_stats->tx_stat.fcp_tx_pkt_cnt;
	port_stats->seq_timeouts = fw_stats->rx_stat2.seq_timeout_cnt;
	port_stats->missing_frames = fw_stats->rx_stat2.miss_frame_cnt;
	port_stats->fc_crc_err_frames = fw_stats->rx_stat2.fc_crc_cnt;
	port_stats->seq_dropped = fw_stats->rx_stat2.drop_seq_cnt;
	port_stats->eofa_frames = fw_stats->rx_stat2.eofa_del_cnt;

	get_port_stat->status = EXT_STATUS_OK;
	return VMK_OK;
}

#define QFL3F_TGT_PORT_LINKUP 2
#define QFL3F_TGT_PORT_LINKDOWN 7

int ql_fcoe_get_tgt_info(UINT64 api_cookie, UINT64 instance_id,
	struct ql_fcoe_tgt_info_cb *get_tgt_info)
{
	struct qfle3fHBA *hba = NULL;
	struct ql_fcoe_tgt_info *tgt_info = &get_tgt_info->tgt;
	struct qfle3f_rport *target;

	if (qedfc_find_qfle3fHBA(get_tgt_info->host_no, &hba, 1) != VMK_OK) {
		get_tgt_info->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	qfle3f_log(hba, LOG_APP, "Enter for host = 0x%x\n",
		get_tgt_info->host_no);

	if (tgt_info->scsi_tgt_num >= QFLE3_FCOE_NUM_CONNECTIONS) {
		qfle3f_warning(hba, "Invalid Target index (0x%x)\n",
				tgt_info->scsi_tgt_num);
		get_tgt_info->status = EXT_STATUS_INVALID_PARAM;
		return VMK_INVALID_TARGET;
	}

	vmk_SemaLock(&hba->hbaMutex);
	target = hba->targetOffloadList[tgt_info->scsi_tgt_num];
	if (target) {
		tgt_info->wwnn = target->nodeName;
		tgt_info->wwpn = target->portName;
		tgt_info->portid = target->targetPortID;
		if (target->state == QFLE3F_RPORT_ST_INIT)
			tgt_info->port_state = QFL3F_TGT_PORT_LINKUP;
		else
			tgt_info->port_state = QFL3F_TGT_PORT_LINKDOWN;
	} else {
		get_tgt_info->status = EXT_STATUS_INVALID_PARAM;
	}
	vmk_SemaUnlock(&hba->hbaMutex);

	get_tgt_info->status = EXT_STATUS_OK;

	return VMK_OK;
}

int ql_fcoe_get_fcf_info(UINT64 api_cookie, UINT64 instance_id,
	struct ql_fcoe_fcf_info_cb *get_fcf_info)
{
	struct qfle3fHBA *hba = NULL;
	struct  ql_fcoe_fcf_info *fcf_info = &get_fcf_info->fcf;
	int fcf_cnt, i = 0;

	if (qedfc_find_qfle3fHBA(get_fcf_info->host_no, &hba, 1) != VMK_OK) {
		get_fcf_info->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	qfle3f_log(hba, LOG_APP, "Enter for host = 0x%x\n",
		get_fcf_info->host_no);


	fcf_cnt = 0;
	for (i = 0; (i < MAX_NUM_FABRICS + MAX_NUM_NPIV_LOGIN); i++) {
		if (fcf_cnt != fcf_info->fcf_num) {
			fcf_cnt++;
			continue;
		}
		struct ql_fcoe_fabric *Fabric = hba->qlfcoeAdapter->FabricArray[i];

		if (Fabric) {
			fcf_info->switch_name = (vmk_uint64)Fabric->FcfSwitchName[0] << 56 |
				(vmk_uint64)Fabric->FcfSwitchName[1] << 48 |
				(vmk_uint64)Fabric->FcfSwitchName[2] << 40 |
				(vmk_uint64)Fabric->FcfSwitchName[3] << 32 |
				(vmk_uint64)Fabric->FcfSwitchName[4] << 24 |
				(vmk_uint64)Fabric->FcfSwitchName[5] << 16 |
				(vmk_uint64)Fabric->FcfSwitchName[6] <<  8 |
				(vmk_uint64)Fabric->FcfSwitchName[7];

			fcf_info->fabric_name = (vmk_uint64)Fabric->FcfFabricName[0] << 56 |
				(vmk_uint64)Fabric->FcfFabricName[1] << 48 |
				(vmk_uint64)Fabric->FcfFabricName[2] << 40 |
				(vmk_uint64)Fabric->FcfFabricName[3] << 32 |
				(vmk_uint64)Fabric->FcfFabricName[4] << 24 |
				(vmk_uint64)Fabric->FcfFabricName[5] << 16 |
				(vmk_uint64)Fabric->FcfFabricName[6] <<  8 |
				(vmk_uint64)Fabric->FcfFabricName[7];

			fcf_info->fc_map = (vmk_uint32)Fabric->fc_map[0] << 16 |
				Fabric->fc_map[1] << 8 | Fabric->fc_map[2];

			fcf_info->vfid = Fabric->FcfVFId[0] << 8 | Fabric->FcfVFId[1];
			vmk_Memcpy(fcf_info->fcf_mac, Fabric->fcf_fcp_mac, QLEVFC_SIZE_6);
			//fcf_info->flogi_sent
			fcf_info->priority = Fabric->Priority;
			fcf_info->ka_timeout = Fabric->e_d_tov;
			//fcf_info->fcf_selected
		}
		break;
	}

	get_fcf_info->status = EXT_STATUS_OK;

	return VMK_OK;
}

static void ql_fcoe_scsi_pt_done(struct vmk_ScsiCommand *pscsi_cmd)
{
	qfle3fHBA_t *hba;

	hba = (qfle3fHBA_t *)pscsi_cmd->doneData;

	vmk_SpinlockLock(hba->ioctl_cmpl_lock);
	hba->SCSIPT_InProgress = 0;
	hba->cmpl_done = 1;
	vmk_SpinlockUnlock(hba->ioctl_cmpl_lock);

	vmk_WorldWakeup((vmk_WorldEventID)&hba->ioctl_cmpl_sem);
}

#define QL_FCOE_MAX_BUFFER_SIZE   (256 * 1024)

int ql_fcoe_send_scsi_cmd(UINT64 api_cookie, UINT64 instance_id,
	struct ql_fcoe_scsi_cmd_cb *scsi_cmd)
{
	struct qfle3fHBA *hba = NULL;
	struct ql_fcoe_scsi_cmd *ql_fcoe_scsi_cmd = &scsi_cmd->cmnd;
	struct qfle3f_rport *target;
	struct qfle3fCommand *ioRequest;
	struct qfle3fFCLun *fclun,  *temp_fclun;
	struct vmk_ScsiCommand *pscsi_cmd = NULL;
	vmk_uint32 cmd_start, cmd_cur, cmd_wait_time;
	vmk_uint32  transfer_len;
	VMK_ReturnStatus status;

	if (qedfc_find_qfle3fHBA(scsi_cmd->host_no, &hba, 1) != VMK_OK) {
		scsi_cmd->status = EXT_STATUS_DEV_NOT_FOUND;
		return VMK_OK;
	}

	qfle3f_log(hba, LOG_APP, "Enter for host = 0x%x\n",
		scsi_cmd->host_no);
	fclun = temp_fclun = NULL;
	cmd_start = cmd_cur = cmd_wait_time = 0;

	pscsi_cmd = qfle3f_alloc(sizeof(struct vmk_ScsiCommand));
	if (!pscsi_cmd) {
		scsi_cmd->status = EXT_STATUS_NO_MEMORY;
		qfle3f_warning(hba, "No memory for the requested cmd (%ld B)\n",
			sizeof(struct vmk_ScsiCommand));
		return VMK_NO_MEMORY;
	}

	if (ql_fcoe_scsi_cmd->data_len > QL_FCOE_MAX_BUFFER_SIZE) {
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		qfle3f_free(pscsi_cmd);
		qfle3f_warning(hba, "Error in data_len (0x%x)\n",
			ql_fcoe_scsi_cmd->data_len);
		return VMK_BAD_PARAM;
	}

	if (hba->ioctl_mem) {
		if (ql_fcoe_scsi_cmd->data_len &&
				ql_fcoe_scsi_cmd->data_len != hba->ioctl_mem_phys.dma_size) {
			qfle3f_dma_free(hba, &hba->ioctl_mem_phys);
			hba->ioctl_mem = qfle3f_dma_alloc(hba,
					&hba->ioctl_mem_phys, ql_fcoe_scsi_cmd->data_len);
		}
	} else {
		hba->ioctl_mem = qfle3f_dma_alloc(hba,
			&hba->ioctl_mem_phys,
			 ql_fcoe_scsi_cmd->data_len ? ql_fcoe_scsi_cmd->data_len : VMK_PAGE_SIZE);
	}

	if (!hba->ioctl_mem) {
		scsi_cmd->status = EXT_STATUS_NO_MEMORY;
		qfle3f_free(pscsi_cmd);
		qfle3f_warning(hba, "Can not alloc requested(0x%x) DMA buffer\n",
			ql_fcoe_scsi_cmd->data_len);
		return VMK_NO_MEMORY;
	}

	switch (ql_fcoe_scsi_cmd->cdb_len) {
		case 0x6:
			pscsi_cmd->cdbLen = 6;
			break;
		case 0x0A:
			pscsi_cmd->cdbLen = 0x0A;
			break;
		case 0x0C:
			pscsi_cmd->cdbLen = 0x0C;
			break;
		case 0x10:
			pscsi_cmd->cdbLen = 0x10;
			break;
		default:
			qfle3f_log(hba, LOG_APP, "Unsupported Cdb Length = 0x%x\n",
				ql_fcoe_scsi_cmd->cdb_len);
			scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
	}
	vmk_Memcpy(pscsi_cmd->cdb, ql_fcoe_scsi_cmd->cdb, pscsi_cmd->cdbLen);

	switch (ql_fcoe_scsi_cmd->direction) {
		case QL_FCOE_DATA_DIRECTION_OUT:
			pscsi_cmd->dataDirection = VMK_SCSI_COMMAND_DIRECTION_WRITE;
			break;
		case QL_FCOE_DATA_DIRECTION_IN:
			pscsi_cmd->dataDirection = VMK_SCSI_COMMAND_DIRECTION_READ;
			break;
		default :
			pscsi_cmd->dataDirection = VMK_SCSI_COMMAND_DIRECTION_NONE;
			break;
	}

	if (ql_fcoe_scsi_cmd->tgt >= QFLE3_FCOE_NUM_CONNECTIONS) {
		qfle3f_warning(hba, "Invalid Target index (0x%x)\n",
			ql_fcoe_scsi_cmd->tgt);
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		status = VMK_INVALID_TARGET;
		goto fail_command;
	}

	target = hba->targetOffloadList[ql_fcoe_scsi_cmd->tgt];
	if (target == NULL) {
		qfle3f_warning(hba, "Invalid Target index (0x%x)\n",
				ql_fcoe_scsi_cmd->tgt);
		scsi_cmd->status = EXT_STATUS_INVALID_PARAM;
		status = VMK_INVALID_TARGET;
		goto fail_command;
	}

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)) {
		pscsi_cmd->status.host = VMK_SCSI_HOST_RETRY;
		goto fail_command;
	}

	if (pscsi_cmd->dataDirection == VMK_SCSI_COMMAND_DIRECTION_WRITE) {
		vmk_Memcpy(hba->ioctl_mem, ql_fcoe_scsi_cmd->data_buffer,
				ql_fcoe_scsi_cmd->data_len);
	}

	pscsi_cmd->done = ql_fcoe_scsi_pt_done;
	pscsi_cmd->doneData = hba;
	pscsi_cmd->sgArray = hba->ioctl_mem_phys.MachSg;

	vmk_SpinlockLock(target->fclunsListLock);
	fclun = qfle3f_getFCLun(hba, target, ql_fcoe_scsi_cmd->lun);
	vmk_SpinlockUnlock(target->fclunsListLock);
	if (!fclun) {
		qfle3f_log(hba, LOG_APP, "Creating a Dummy lun\n");
		temp_fclun = qfle3f_alloc(sizeof(struct qfle3fFCLun));
		fclun = temp_fclun;
		fclun->target = target;
		fclun->vmkLunId = ql_fcoe_scsi_cmd->lun;
	}

	/* Allocate IO request (CC) */
	ioRequest = qfle3f_commandAlloc(target);
	if (!ioRequest) {
		pscsi_cmd->status.host = VMK_SCSI_HOST_RETRY;
		goto fail_command;
	}

	ioRequest->ioctl_flags = 1;
	ioRequest->sc_cmd = pscsi_cmd;
	ioRequest->fclun = fclun;

	hba->cmpl_done = 0;
	hba->SCSIPT_InProgress = 1;

	if (qfle3f_postIORequest(target, ioRequest)) {
		qfle3f_err(hba, "Unable to post ioRequest\n");
		pscsi_cmd->status.host = VMK_SCSI_HOST_RETRY;
		pscsi_cmd->done(pscsi_cmd);
		goto fail_command;
	}

	qfle3f_vmk_spin_lock_init(&hba->ioctl_cmpl_lock, LOCK_RANK_HIGHEST, "ioctl_cmpl_lock");
	vmk_SpinlockLock(hba->ioctl_cmpl_lock);

	/* Wait for completion - Note: we wait longer here then the timer*/
	/* function calledin queuecommand */
	cmd_start = vmk_GetTimerCycles() / vmk_TimerCyclesPerSecond();
	cmd_wait_time = QED_PT_CMD_TOV;

wait_again:
	status = vmk_WorldWait((vmk_WorldEventID)&hba->ioctl_cmpl_sem,
			hba->ioctl_cmpl_lock, cmd_wait_time * VMK_MSEC_PER_SEC,
			"waiting for IOCTL cmd completion");

	/* If VMK_BAD_PARAM is returned, still holding the lock. */
	if(status != VMK_BAD_PARAM) {
		vmk_SpinlockLock(hba->ioctl_cmpl_lock);
	}

	cmd_cur = vmk_GetTimerCycles() / vmk_TimerCyclesPerSecond();

	if (!hba->cmpl_done) {
		if ((cmd_cur - cmd_start) < cmd_wait_time) {
			cmd_wait_time -= (cmd_cur - cmd_start);
			cmd_start = cmd_cur;
			goto wait_again;
		} else {
			qfle3f_warning(hba, "IOCTL cmd timeout");
			vmk_SpinlockUnlock(hba->ioctl_cmpl_lock);
			vmk_SpinlockDestroy(hba->ioctl_cmpl_lock);
			scsi_cmd->status = EXT_STATUS_BUSY;
			return VMK_TIMEOUT;
		}
	}
	vmk_SpinlockUnlock(hba->ioctl_cmpl_lock);

	if (hba->SCSIPT_InProgress == 1) {
		qfle3f_warning(hba, "ERROR passthru command timeout.\n");
		status = VMK_FAILURE;
		goto fail_command;
	}

	if (CMD_IOCTL_COMP_STATUS(pscsi_cmd) == (int)IOCTL_INVALID_STATUS) {
		qfle3f_warning(hba, "ERROR command not completed\n");
		status = VMK_FAILURE;
		goto fail_command;
	}

	switch (CMD_IOCTL_COMP_STATUS(pscsi_cmd)) {
		case CS_INCOMPLETE:
		case CS_ABORTED:
		case CS_PORT_UNAVAILABLE:
		case CS_PORT_LOGGED_OUT:
		case CS_PORT_CONFIG_CHG:
		case CS_PORT_BUSY:
		case CS_TIMEOUT:
			qfle3f_warning(hba, "host err = %x.",
					CMD_IOCTL_COMP_STATUS(pscsi_cmd));
			scsi_cmd->status = EXT_STATUS_BUSY;
			status = VMK_BUSY;
			goto fail_command;
		case CS_RESET:
		case CS_QUEUE_FULL:
			scsi_cmd->status = EXT_STATUS_ERR;
			break;
		case CS_DATA_OVERRUN:
			scsi_cmd->status = EXT_STATUS_DATA_OVERRUN;
			qfle3f_warning(hba, "return overrun.");
			break;
		case CS_DATA_UNDERRUN:
			scsi_cmd->status = EXT_STATUS_DATA_UNDERRUN;
			qfle3f_warning(hba, "return underrun.");
			if (CMD_IOCTL_SCSI_STATUS(pscsi_cmd) & SS_RESIDUAL_UNDER)
				scsi_cmd->status = EXT_STATUS_OK;
			break;
		default:
			vmk_LogMessage("0x%x",
					CMD_IOCTL_COMP_STATUS(pscsi_cmd));
			scsi_cmd->status = EXT_STATUS_ERR;
			break;
	}

	if (CMD_IOCTL_COMP_STATUS(pscsi_cmd) == CS_COMPLETE &&
			CMD_IOCTL_SCSI_STATUS(pscsi_cmd) == 0) {
		scsi_cmd->status = EXT_STATUS_OK;
		qfle3f_log(hba, LOG_APP, "Correct completion");

	} else {
		qfle3f_warning(hba, "scsi err. "
				"host status =0x%x, scsi status = 0x%x.",
				CMD_IOCTL_COMP_STATUS(pscsi_cmd),
				CMD_IOCTL_SCSI_STATUS(pscsi_cmd));

		if (CMD_IOCTL_SCSI_STATUS(pscsi_cmd) & SS_CHECK_CONDITION) {
			scsi_cmd->status = EXT_STATUS_ERR;
		}
	}

	ql_fcoe_scsi_cmd->sense_len = CMD_IOCTL_ACTUAL_SNSLEN(pscsi_cmd);
	if (ql_fcoe_scsi_cmd->sense_len) {
		vmk_ScsiCmdGetSenseData(pscsi_cmd,
				(struct vmk_ScsiSenseData *)ql_fcoe_scsi_cmd->sense,
				ql_fcoe_scsi_cmd->sense_len);
	}

	if (ql_fcoe_scsi_cmd->direction == QL_FCOE_DATA_DIRECTION_IN) {
		qfle3f_log(hba, LOG_APP, "Copying data.\n");

		if ((CMD_IOCTL_COMP_STATUS(pscsi_cmd) == CS_DATA_UNDERRUN) &&
				(vmk_SgGetDataLen(pscsi_cmd->sgArray) - pscsi_cmd->bytesXferred)) {
			transfer_len = pscsi_cmd->bytesXferred;
			ql_fcoe_scsi_cmd->data_len = transfer_len;
		} else
			transfer_len = ql_fcoe_scsi_cmd->data_len;

		qfle3f_log(hba, LOG_APP, "transfer_len = 0x%x\n", transfer_len);
		vmk_Memcpy(ql_fcoe_scsi_cmd->data_buffer, hba->ioctl_mem,
				transfer_len);
	}

	vmk_SpinlockDestroy(hba->ioctl_cmpl_lock);
fail_command:
	if (hba->ioctl_mem) {
		qfle3f_dma_free(hba, &hba->ioctl_mem_phys);
		hba->ioctl_mem = NULL;
	}

	qfle3f_free(pscsi_cmd);
	return VMK_OK;
}
