/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

#include "qfle3i.h"
#include "qfle3i_vmk.h"

struct qfle3i_driver_info_t qfle3i_driver_info;
vmk_LogComponent qfle3i_vmkLog;

static vmk_ListLinks adapter_list;

struct qfle3i_hba* hba_list[QFLE3I_MAX_HBA];
vmk_BitVector *qfle3i_host_instance;
vmk_uint32 adapter_count;

/*
 * A default, empty vmk_MgmtApiSignature on which we can attach
 * key-value pairs.
 */
vmk_MgmtApiSignature kvSig = {
	.version = VMK_REVISION_FROM_NUMBERS(1,0,0,0),
	.name.string = "qfle3i",
	.vendor.string = "qlogic",
	.numCallbacks = 0,
	.callbacks = NULL,
};
/* Driver Version */
char qfle3i_version_str[40];

extern vmk_MgmtApiSignature ql_iscsi_sig;
extern vmk_MgmtHandle qed_iscsi_vmkmgmt_api_handle;

static vmk_Semaphore qfle3i_dev_lock;
//static DEFINE_MUTEX(qfle3i_dev_lock);

unsigned int event_coal_div = 1;
VMK_MODPARAM(event_coal_div, int,
	"Event Coalescing Divide Factor, default:1");

unsigned int cmd_cmpl_per_work = QFLE3I_CQ_WQES_DEFAULT;
VMK_MODPARAM(cmd_cmpl_per_work, int,
	"Number of CQE's processed per work, min:16; max:256; default:24");

unsigned int tcp_buf_size = QFLE3I_TCP_WINDOW_DEFAULT;
VMK_MODPARAM(tcp_buf_size, int,
	"TCP send/receive buffer size, min:16k; max:1MB; default:64k");

unsigned int qfle3i_nopout_when_cmds_active = 1;
VMK_MODPARAM(qfle3i_nopout_when_cmds_active, int,
	"iSCSI NOOP even when connection is not idle, 0:disable; 1:enable(default)");

unsigned int en_tcp_dack = 1;
VMK_MODPARAM(en_tcp_dack, int,
	"Enable TCP Delayed ACK, 0:disable; 1:enable(default)");

unsigned int time_stamps = 1;
VMK_MODPARAM(time_stamps, int,
	"Enable TCP TimeStamps, default:1");

unsigned int error_mask1 = 0x0;
VMK_MODPARAM(error_mask1, uint,
	"Config FW iSCSI Error Mask #1, default:0x0");

unsigned int error_mask2 = 0x0;
VMK_MODPARAM(error_mask2, uint,
	"Config FW iSCSI Error Mask #2, default:0x0");

unsigned int sq_size = QFLE3I_5770X_SQ_WQES_DEFAULT;
VMK_MODPARAM(sq_size, int,
	"Configure SQ size, min:16; max:512; default:128");

unsigned int rq_size = QFLE3I_RQ_WQES_DEFAULT;
VMK_MODPARAM(rq_size, int,
	"Configure RQ size, min:16(default) max:32");

unsigned int event_coal_min = 24;
VMK_MODPARAM(event_coal_min, int,
	"Event Coalescing Minimum Commands, default:24");

unsigned int qfle3i_esx_mtu_max = 9000;
VMK_MODPARAM(qfle3i_esx_mtu_max, int,
	"Max MTU Supported for Offload Sessions, default:9000");

unsigned int en_hba_poll = 0;
VMK_MODPARAM(en_hba_poll, int,
	"Enable HBA poll timer, 0:disable(default), 1:enable");

unsigned int qfle3i_chip_cmd_max = 24;
VMK_MODPARAM(qfle3i_chip_cmd_max, int,
	"Max IOs queued to chip, default:24");

unsigned int qfle3i_max_task_pgs = 2;
VMK_MODPARAM(qfle3i_max_task_pgs, int,
	"Maximum pages allocated for iSCSi tasks per connection\n"
	"\t min:2(default); max:8");

unsigned int qfle3i_debug_level = 0x0;
VMK_MODPARAM(qfle3i_debug_level, int,
		"Bit mask to enable/disable debug logs\n"
		    "\tInitalization:\t0x01\n"
		    "\tConn Setup:\t0x02\n"
		    "\tTMF:\t\t0x04\n"
		    "\tiSCSI NOP:\t0x08\n"
		    "\tCNIC IF:\t0x10\n"
		    "\tITT CLEANUP:\t0x20\n"
		    "\tCONN EVT:\t0x40\n"
		    "\tSESS Recovery:\t0x80\n"
		    "\tInternal Messages:\t0x100\n"
		    "\tIO Path:\t0x200\n");

unsigned int ooo_enable = 1;
VMK_MODPARAM(ooo_enable, int,
	"Enable TCP out-of-order support, 0:disable; 1:enable(default)");

unsigned int qfle3i_ssan_feature = 1;
VMK_MODPARAM(qfle3i_ssan_feature, int,
	"SmartSAN feature, 0:disable; 1:enable(default)");

unsigned int qfle3i_dump_queue = 1;
VMK_MODPARAM(qfle3i_dump_queue, int,
	"dump sp, cq and spq, 0:disable; 1:enable(default)");

vmk_uint64 iscsi_error_mask = 0x00;

struct qfle3i_driver_info_t qfle3i_driver_info;

extern struct cnic_ulp_ops qfle3i_cnic_cb;
vmk_Lock qfle3i_resc_lock; /* protects global data structures */
extern struct tcp_port_mngt qfle3i_tcp_port_tbl;

//static void qfle3i_unreg_one_device(struct qfle3i_hba *hba) ;
void qfle3i_unbind_adapter_devices(struct qfle3i_hba *hba);


/* Validate module parameter's range and zero value
 */
static void qfle3i_param_check_range(void)
{
	if (!event_coal_div)
		event_coal_div = 1;
	if (!cmd_cmpl_per_work) {
		vmk_LogMessage("qfle3i: cmd_cmpl_per_work %d out of range,"
			" using %d\n", cmd_cmpl_per_work, QFLE3I_CQ_WQES_MIN);
		cmd_cmpl_per_work = QFLE3I_CQ_WQES_MIN;
	}

	tcp_buf_size = QL_VMK_ROUNDUP_POW_OF_TWO(tcp_buf_size);
	if (tcp_buf_size < QFLE3I_TCP_WINDOW_MIN ||
	    tcp_buf_size > QFLE3I_TCP_WINDOW_MAX) {
		vmk_LogMessage("qfle3i: TCP window %d out of range, using %d\n",
			tcp_buf_size, QFLE3I_TCP_WINDOW_DEFAULT);
		tcp_buf_size = QFLE3I_TCP_WINDOW_DEFAULT;
	}
}

void qfle3i_remove_hba_from_adapter_list(struct qfle3i_hba *hba)
{
	vmk_SemaLock(&qfle3i_dev_lock);
	if (!vmk_ListIsEmpty(&hba->link)) {
		/* detach hba from active list */
		hba_list[hba->instance] = NULL;
		vmk_BitVectorClear(qfle3i_host_instance, hba->instance);
		adapter_count--;
		vmk_ListRemove(&hba->link);
	}
	vmk_SemaUnlock(&qfle3i_dev_lock);
}

void qfle3i_add_hba_to_adapter_list(struct qfle3i_hba *hba)
{
	vmk_uint32 result;
	VMK_ReturnStatus status = VMK_OK;
	vmk_SemaLock(&qfle3i_dev_lock);
	vmk_ListInsert(&hba->link, vmk_ListAtRear(&adapter_list));

	status = vmk_BitVectorNextBit(qfle3i_host_instance, 0, VMK_FALSE, &result);
	if ((status != VMK_TRUE) &&  (result >= MAX_HBA)) {
		vmk_LogMessage("%s: %d: Host instance exhausted\n",
			__func__, __LINE__);
		vmk_ListRemove(&hba->link);
		vmk_SemaUnlock(&qfle3i_dev_lock);
		return;
	}
	hba->instance = result;
	vmk_BitVectorSet(qfle3i_host_instance, hba->instance);
	adapter_count++;
	hba_list[hba->instance] = hba;
	vmk_SemaUnlock(&qfle3i_dev_lock);
}


/*
 * qfle3i_find_hba_for_cnic - maps cnic device instance to qfle3i adapter instance
 *
 * @cnic: 		pointer to cnic device instance
 *
 **/
struct qfle3i_hba *qfle3i_find_hba_for_cnic(struct cnic_dev *cnic)
{
	struct qfle3i_hba *hba;

	vmk_SemaLock(&qfle3i_dev_lock);
	ql_vmk_list_each_entry(hba, &adapter_list, link, qfle3i_hba) {
		if (hba->cnic == cnic) {
			vmk_SemaUnlock(&qfle3i_dev_lock);
			return hba;
		}
	}
	vmk_SemaUnlock(&qfle3i_dev_lock);
	return NULL;
}


/*
 * qfle3i_get_stats - Retrieve various statistic from iSCSI offload
 * @handle:		qfle3i_hba
 *
 * function callback exported via qfle3i - cnic driver interface to
 *	retrieve various iSCSI offload related statistics.
 */
int qfle3i_get_stats(void *handle)
{
	struct qfle3i_hba *hba = handle;
	struct iscsi_stats_info *stats = NULL;
	struct qfle3i_sess *sess = NULL;

	if (!hba)
		return VMK_BAD_PARAM;

	if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_FW_RECOVERY))
		return 0;

	stats = (struct iscsi_stats_info *)hba->cnic->stats_addr;

	if (!stats)
		return VMK_NO_MEMORY;

	vmk_Memcpy(stats->version, DRV_MODULE_VERSION, sizeof(stats->version));
//	vmk_Memcpy(stats->mac_add1 + 2, hba->cnic->mac_addr, ETH_ALEN);
	stats->max_frame_size = hba->cnic->mtu;
	stats->txq_size = hba->max_sqes;
	stats->rxq_size = hba->max_cqes;

	/* Loop through all ep to get the cqe_left average */
	stats->txq_avg_depth = 0;
	stats->rxq_avg_depth = 0;

	vmk_SpinlockLock(hba->stat_lock);
	GET_STATS_64(hba, stats, rx_pdus);
	GET_STATS_64(hba, stats, rx_bytes);
	GET_STATS_64(hba, stats, tx_pdus);
	GET_STATS_64(hba, stats, tx_bytes);
	vmk_SpinlockUnlock(hba->stat_lock);

	vmk_Memset(&stats->boot_target_ip, 0, sizeof(stats->boot_target_ip));
	vmk_Memset(&stats->boot_init_ip, 0, sizeof(stats->boot_init_ip));

	vmk_SpinlockLock(hba->lock);
	ql_vmk_list_each_entry(sess, &hba->active_sess, link, struct qfle3i_sess) {
		if (sess->lead_conn && sess->lead_conn->ep) {
			if (vmk_BitVectorTest(sess->lead_conn->ep->cm_sk->flags, SK_F_IPV6)) {
				vmk_Memcpy(stats->boot_target_ip, sess->lead_conn->ep->cm_sk->dst_ip,
						IPv6_ADDRLEN);
				vmk_Memcpy(stats->boot_init_ip, (void *)hba->cnic->ipv6_addr.__u6_addr.__u6_addr8,
						IPv6_ADDRLEN);
			} else {
				/* If IP address is IPv4, Set bytes 10 and 11 in
				 *               * IP address field to 0xFF,
				 *               * Copy IP address to bytes 12-15
				 *
				 */
				vmk_Memset(&stats->boot_target_ip[10], 0xFF, 2*sizeof(char));
				vmk_Memset(&stats->boot_init_ip[10], 0xFF, 2*sizeof(char));
				vmk_Memcpy(&stats->boot_target_ip[12], &sess->lead_conn->ep->cm_sk->dst_ip[0],
						IPv4_ADDRLEN);
				vmk_Memcpy(&stats->boot_init_ip[12], &sess->lead_conn->ep->cm_sk->src_ip[0],
						IPv4_ADDRLEN);
			}
			break;
		}
	}
	vmk_SpinlockUnlock(hba->lock);
	return 0;
}

void qfle3i_link_update(void *ulp_ctx, unsigned long event)
{
	struct qfle3i_hba *hba = ulp_ctx;
	if (!hba) {
		vmk_WarningMessage("%s:%d: Fix this, it should not happen.\n",
				__func__, __LINE__);
		return;
	}

	if(event == 1) {
		PRINT_INFO(hba, "Link Up Event\n");
		hba->is_nic_up = 1;
	} else {
		PRINT_INFO(hba, "Link Down Event\n");
		hba->is_nic_up = 0;
		if(vmk_BitVectorTest(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED) &&
				(!vmk_ListIsUnlinkedElement(&(hba->link_update).links)) &&
				(hba->num_active_sess != 0))
			ql_vmk_singlethread_queue_work(qfle3i_driver_info.delayed_wq,
					&hba->link_update);
	}
}

/*
 * qfle3i_start - cnic callback to initialize & start adapter instance
 *
 * @handle: 		transparent handle pointing to adapter structure
 *
 * This function maps adapter structure to pcidev structure and initiates
 *	firmware handshake to enable/initialize on chip iscsi components
 * 	This qfle3i - cnic interface api callback is issued after following
 *	2 conditions are met -
 *	  a) underlying network interface is up (marked by event 'NETDEV_UP'
 *		from netdev
 *	  b) qfle3i adapter instance is registered
 **/
void qfle3i_start(void *handle)
{
#define QFLE3I_INIT_POLL_TIME	(1000 / vmk_TimerCyclesPerSecond())
	struct qfle3i_hba *hba = handle;
	int i = vmk_TimerCyclesPerSecond();

	PRINT_INFO(hba, "qfle3i_start called for hba:%p\n", hba);
	if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_FW_RECOVERY)) {
		if (hba->cnic) {
			hba->regview = hba->cnic->regview;
			hba->doorbells = hba->cnic->doorbells;
		}
	}

	qfle3i_send_fw_iscsi_init_msg(hba);
	do {
		if (vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_UP) ||
		    vmk_BitVectorTest(hba->adapter_state, ADAPTER_STATE_INIT_FAILED))
			break;
		PRINT_INFO(hba, "Wait counter:%d \n", i);
		vmk_WorldSleep(QFLE3I_INIT_POLL_TIME * VMK_USEC_PER_MSEC);
	} while (i--);
}

/*
 * qfle3i_stop - cnic callback to shutdown adapter instance
 *
 * @handle: 		transparent handle pointing to adapter structure
 *
 * driver checks if adapter is already in shutdown mode, if not start
 *	the shutdown process
 **/
void qfle3i_stop(void *handle)
{
	struct qfle3i_hba *hba = handle;
	/* qfle3i will have to wait this long because it depends
	 * many other modules to complete the assigned tasks,
	 * inflight connections to be established/logged in,
	 * or clean-up the active connections. It depends of
	 * outside agents such as vmkiscsid, cnic, firmware to
	 * to complete this process.
	 */
	int wait_num_secs = 0;
	int timeout, ulp_tmo;
	int repeat = 0;

	PRINT_INFO(hba, "qfle3i_stop called for hba:%p\n", hba);
	/* cnic will guarantee 'ulp_stop' will be called only once */
	vmk_BitVectorSet(hba->adapter_state, ADAPTER_STATE_GOING_DOWN);
	/* Initially wait 40-secs and later keep polling every second for
	 * both previously active and inflight connections to be torn down.
	 */

	if (vmk_AtomicRead64(&hba->ep_tmo_poll_enabled))
		ql_vmk_world_wakeup(&hba->ep_tmo_wait);

   	if (vmk_SystemCheckState(VMK_SYSTEM_STATE_NORMAL) == VMK_TRUE) {
		timeout = 40;
		ulp_tmo = 40;

		while (hba->ofld_conns_active) {
			qfle3i_start_iscsi_hba_shutdown(hba);

			/* timeout value should be in msec */
			ql_vmk_wait_for_completion(&hba->eh_wait,
								(hba->ofld_conns_active == 0),
								(timeout * VMK_MSEC_PER_SEC));
			wait_num_secs += timeout;
			if (ulp_tmo == 40) {
				vmk_LogMessage("%s:%d %s - hba %p, ulp_stop wait time %d seconds (%d)\n",
					__func__, __LINE__, vmk_NameToString(&hba->vmnic_name), hba, 
					wait_num_secs, hba->ofld_conns_active);
				ulp_tmo = 0;
			} else
				ulp_tmo++;

			repeat++;
			timeout = VMK_MSEC_PER_SEC;
		}
	}
	else {
	/*
	* If network stop event is received when not in normal system
	* state we are in shutdown.  At this point vmkiscsid is already
	* dead so active connection cleanup processing won't complete.
	* just skip full processing since we are in the shutdown path.
	*/
		qfle3i_iscsi_hba_cleanup(hba);
		vmk_AlertMessage("%s:%d %s - hba %p, stop event during shutdown\n",
			__func__, __LINE__, vmk_NameToString(&hba->vmnic_name), hba);
	}
	vmk_LogMessage("%s:%d %s - hba %p, ulp_stop completed in %d seconds\n",
		__func__, __LINE__,  vmk_NameToString(&hba->vmnic_name), hba,
		wait_num_secs);

	if (repeat)
		hba->stop_event_repeat++;

	if (hba->ofld_conns_active)
		vmk_AlertMessage("%s:%d %s - hba %p, there are still some active "
				"connections, %d at the end of stop call\n",
				__func__, __LINE__, vmk_NameToString(&hba->vmnic_name), hba,
				hba->ofld_conns_active);

	vmk_BitVectorClear(hba->adapter_state, ADAPTER_STATE_UP);
	vmk_BitVectorClear(hba->adapter_state, ADAPTER_STATE_GOING_DOWN);
}


/*
 * qfle3i_init_one - initialize an adapter instance and allocate necessary
 *		memory resources
 *
 * @hba: 		qfle3i adapter instance
 * @cnic: 		cnic device handle
 *
 * Global resource lock and host adapter lock is held during critical sections
 *	below. This routine is called from cnic_register_driver() context and
 *	work horse thread which does majority of device specific initialization
 **/
static int qfle3i_init_one(struct qfle3i_hba *hba, struct cnic_dev *cnic)
{
	int rc = 0;

	vmk_SemaLock(&qfle3i_dev_lock);
#if 0
	hba->netdev = cnic->netdev;
#endif
	rc = cnic->register_device(cnic, CNIC_ULP_ISCSI, (void *)hba, &qfle3i_cnic_cb);
	if (!rc) {
		hba->age++;
		vmk_BitVectorSet(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED);
	} else if (rc == VMK_BUSY) 	/* duplicate registration */
		PRINT_ALERT(hba, "duplicate registration"
				  "cnic=%p\n", cnic);
	else if (rc == VMK_STATUS_PENDING)
		PRINT_ERR(hba, "driver not registered\n");
	else if (rc == VMK_BAD_PARAM)
		PRINT_ERR(hba, "invalid type %d\n", CNIC_ULP_ISCSI);
	else
		PRINT_ERR(hba, "reg, unknown error, %d\n", rc);

	if (rc)
		goto ret_err;

	if (ooo_enable)
		vmk_BitVectorSet(cnic->flags, CNIC_F_ISCSI_OOO_ENABLE);
	else
		vmk_BitVectorClear(cnic->flags, CNIC_F_ISCSI_OOO_ENABLE);

	PRINT_INFO(hba, "returning success from qfle3i_init_one \n");

ret_err:
	vmk_SemaUnlock(&qfle3i_dev_lock);
	return rc;
}

void qfle3i_fw_recovery_event(void *handle, int event)
{
	struct qfle3i_hba *hba = handle;

	if (event == CNIC_CTL_NIC_RECOVERY_CMD) {
		PRINT_INFO(hba, "FW recovery initiated for hba:%p \n", hba);
		vmk_BitVectorSet(hba->adapter_state, ADAPTER_STATE_FW_RECOVERY);
	} else if (event == CNIC_CTL_NIC_RECOVERY_DONE_CMD) {
		PRINT_INFO(hba, "FW recovery done for hba:%p \n", hba);
		vmk_BitVectorClear(hba->adapter_state, ADAPTER_STATE_FW_RECOVERY);
	}
		PRINT_INFO(hba, "Exit. \n");
}
/*
 * Driver attach/detach/start/scan 
 */

static VMK_ReturnStatus
qfle3i_removeDevice(vmk_Device device)
{
	vmk_ScsiAdapter *scsiAdapter = NULL;
	qfle3i_hba *hba = NULL;
	VMK_ReturnStatus status = VMK_OK;

	status = vmk_DeviceGetRegistrationData(device,
				(vmk_AddrCookie *)&scsiAdapter);
	if ((status != VMK_OK) || (scsiAdapter == NULL)) {
		vmk_WarningMessage("Can't get scsi adapter, status:%s\n",
						vmk_StatusToString(status));
		return status;
	}

	hba = (qfle3i_hba *)scsiAdapter->clientData;
	QFLE3I_DBG(DBG_INIT, hba, "hba:%p hostnum:%d\n",
					hba, hba->host_num);

	/* Cleaning up stuff done in scanDevice over here. */
	status = vmk_DeviceUnregister(device);
	if (status != VMK_OK) {
		vmk_WarningMessage("Failed to unregister adapter status:%s\n",
						vmk_StatusToString(status));
		return status;
	}
	return status;
}

static vmk_DeviceOps qfle3iDeviceOps = {
   .removeDevice = qfle3i_removeDevice,
};

static VMK_ReturnStatus qfle3i_attachDevice(vmk_Device device)
{
	VMK_ReturnStatus status = VMK_OK;
	qfle3i_hba *hba = NULL;
	struct cnic_dev *cnic;

	PRINT_INFO(hba, "Entered.\n");

	status = vmk_DeviceGetRegistrationData(device,
						(vmk_AddrCookie *)&cnic);
	if ((status != VMK_OK) || (!cnic)) {
		PRINT_ERR(hba, "error in vmk_DeviceGetRegistrationData status:%s\n",
						vmk_StatusToString(status));
		return VMK_BAD_PARAM;
	}

	hba = qfle3i_alloc_hba(cnic, device);
	if (hba == NULL) {
		PRINT_ERR(hba, "setting up hba failed\n");
		return VMK_NO_MEMORY;
	}

	hba->isRegisteredWithTransport = VMK_FALSE;

	QFLE3I_DBG(DBG_INIT, hba, "hba memory allocated, addr:0x%lx\n",
					(unsigned long)hba);
	/* Lets register with cnic over here */
	/* Get PCI related information and update hba struct members */
	vmk_BitVectorClear(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED);
	if (qfle3i_init_one(hba, cnic)) {
		PRINT_ERR(hba, "qfle3i_init_one failed\n");
		status = VMK_FAILURE;
		goto reg_with_cnic_failed;
	}

	hba->cnic = cnic;
#if 0
	/* copy mac address from cnic_dev */
	vmk_Memcpy(hba->mac_addr, cnic->nic_mac, VMK_ETH_ADDR_LENGTH);
	vmk_NameCopy(&hba->vmnic_name, &cnic->uplinkName);

	QFLE3I_DBG(DBG_INIT, hba, "nic_name:%s "
		"mac-addr: %02hhx:%02hhx:%02hhx:%02hhx:%02hhx:%02hhx\n",
		vmk_NameToString(&hba->vmnic_name),
		hba->mac_addr[0], hba->mac_addr[1], hba->mac_addr[2],
		hba->mac_addr[3], hba->mac_addr[4], hba->mac_addr[5]);
#endif

#if 0
	status = qfle3i_register_transport(hba);
	if (status != VMK_OK) {
		PRINT_INFO(hba, "qfle3i_register_transport failed. status:%s\n",
						vmk_StatusToString(status));
		goto reg_trans_failed;
	}
#endif

        /* allocate scsi Adapter structure */
	hba->scsiAdapter = vmk_ScsiAllocateAdapter();
	if (hba->scsiAdapter == NULL) {
		PRINT_ERR(hba, "memory allocation for scsiadapter failed. status:%s\n",
						vmk_StatusToString(status));
		goto allocate_adapter_failed;
	}

	status = vmk_DeviceSetAttachedDriverData(device, hba);
	if (status != VMK_OK) {
		PRINT_INFO(hba, "vmk_DeviceSetAttachedDriverData failed. status:%s\n",
						vmk_StatusToString(status));
		goto set_data_failed;
	}

	PRINT_INFO(hba, "returning success.\n");
	return VMK_OK;

set_data_failed:
	vmk_ScsiFreeAdapter(hba->scsiAdapter);
allocate_adapter_failed:
	cnic->unregister_device(cnic, CNIC_ULP_ISCSI);
	vmk_BitVectorClear(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED);
reg_with_cnic_failed:
	qfle3i_free_hba(hba);

	PRINT_ERR(hba, "Returned (VMK_FAILURE)\n");
	return status;
}

static VMK_ReturnStatus qfle3i_detachDevice(vmk_Device device)
{
	VMK_ReturnStatus status = VMK_OK;
	qfle3i_hba *hba = NULL;

	PRINT_INFO(hba, "Entered\n");

	status = vmk_DeviceGetAttachedDriverData(device,
					(vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		DRV_WARN("Failed to get pci device handle status\n", status);
		return status;
	}
	if (hba->isRegisteredWithTransport == VMK_TRUE) {
		if (hba->kvMgmtHandle && hba->mgmtKeyAdded == VMK_TRUE) {
			vmk_MgmtDestroy(hba->kvMgmtHandle);
			hba->mgmtKeyAdded = VMK_FALSE;
		}
		/* Unbind iSCSI adapter from iSCSI transport */
		qfle3i_unreg_iscsi_adapter(hba);
		/* Unregister from iscsi_transport */
		if (hba->iscsiTransport)
			qfle3i_unregister_transport(hba);
		hba->isRegisteredWithTransport = VMK_FALSE;
	}

	hba->cnic->unregister_device(hba->cnic, CNIC_ULP_ISCSI);
	vmk_BitVectorClear(hba->reg_with_cnic, QFLE3I_CNIC_REGISTERED);

retry:
	if (!vmk_ListIsUnlinkedElement(&(hba->link_update).links)) {
		vmk_WorldSleep(1000 * VMK_USEC_PER_MSEC);
		goto retry;
	}

	if (hba->scsiAdapter) {
		vmk_ScsiFreeAdapter(hba->scsiAdapter);
		hba->scsiAdapter = NULL;
	}
	qfle3i_free_hba(hba);

#if 0
/* VK: this need to be shifted to opposite of scanDevice
		as scsiadapter allocation is done in scanDevice
*/
	status = qfle3i_unregister_transport(hba);
	if (status != VMK_OK) {
		DRV_WARN("Failed to destory iscsi hba status\n", status);
		return status;
	}

	status = vmk_DMAEngineDestroy(hba->scsiAdapter->engine);
	if (status != VMK_OK) {
		DRV_WARN("Failed to destory dma engine status", status);
		return status;
	}

	vmk_ScsiFreeAdapter(hba->scsiAdapter);
	hba->scsiAdapter = NULL;
	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), hba);
#endif
	PRINT_INFO(hba, "returning success.\n");
	return status;
}

static VMK_ReturnStatus qfle3i_scanDevice(vmk_Device device)
{
	VMK_ReturnStatus status = VMK_OK;
	vmk_BusType bus_type;
	vmk_Name bus_name;
	qfle3i_hba *hba = NULL;
	vmk_DeviceID dev_id;
	vmk_DeviceProps device_props;

	PRINT_INFO(hba, "Entered.\n");

	status = vmk_DeviceGetAttachedDriverData(device,
 					(vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		DRV_WARN("Failed to get qfle3i hba\n", status);
		return status;
	}

	/* setup scsi adapter */
	status = qfle3i_setup_scsiadapter(hba);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "qfle3i_setup_scsiadapter failed\n");
		goto setup_adapter_failed;
	}

	hba->host_num = hba->scsiAdapter->hostNum;
	status = vmk_NameInitialize(&bus_name, VMK_LOGICAL_BUS_NAME);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "Failed to initialize qfle3i name. status:%s\n",
						vmk_StatusToString(status));
		goto name_initialize_failed;
	}

	status = vmk_BusTypeFind(&bus_name, &bus_type);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "Failed to BusTypeFind status:%s\n",
						vmk_StatusToString(status));
		goto bustype_find_failed;
	}

	status = vmk_LogicalCreateBusAddress(
				qfle3i_driver_info.qfle3i_driver,
				device, 0, &dev_id.busAddress,
				&dev_id.busAddressLen);

	if ( status != VMK_OK) {
		PRINT_ERR(hba, "Failed to created logicalbus. status:%s\n",
				vmk_StatusToString(status));
		goto  fail_create_logical_bus;
	}

	dev_id.busType = bus_type;
	dev_id.busIdentifier = VMK_SCSI_PSA_DRIVER_BUS_ID;
	dev_id.busIdentifierLen = vmk_Strnlen(dev_id.busIdentifier,
								VMK_MISC_NAME_MAX);
	device_props.registeringDriver = qfle3i_driver_info.qfle3i_driver;
	device_props.deviceID = &dev_id;
	device_props.deviceOps = &qfle3iDeviceOps;
	device_props.registeringDriverData.ptr = hba;
	device_props.registrationData.ptr = hba->scsiAdapter;

	status = vmk_DeviceRegister(&device_props, device, &hba->iSCSIDevice);
	if (status != VMK_OK) {
		PRINT_ERR(hba, "Failed to register device. status:%s\n",
						vmk_StatusToString(status));
 		goto device_register_failed;
	}

	vmk_LogicalFreeBusAddress(qfle3i_driver_info.qfle3i_driver,
							dev_id.busAddress);
	vmk_BusTypeRelease(bus_type);

	QFLE3I_DBG(DBG_INIT, hba, "returning success.scsiAdaper->hostNum:%d\n",
						hba->scsiAdapter->hostNum);
	return status;

device_register_failed:
	vmk_LogicalFreeBusAddress(qfle3i_driver_info.qfle3i_driver,
							dev_id.busAddress);
fail_create_logical_bus:
	vmk_BusTypeRelease(bus_type);
bustype_find_failed:
name_initialize_failed:
setup_adapter_failed:
	PRINT_ERR(hba, "returning error \n");
	 return status;
}

static VMK_ReturnStatus qfle3i_quiesceDevice(vmk_Device device)
{
	VMK_ReturnStatus status = VMK_OK;
	qfle3i_hba *hba = NULL;

	PRINT_INFO(hba, "Entered\n");

	status = vmk_DeviceGetAttachedDriverData(device,
				(vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		DRV_WARN("Failed to get pci device handle status\n", status);
		return status;
	}

	/* block unload if sessions are not removed */

	if (hba->num_active_sess || hba->ofld_conns_active) {
		PRINT_INFO(hba, "unload during active sess/conn, return VMK_FAILURE.\n");
		return VMK_FAILURE;
	}
	PRINT_INFO(hba, "returning success.\n");
	return VMK_OK;
}

static VMK_ReturnStatus qfle3i_startDevice(vmk_Device device)
{
	VMK_ReturnStatus status = VMK_OK;
	qfle3i_hba *hba = NULL;

	PRINT_INFO(hba, "Entered\n");

	status = vmk_DeviceGetAttachedDriverData(device,
				(vmk_AddrCookie *)&hba);
	if (status != VMK_OK) {
		DRV_WARN("Failed to get pci device handle status\n", status);
		return status;
	}

	PRINT_INFO(hba, "Returned (VMK_OK)\n");
	return status;
}

static void qfle3i_forgetDevice(vmk_Device device)
{
	DRV_DEBUG(0, "Entered\n");

	DRV_DEBUG(0, "Returned (VMK_OK)\n");
}

VMK_ReturnStatus qfle3i_set_driver_info(vmk_uint64 cookie, void *stringIn)
{
	int length = vmk_Strnlen("qfle3i_debug_level=", 4096);
	vmk_LogMessage("stringIn:%s\n", (char *)stringIn);

	if (vmk_Strncmp(stringIn, "qfle3i_debug_level=",
		length) == 0) {
		vmk_LogMessage("Before qfle3i_debug_level:0x%x\n", qfle3i_debug_level);
		vmk_Sscanf((stringIn + length), "%x", &qfle3i_debug_level);

		if (qfle3i_debug_level == 1)
			qfle3i_debug_level = DEF_DBG_LEVEL;

		vmk_LogMessage("After qfle3i_debug_level:0x%x\n", qfle3i_debug_level);
	}
	return VMK_OK;
}

VMK_ReturnStatus qfle3i_get_driver_info(vmk_uint64 cookie, void *string_out)
{
	int length = 0;
	int final_length = MAX_KEY_VAL_STRING_LEN;

	char *buffer = vmk_HeapAlign(vmk_ModuleGetHeapID(vmk_ModuleCurrentID),
								(final_length+1), VMK_L1_CACHELINE_SIZE);
	if (!buffer) {
		vmk_LogMessage("memory allocation failed.\n");
		return VMK_NO_MEMORY;
	}

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\nDriver: %s-%s\n", QFLE3I_DRIVER_NAME, qfle3i_version_str);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\n #Module Parameters:\n");
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tqfle3i_debug_level: 0x%x\n", qfle3i_debug_level);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tevent_coal_div: 0x%x\n", event_coal_div);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tcmd_cmpl_per_work: 0x%x\n", cmd_cmpl_per_work);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\ttcp_buf_size: 0x%x\n", tcp_buf_size);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tqfle3i_nopout_when_cmds_active: 0x%x\n", qfle3i_nopout_when_cmds_active);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\ten_tcp_dack: 0x%x\n", en_tcp_dack);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\ttime_stamps: 0x%x\n", time_stamps);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\terror_mask1: 0x%x\n", error_mask1);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\terror_mask2: 0x%x\n", error_mask2);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tsq_size: 0x%x\n", sq_size);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\trq_size: 0x%x\n", rq_size);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tevent_coal_min: 0x%x\n", event_coal_min);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tqfle3i_esx_mtu_max: 0x%x\n", qfle3i_esx_mtu_max);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tqfle3i_esx_mtu_max: 0x%x\n", qfle3i_esx_mtu_max);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\ten_hba_poll: 0x%x\n", en_hba_poll);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tqfle3i_chip_cmd_max: 0x%x\n", qfle3i_chip_cmd_max);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tqfle3i_max_task_pgs: 0x%x\n", qfle3i_max_task_pgs);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tooo_enable: 0x%x\n", ooo_enable);
	length = vmk_Strnlen(buffer, final_length);

	vmk_StringFormat(buffer+length, final_length - length, NULL,
	"\tqfle3i_dump_queue: 0x%x\n", qfle3i_dump_queue);
	length = vmk_Strnlen(buffer, final_length);

	if (length < final_length)
		vmk_StringCopy(string_out, buffer, length);
	else {
		vmk_LogMessage("Too much data! Cannot write full data in mgmt_info. Data truncated.");
		vmk_StringCopy(string_out, buffer, final_length);
	}

	vmk_HeapFree(vmk_ModuleGetHeapID(vmk_ModuleCurrentID), buffer);
	return VMK_OK;
}

static vmk_DriverOps qfle3i_iSCSIOps = {
	.attachDevice = qfle3i_attachDevice,
	.scanDevice = qfle3i_scanDevice,
	.detachDevice = qfle3i_detachDevice,
	.quiesceDevice = qfle3i_quiesceDevice,
	.startDevice = qfle3i_startDevice,
	.forgetDevice = qfle3i_forgetDevice,
};


int init_module(void)
{
	vmk_HeapID heap_id;
	vmk_Name heap_name;
	vmk_HeapCreateProps heap_props;
	VMK_ReturnStatus vmk_stat = VMK_OK;
#if (VMWARE_ESX_DDK_VERSION <= 65000)
	vmk_ModuleID module_id;
#endif
	vmk_DriverProps qfle3i_drv_props;
	vmk_LogProperties log_props;
	vmk_TimerQueueProps timerqueue_props;
	vmk_SpinlockCreateProps lock_props;

	vmk_Name kv_name;
	vmk_Name vmhba_name;
	vmk_MgmtProps mgmtProps;

	/*
	* Register module with vmkernel
	*/
#if (VMWARE_ESX_DDK_VERSION <= 65000)
	vmk_stat = vmk_ModuleRegister(&module_id, VMKAPI_REVISION);
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("qfle3i:vmk_ModuleRegister failed: %s",
				vmk_StatusToString(vmk_stat));
		vmk_WarningMessage("qfle3i:Could not load %s module",
				QFLE3I_DRIVER_NAME);
	goto module_register_fail;
	}
	qfle3i_driver_info.module_id = module_id;
#else
	qfle3i_driver_info.module_id = vmk_ModuleCurrentID;
#endif

	/*
	* Set driver name here
	*/
	vmk_stat = vmk_NameInitialize(&qfle3i_driver_info.driver_name,
				QFLE3I_DRIVER_NAME);

	/*SV: make separate module heap*/
	heap_props.type = VMK_HEAP_TYPE_SIMPLE;
	vmk_NameFormat(&heap_name, "%s", QFLE3I_DRIVER_NAME);
	vmk_stat = vmk_NameInitialize(&heap_props.name, (const char*)&heap_name);
	heap_props.module = vmk_ModuleCurrentID;
	heap_props.initial = QFLE3I_HEAP_INITIAL_SIZE;
	heap_props.max = QFLE3I_HEAP_MAXIMUM_SIZE;
	heap_props.creationTimeoutMS = VMK_TIMEOUT_NONBLOCKING;

	vmk_stat = vmk_HeapCreate(&heap_props, &heap_id);
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("qfle3i: vmk_HeapCreate failed: %s",
				vmk_StatusToString(vmk_stat));
		goto heap_create_fail;
	}
	/* Creating driver heap */
	qfle3i_driver_info.heap_id = heap_id;
	vmk_ModuleSetHeapID(vmk_ModuleCurrentID, heap_id);

	/* Creating driver dma heap */
	heap_props.type = VMK_HEAP_TYPE_CUSTOM;
	vmk_NameFormat(&heap_name, "%s-dma", QFLE3I_DRIVER_NAME);
	vmk_stat = vmk_NameInitialize(&heap_props.name, (const char*)&heap_name);
	heap_props.module = vmk_ModuleCurrentID;
	heap_props.typeSpecific.custom.physContiguity = VMK_MEM_PHYS_CONTIGUOUS;
	heap_props.typeSpecific.custom.physRange = VMK_PHYS_ADDR_ANY;
	heap_props.initial = QFLE3I_HEAP_INITIAL_SIZE;
	heap_props.max = QFLE3I_HEAP_MAXIMUM_SIZE;
	heap_props.creationTimeoutMS = VMK_TIMEOUT_NONBLOCKING;

	vmk_stat = vmk_HeapCreate(&heap_props, &heap_id);
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("qfle3i: vmk_HeapCreate of dma failed: %s",
				vmk_StatusToString(vmk_stat));
		goto heap_create_dma_fail;
	}
	qfle3i_driver_info.heap_id_dma = heap_id;


	/* Derive version string. */
	if (qfle3i_debug_level)
		vmk_Sprintf(qfle3i_version_str, "%s-debug", QFLE3I_DRV_MODULE_VERSION);
	else
		vmk_Sprintf(qfle3i_version_str, "%s", QFLE3I_DRV_MODULE_VERSION);
	if (qfle3i_debug_level == 1)
		qfle3i_debug_level = DEF_DBG_LEVEL;

	/* key-value pair interface */
	vmk_NameInitialize(&kv_name, "QFLE3I");
	vmk_NameInitialize(&kvSig.name, vmk_NameToString(&kv_name));

	mgmtProps.modId = vmk_ModuleCurrentID;
	mgmtProps.heapId = qfle3i_driver_info.heap_id;
	mgmtProps.sig = &kvSig;
	mgmtProps.cleanupFn = NULL;
	mgmtProps.sessionAnnounceFn = NULL;
	mgmtProps.sessionCleanupFn = NULL;
	mgmtProps.handleCookie = 0;
	vmk_stat = vmk_MgmtInit(&mgmtProps, &qfle3i_driver_info.kv_mgmt_handle);
        if (vmk_stat != VMK_OK) {
                DRV_WARN_MODLOAD_FAIL("qfle3i: vmk_Mgmtinit failed\n", vmk_stat);
                goto mgmt_init_fail;
        }

	vmk_NameInitialize(&vmhba_name, "DRIVERINFO");
	vmk_stat = vmk_MgmtAddKey(qfle3i_driver_info.kv_mgmt_handle,
				VMK_MGMT_KEY_TYPE_STRING,
				&vmhba_name,
				qfle3i_get_driver_info,
				qfle3i_set_driver_info);
	if (vmk_stat != VMK_OK) {
		DRV_WARN_MODLOAD_FAIL("qfle3i: vmk_MgmtAddKey failed\n", vmk_stat);
		goto mgmt_addkey_fail;
	}
	/*
	* Register driver log
	*/
	vmk_stat = vmk_NameInitialize(&log_props.name, QFLE3I_DRIVER_NAME);
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("qfle3i: vmk_NameInitialize failed for vmk_LogRegister (%s)\n",
					vmk_StatusToString(vmk_stat));
		goto log_create_fail;
	}
	log_props.module = vmk_ModuleCurrentID;
	log_props.heap = heap_id;
	log_props.defaultLevel = 0;
	log_props.throttle = NULL;
	vmk_stat = vmk_LogRegister(&log_props, &qfle3i_vmkLog);
	if (vmk_stat != VMK_OK) {
		DRV_WARN_MODLOAD_FAIL("qfle3i: vmk_LogRegister failed\n", vmk_stat);
		goto log_create_fail;
	}
	vmk_LogSetCurrentLogLevel(qfle3i_vmkLog, QFLE3I_DBG_DEBUG);

	/* Create timer queue */
	vmk_NameInitialize(&timerqueue_props.name, "qfle3i_timerq");
	timerqueue_props.moduleID = vmk_ModuleCurrentID;
	timerqueue_props.heapID = vmk_ModuleGetHeapID(vmk_ModuleCurrentID);
	timerqueue_props.attribs = VMK_TIMER_QUEUE_ATTR_NONE;

	vmk_stat = vmk_TimerQueueCreate(&timerqueue_props,
			&qfle3i_driver_info.timer_queue);
	if (vmk_stat != VMK_OK) {
		vmk_WarningMessage("qfle3i: vmk_TimerQueueCreate Failed: %s",
			vmk_StatusToString(vmk_stat));
		goto timerqueue_failed;
	}

	/*
	 * Create lock domain
	 */
	vmk_stat = vmk_LockDomainCreate(vmk_ModuleCurrentID, heap_id,
				&qfle3i_driver_info.driver_name,
				&qfle3i_driver_info.lock_domain);
	if (vmk_stat != VMK_OK) {
		DRV_WARN_MODLOAD_FAIL("qfle3i: vmk_LockDomainCreate failed", vmk_stat);
		goto lock_dom_fail;
	}

	/*
	* Create a global driver lock to access global data
	*/
	lock_props.moduleID = vmk_ModuleCurrentID;
	lock_props.heapID   = heap_id;
	lock_props.type     = VMK_SPINLOCK;
	lock_props.domain   = qfle3i_driver_info.lock_domain;
	lock_props.rank     = QFLE3I_HIGH_LOCK_RANK;
	vmk_NameInitialize(&lock_props.name, "qfle3i_drv_lock");

	vmk_stat = vmk_SpinlockCreate(&lock_props, &qfle3i_driver_info.drv_lock);
	if (vmk_stat != VMK_OK) {
		DRV_WARN_MODLOAD_FAIL("qfle3i: Failed to create global lock for qfle3i driver", vmk_stat);
		goto spin_lock_fail;
	}

	vmk_stat = vmk_BinarySemaCreate(&qfle3i_dev_lock, heap_id, "device_lock");
	if (vmk_stat != VMK_OK) {
		DRV_WARN_MODLOAD_FAIL("qfle3i: Failed to create BinarySemaPhore", vmk_stat);
		goto semaphore_fail;
	}

	qfle3i_driver_info.delayed_wq =
			ql_vmk_create_singlethread_workqueue("qfle3i_delayed_wq",
			VMK_WORLD_SCHED_CLASS_QUICK);
	if (!qfle3i_driver_info.delayed_wq) {
		vmk_WarningMessage("qfle3i: create_singlethread_workqueue Failed:");
		goto workq_create_fail;
	}

	/* Check parameter range */
	qfle3i_param_check_range();

	/* Initialize global params */
	vmk_ListInit(&adapter_list);
	adapter_count = 0;

	qfle3i_host_instance = vmk_BitVectorAlloc(qfle3i_driver_info.heap_id,
						(sizeof(long) * 32 * VMK_BITS_PER_BYTE));
	if (qfle3i_host_instance == NULL) {
		vmk_AlertMessage("qfle3i: unable to alloc vmk_BitVectorAlloc\n");
		goto alloc_vector_fail;
	}

	vmk_LogMessage("qfle3i: Allocating ep pool memory \n");
	/* create ep pool */
	vmk_stat = qfle3i_alloc_ep_pool();
	if (vmk_stat) {
		vmk_AlertMessage("qfle3i: unable to alloc ep pool\n");
		goto ep_alloc_failed;
	}

#if (VMWARE_ESX_DDK_VERSION >= 65000)
    mgmtProps.modId = vmk_ModuleCurrentID;
    mgmtProps.heapId = qfle3i_driver_info.heap_id;
    mgmtProps.sig = &ql_iscsi_sig;
    mgmtProps.cleanupFn = NULL;
    mgmtProps.sessionAnnounceFn = NULL;
    mgmtProps.sessionCleanupFn = NULL;
    mgmtProps.handleCookie = 0;
    vmk_stat = vmk_MgmtInit(&mgmtProps, &qed_iscsi_vmkmgmt_api_handle);
#else
    vmk_stat = vmk_MgmtInit(module_id,
            qfle3i_driver_info.heap_id,
            &ql_iscsi_sig,
            NULL,
            0,
            &qed_iscsi_vmkmgmt_api_handle);
#endif
    if (vmk_stat != VMK_OK) {
        vmk_WarningMessage("qfle3i: "
                "Unable to init common vmkmgmt-api. status=%s\n",
                vmk_StatusToString(vmk_stat));
		goto common_vmkmgmt_fail;
    }


	vmk_Memset(&qfle3i_drv_props, 0, sizeof(qfle3i_drv_props));
	vmk_NameCopy(&qfle3i_drv_props.name, &(qfle3i_driver_info.driver_name));
	qfle3i_drv_props.moduleID = vmk_ModuleCurrentID;
	qfle3i_drv_props.ops = &qfle3i_iSCSIOps;

	vmk_LogMessage("qfle3i: Registering qfle3i_driver Driver.\n");
	/* Register PCI Callback function */
	vmk_stat = vmk_DriverRegister(&qfle3i_drv_props,
			&qfle3i_driver_info.qfle3i_driver);
	if (vmk_stat != VMK_OK) {
		DRV_WARN_MODLOAD_FAIL("qfle3i: Unable to register PCI callback function.", vmk_stat);
		goto pci_callback_register_fail;
	}

	vmk_LogMessage("QFLE3I_DRV_MODULE_VERSION: %s\n", QFLE3I_DRV_MODULE_VERSION);

	return VMK_OK;

pci_callback_register_fail:
	vmk_MgmtDestroy(qed_iscsi_vmkmgmt_api_handle);
common_vmkmgmt_fail:
	qfle3i_release_ep_pool();
ep_alloc_failed:
	vmk_BitVectorFree(qfle3i_driver_info.heap_id, qfle3i_host_instance);
alloc_vector_fail:
	ql_vmk_destroy_singlethread_workqueue(qfle3i_driver_info.delayed_wq);
workq_create_fail:
	vmk_SemaDestroy(&qfle3i_dev_lock);
semaphore_fail:
	vmk_SpinlockDestroy(qfle3i_driver_info.drv_lock);
spin_lock_fail:
	vmk_LockDomainDestroy(qfle3i_driver_info.lock_domain);
lock_dom_fail:
	vmk_TimerQueueDestroy(qfle3i_driver_info.timer_queue);
timerqueue_failed:
	vmk_LogUnregister(qfle3i_vmkLog);
log_create_fail:
mgmt_addkey_fail:
	vmk_MgmtDestroy(qfle3i_driver_info.kv_mgmt_handle);
mgmt_init_fail:
	vmk_HeapDestroy(qfle3i_driver_info.heap_id_dma);
heap_create_dma_fail:
	vmk_HeapDestroy(qfle3i_driver_info.heap_id);
heap_create_fail:
#if (VMWARE_ESX_DDK_VERSION <= 65000)
	vmk_ModuleUnregister(vmk_ModuleCurrentID);
module_register_fail:
#endif
	return VMK_FAILURE;
}

void cleanup_module(void)
{
	/* Unregister the driver */
	DRV_DEBUG(0, "Entered");
	vmk_DriverUnregister(qfle3i_driver_info.qfle3i_driver);
	vmk_MgmtDestroy(qed_iscsi_vmkmgmt_api_handle);
	qfle3i_release_ep_pool();
	vmk_BitVectorFree(qfle3i_driver_info.heap_id, qfle3i_host_instance);
	ql_vmk_destroy_singlethread_workqueue(qfle3i_driver_info.delayed_wq);
	vmk_SemaDestroy(&qfle3i_dev_lock);
	vmk_SpinlockDestroy(qfle3i_driver_info.drv_lock);
	vmk_LockDomainDestroy(qfle3i_driver_info.lock_domain);
	vmk_TimerQueueDestroy(qfle3i_driver_info.timer_queue);
	vmk_LogUnregister(qfle3i_vmkLog);
	vmk_MgmtDestroy(qfle3i_driver_info.kv_mgmt_handle);
	vmk_HeapDestroy(qfle3i_driver_info.heap_id_dma);
	vmk_HeapDestroy(qfle3i_driver_info.heap_id);
#if (VMWARE_ESX_DDK_VERSION <= 65000)
	vmk_ModuleUnregister(qfle3i_driver_info.module_id);
#endif
	return;
}

