/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.sql.tool;

import com.google.appengine.repackaged.com.google.common.base.Preconditions;
import com.google.appengine.repackaged.com.google.common.base.StringUtil;
import com.google.appengine.repackaged.com.google.common.flags.DocLevel;
import com.google.appengine.repackaged.com.google.common.flags.Flag;
import com.google.appengine.repackaged.com.google.common.flags.FlagSpec;
import com.google.cloud.sql.jdbc.internal.Util;
import com.google.cloud.sql.tool.Context;
import com.google.cloud.sql.tool.KeepAlive;
import com.google.cloud.sql.tool.Log;
import com.google.cloud.sql.tool.Results;
import com.google.cloud.sql.tool.printers.PrinterFactory;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.concurrent.GuardedBy;

public class SqlProcessor {
    @FlagSpec(altName="keep_alive_minutes", docLevel=DocLevel.SECRET, help="Number of minutes to try to keep the connection alive.")
    private static final Flag<Integer> keepAliveMinutes = Flag.value(30);
    @FlagSpec(altName="keep_alive_interval_seconds", docLevel=DocLevel.SECRET, help="Number of seconds between queries to keep the connection alive.")
    private static final Flag<Integer> keepAliveIntervalSeconds = Flag.value(30);
    @FlagSpec(altName="statement_fetch_size", docLevel=DocLevel.SECRET, help="Fetch size to set on the JDBC Statement.")
    private static final Flag<Integer> statementFetchSize = Flag.value(1000);
    private static final Logger logger = Logger.getLogger(SqlProcessor.class.getName());
    private static final String EXECUTE_NORMAL_COMMAND_DELIMITER = "\\g";
    private static final String EXECUTE_VERTICAL_COMMAND_DELIMITER = "\\G";
    private final KeepAlive keepAlive;
    @GuardedBy(value="connection")
    private final Connection connection;
    private final Context context;
    private final boolean hasTerminal;

    SqlProcessor(Connection connection, Context context, boolean hasTerminal) {
        this(connection, context, new KeepAlive(Executors.newSingleThreadScheduledExecutor(), connection, keepAliveMinutes.get(), keepAliveIntervalSeconds.get()), hasTerminal);
    }

    SqlProcessor(Connection connection, Context context, KeepAlive keepAlive, boolean hasTerminal) {
        this.connection = Preconditions.checkNotNull(connection);
        this.context = Preconditions.checkNotNull(context);
        this.hasTerminal = hasTerminal;
        this.keepAlive = Preconditions.checkNotNull(keepAlive);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void close() {
        try {
            Connection connection = this.connection;
            synchronized (connection) {
                this.connection.commit();
                this.connection.close();
            }
        }
        catch (SQLException e) {
            logger.log(Level.FINE, "Exception closing connection", e);
        }
    }

    void initialize() {
        this.keepAlive.enable();
    }

    void processSql(String line) throws SQLException {
        Log.vlog("processSql: " + line, new Object[0]);
        String delimiter = this.context.getDelimiter();
        if (line.endsWith(delimiter)) {
            this.context.addStatementLine(StringUtil.stripSuffix(line, delimiter));
            this.execute(PrintMode.TABULAR);
        } else if (line.endsWith(EXECUTE_NORMAL_COMMAND_DELIMITER)) {
            this.context.addStatementLine(StringUtil.stripSuffix(line, EXECUTE_NORMAL_COMMAND_DELIMITER));
            this.execute(PrintMode.TABULAR);
        } else if (line.endsWith(EXECUTE_VERTICAL_COMMAND_DELIMITER)) {
            this.context.addStatementLine(StringUtil.stripSuffix(line, EXECUTE_VERTICAL_COMMAND_DELIMITER));
            this.execute(PrintMode.VERTICAL);
        } else {
            this.context.addStatementLine(line);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void execute(PrintMode executionMode) throws SQLException {
        Log.vlog("execute", new Object[0]);
        try {
            this.keepAlive.disable();
            this.executeStatement(this.context.getCurrentStatementSql(), executionMode);
        }
        finally {
            this.keepAlive.enable();
            this.context.clearCurrentStatement();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void executeStatement(String sql, PrintMode executionMode) throws SQLException {
        Log.vlog("executeStatement: " + sql, new Object[0]);
        Connection connection = this.connection;
        synchronized (connection) {
            Statement stmt = this.connection.createStatement(1003, 1007);
            try {
                stmt.setFetchSize(statementFetchSize.get());
                long startTime = System.nanoTime();
                boolean hasResultSet = stmt.execute(sql);
                float elapsedSeconds = (float)(System.nanoTime() - startTime) / 1.0E9f;
                if (hasResultSet) {
                    this.processResultSet(stmt.getResultSet(), elapsedSeconds, executionMode);
                } else if (this.hasTerminal) {
                    List<String> warnings = this.getWarningMessages(stmt.getWarnings());
                    this.displayUpdateCount(stmt.getUpdateCount(), warnings);
                    if (this.context.isShowWarningsEnabled() && !warnings.isEmpty()) {
                        this.showWarnings(warnings);
                    }
                }
            }
            finally {
                this.context.getWriter().flush();
                SqlProcessor.close(stmt);
            }
        }
    }

    private void displayUpdateCount(int updateCount, List<String> warnings) {
        StringBuilder line = new StringBuilder().append(updateCount).append(" row(s) affected");
        if (!warnings.isEmpty()) {
            line.append(", ").append(warnings.size()).append(" warning(s)");
        }
        line.append(".");
        this.context.getWriter().println(line.toString());
    }

    private List<String> getWarningMessages(SQLWarning warnings) {
        ArrayList<String> messages = Util.newArrayList();
        while (warnings != null) {
            messages.add(warnings.getLocalizedMessage());
            warnings = warnings.getNextWarning();
        }
        return messages;
    }

    private void showWarnings(List<String> messages) {
        PrintWriter out = this.context.getWriter();
        if (messages.isEmpty()) {
            return;
        }
        out.println();
        for (String message : messages) {
            out.println("Warning: " + message);
        }
    }

    private void processResultSet(ResultSet rs, float elapsedSeconds, PrintMode executionMode) throws SQLException {
        Results.createResults(rs, elapsedSeconds, statementFetchSize.get(), new PrinterFactory(executionMode, this.hasTerminal, this.context.getWriter())).process();
    }

    private static void close(Statement statement) {
        if (statement == null) {
            return;
        }
        try {
            statement.close();
        }
        catch (SQLException e) {
            logger.log(Level.WARNING, "Exception closing statement", e);
        }
    }

    public static enum PrintMode {
        TABULAR,
        VERTICAL;

    }
}

