/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2003 Andy Jefferson - coding standards
2003 Andy Jefferson - addition of equalsMethod()
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.List;

import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.util.RegularExpressionConverter;

/**
 * Representation of an expression with a string.
 **/
public class StringExpression extends ScalarExpression
{   
    protected StringExpression(QueryExpression qs)
    {
        super(qs);
    }

    /**
     * 
     * @param qs the QueryExpression
     * @param mapping the mapping associated to this expression
     * @param te the TableExpression where this expression refers to
     */
    public StringExpression(QueryExpression qs, JavaTypeMapping mapping, LogicSetExpression te)
    {
        super(qs, mapping, te);
    }

    /**
     * Generates statement as e.g. FUNCTION_NAME(arg[,argN])
     * @param functionName
     * @param args ScalarExpression list
     */    
    public StringExpression(String functionName, List args)
    {
        super(functionName, args);
    }
    
    /**
     * Generates statement as e.g. FUNCTION_NAME(arg AS type[,argN as typeN])
     * @param functionName Name of function
     * @param args ScalarExpression list
     * @param types String or ScalarExpression list
     */
    public StringExpression(String functionName, List args, List types)
    {
        super(functionName, args, types);
    }    

    /**
     * Performs a function on two arguments.
     * op(operand1,operand2)
     * operand1 op operand2 
     * @param operand1 the first expression
     * @param op the operator between operands
     * @param operand2 the second expression
     */    
    public StringExpression(ScalarExpression operand1, DyadicOperator op, ScalarExpression operand2)
    {
        super(operand1, op, operand2);
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        // Allow for parameter comparison
        assertValidTypeForParameterComparison(expr, StringExpression.class);
        expr = getConsistentTypeForParameterComparison(expr);

        if (expr instanceof NullLiteral)
        {
            return expr.eq(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_EQ, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr);   
        }
        else if (expr instanceof ByteLiteral)
        {
            int value = ((BigInteger)((ByteLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));    
            return new BooleanExpression(this, OP_EQ, literal);
        }                   
        else if (expr instanceof IntegerLiteral)
        {
            int value = ((Number)((IntegerLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));                
            return new BooleanExpression(this, OP_EQ, literal);
        }
        else if (expr instanceof FloatingPointLiteral)
        {
            int value = ((BigDecimal)((FloatingPointLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_EQ, literal);
        }
        else if (expr instanceof NumericExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr);
        }
        else if (expr instanceof SqlTemporalExpression)
        {
            return new BooleanExpression(this, OP_EQ, expr);
        }        
        else
        {
            return super.eq(expr);
        }
    }
    
    public BooleanExpression noteq(ScalarExpression expr)
    {
        // Allow for parameter comparison
        assertValidTypeForParameterComparison(expr, StringExpression.class);
        expr = getConsistentTypeForParameterComparison(expr);

        if (expr instanceof NullLiteral)
        {
            return expr.noteq(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);   
        }
        else if (expr instanceof ByteLiteral)
        {
            int value = ((BigInteger)((ByteLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_NOTEQ, literal);
        }                   
        else if (expr instanceof IntegerLiteral)
        {
            int value = ((Number)((IntegerLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_NOTEQ, literal);
        }
        else if (expr instanceof FloatingPointLiteral)
        {
            int value = ((BigDecimal)((FloatingPointLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_NOTEQ, literal);
        }
        else if (expr instanceof NumericExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);
        }
        else if (expr instanceof SqlTemporalExpression)
        {
            return new BooleanExpression(this, OP_NOTEQ, expr);
        }        
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.lt(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_LT, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_LT, expr);   
        }
        else if (expr instanceof ByteLiteral)
        {
            int value = ((BigInteger)((ByteLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_LT, literal);
        }                   
        else if (expr instanceof IntegerLiteral)
        {
            int value = ((Number)((IntegerLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_LT, literal);
        }
        else if (expr instanceof FloatingPointLiteral)
        {
            int value = ((BigDecimal)((FloatingPointLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_LT, literal);
        }
        else if (expr instanceof NumericExpression)
        {
            return new BooleanExpression(this, OP_LT, expr);
        }
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.lteq(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);   
        }
        else if (expr instanceof ByteLiteral)
        {
            int value = ((BigInteger)((ByteLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_LTEQ, literal);
        }                   
        else if (expr instanceof IntegerLiteral)
        {
            int value = ((Number)((IntegerLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_LTEQ, literal);
        }
        else if (expr instanceof FloatingPointLiteral)
        {
            int value = ((BigDecimal)((FloatingPointLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_LTEQ, literal);
        }
        else if (expr instanceof NumericExpression)
        {
            return new BooleanExpression(this, OP_LTEQ, expr);
        }
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.gt(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_GT, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_GT, expr);   
        }
        else if (expr instanceof ByteLiteral)
        {
            int value = ((BigInteger)((ByteLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_GT, literal);
        }                   
        else if (expr instanceof IntegerLiteral)
        {
            int value = ((Number)((IntegerLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_GT, literal);
        }
        else if (expr instanceof FloatingPointLiteral)
        {
            int value = ((BigDecimal)((FloatingPointLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_GT, literal);
        }
        else if (expr instanceof NumericExpression)
        {
            return new BooleanExpression(this, OP_GT, expr);
        }
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        if (expr instanceof NullLiteral)
        {
            return expr.gteq(this);
        }
        else if (expr instanceof CharacterLiteral)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);            
        }
        else if (expr instanceof StringExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);   
        }
        else if (expr instanceof ByteLiteral)
        {
            int value = ((BigInteger)((ByteLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_GTEQ, literal);
        }                   
        else if (expr instanceof IntegerLiteral)
        {
            int value = ((Number)((IntegerLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_GTEQ, literal);
        }
        else if (expr instanceof FloatingPointLiteral)
        {
            int value = ((BigDecimal)((FloatingPointLiteral)expr).getValue()).intValue(); 
            CharacterLiteral literal = new CharacterLiteral(qs, mapping, String.valueOf((char)value));
            return new BooleanExpression(this, OP_GTEQ, literal);
        }
        else if (expr instanceof NumericExpression)
        {
            return new BooleanExpression(this, OP_GTEQ, expr);
        }
        else
        {
            return super.gteq(expr);
        }
    }

    public ScalarExpression add(ScalarExpression expr)
    {
        if (expr instanceof StringLiteral)
        {
            return new StringExpression(this, OP_CONCAT, new StringLiteral(qs,expr.mapping,(String) ((StringLiteral)expr).getValue()));
        }        
        else if (expr instanceof StringExpression)
        {
            return new StringExpression(this, OP_CONCAT, expr);
        }
        else if (expr instanceof CharacterExpression)
        {
            return new StringExpression(this, OP_CONCAT, expr);
        }
        else if (expr instanceof NumericExpression )
        {
            return new StringExpression(this, OP_CONCAT, qs.getStoreManager().getDatastoreAdapter().toStringExpression((NumericExpression) expr));
        }      
        else if (expr instanceof NullLiteral)
        {
            return expr;
        }
        else
        {
            return new StringExpression(this, OP_CONCAT, expr);
        }
    }
    
    public BooleanExpression in(ScalarExpression expr)
    {
        return new BooleanExpression(this, OP_IN, expr);
    }

    /**
     * Returns whether this string is equal to the expression. This is provided
     * as an operation on String.
     * @param expr The expression to compare against.
     * @return Expression whether they are equal.
     **/
    public BooleanExpression equalsMethod(ScalarExpression expr)
    {
        return eq(expr);
    }

    /**
     * Returns the character at a position.
     * @param index The position of the character required.
     * @return The expression.
     **/
    public StringExpression charAtMethod(ScalarExpression index)
    {
        if (!(index instanceof NumericExpression))
        {
            throw new IllegalArgumentTypeException(index);
        }

        NumericExpression begin = (NumericExpression)index;
        NumericExpression end   = (NumericExpression)begin.add(new IntegerLiteral(qs, mapping, BigInteger.ONE));
        return substringMethod(begin, end);
    }

    /**
     * Returns whether this string ends with the specified string.
     * @param str The string to compare against.
     * @return Whether it ends with the string.
     **/
    public BooleanExpression endsWithMethod(ScalarExpression str)
    {
        if (!(str instanceof StringExpression))
        {
            throw new IllegalArgumentTypeException(str);
        }

        return qs.getStoreManager().getDatastoreAdapter().endsWithMethod(this, str);
    }

    /**
     * Returns the index within this string of the first occurrence of the
     * specified string.
     * @param str The string to find the index of
     * @return The index of the string.
     **/
    public NumericExpression indexOfMethod(ScalarExpression str)
    {
        if (!(str instanceof StringExpression) && !(str instanceof CharacterLiteral))
        {
            throw new IllegalArgumentTypeException(str);
        }

        return qs.getStoreManager().getDatastoreAdapter().indexOfMethod(this, str, null);
    }

    /**
     * Returns the index within this string of the first occurrence of the
     * specified character, starting the search at the specified index. 
     * @param str The string to find the index of
     * @param fromIndex the index to start the search from. 
     * @return The index of the string.
     **/
    public NumericExpression indexOfMethod(ScalarExpression str, ScalarExpression fromIndex)
    {
        if (!(str instanceof StringExpression) && !(str instanceof CharacterLiteral))
        {
            throw new IllegalArgumentTypeException(str);
        }
        if (!(fromIndex instanceof NumericExpression))
        {
            throw new IllegalArgumentTypeException(fromIndex);
        }

        return qs.getStoreManager().getDatastoreAdapter().indexOfMethod(this, str, (NumericExpression)fromIndex);
    }

    /**
     * Method to handle the starts with operation.
     * @return The expression.
     **/
    public NumericExpression lengthMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().getNumericExpressionForMethod("length", this);
    }

    /**
     * Method to handle the starts with operation.
     * @param str The string to compare against 
     * @return The expression.
     **/
    public BooleanExpression startsWithMethod(ScalarExpression str)
    {
        if (!(str instanceof StringExpression))
        {
            throw new IllegalArgumentTypeException(str);
        }

        return qs.getStoreManager().getDatastoreAdapter().startsWithMethod(this, str);
    }

    /**
     * Method to handle the starts with operation.
     * @param str The string to compare against 
     * @param toffset The offset
     * @return The expression.
     **/
    public BooleanExpression startsWithMethod(ScalarExpression str, ScalarExpression toffset)
    {
        return substringMethod(toffset).startsWithMethod(str);
    }

    /**
     * Method to handle the substring operation.
     * @param begin The start position
     * @return The expression.
     **/
    public StringExpression substringMethod(ScalarExpression begin)
    {
        if (!(begin instanceof NumericExpression))
        {
            throw new IllegalArgumentTypeException(begin);
        }

        return qs.getStoreManager().getDatastoreAdapter().substringMethod(this,
                                                                         (NumericExpression)begin);
    }

    /**
     * Method to handle the substring operation.
     * @param begin The start position
     * @param end The end position
     * @return The expression.
     **/
    public StringExpression substringMethod(ScalarExpression begin, ScalarExpression end)
    {
        if (!(begin instanceof NumericExpression))
        {
            throw new IllegalArgumentTypeException(begin);
        }
        if (!(end   instanceof NumericExpression))
        {
            throw new IllegalArgumentTypeException(end);
        }

        return qs.getStoreManager().getDatastoreAdapter().substringMethod(this,
                                                                         (NumericExpression)begin,
                                                                         (NumericExpression)end);
    }

    /**
     * Method to translate all chars in this expression to the <code>fromExpr</code> which 
     * corresponds to <code>toExpr</code>.
     * @return The expression.
     **/
    public StringExpression translateMethod(ScalarExpression toExpr, ScalarExpression fromExpr)
    {
        return qs.getStoreManager().getDatastoreAdapter().translateMethod(this,toExpr,fromExpr);
    }

    /**
     * Method to handle the lower case operation.
     * @return The expression.
     **/
    public StringExpression toLowerCaseMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().lowerMethod(this);
    }

    /**
     * Method to handle the upper case operation.
     * @return The expression.
     **/
    public StringExpression toUpperCaseMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().upperMethod(this);
    }

    /**
     * Matches this to the argument expression pattern. Use "." to find any
     * character and ".*" for wildcard matches. A global case-insensitive flag
     * "(?i)" can be set for the pattern. If used, the global case-insensitive
     * flag must prefix the pattern. The pattern passed to matches must be a
     * literal or parameter.
     * @param expr The literal expression with the pattern.
     * @return the match expression.
     */
    public BooleanExpression matchesMethod(ScalarExpression expr)
    {
        if (expr instanceof StringLiteral)
        {
            //expr = qs.getStoreManager().getDatastoreAdapter().getEscapedPatternExpression(expr);
            String pattern = (String) ((Literal) expr).getValue();
            boolean caseSensitive = false;
            if (pattern.startsWith("(?i)"))
            {
                caseSensitive = true;
                pattern = pattern.substring(4);
            }

            DatastoreAdapter dba = qs.getStoreManager().getDatastoreAdapter();
            RegularExpressionConverter converter = new RegularExpressionConverter(
                dba.getPatternExpressionZeroMoreCharacters().charAt(0),
                dba.getPatternExpressionAnyCharacter().charAt(0),
                dba.getEscapeCharacter().charAt(0));
            if (caseSensitive)
            {
                StringLiteral patternExpr = new StringLiteral(qs, mapping, converter.convert(pattern).toLowerCase());
                //StringExpression escapedExpr = (StringExpression) qs.getStoreManager().getDatastoreAdapter().getEscapedPatternExpression(patternExpr);
                return new LikeExpression(this.toLowerCaseMethod(), ScalarExpression.OP_LIKE, patternExpr);
            }
            else
            {
                StringLiteral patternExpr = new StringLiteral(qs, mapping, converter.convert(pattern));
                //StringExpression escapedExpr = (StringExpression) qs.getStoreManager().getDatastoreAdapter().getEscapedPatternExpression(patternExpr);
                return new LikeExpression(this, ScalarExpression.OP_LIKE, patternExpr);
            }
        }
        else if (expr instanceof StringExpression)
        {
            return qs.getStoreManager().getDatastoreAdapter().matchesMethod(this, (StringExpression)expr);
        }
        throw new IllegalOperationException(this, "matches", expr);
    }

    /**
     * Perform a database "like" operation
     * @param expr The expression with the pattern.
     * @return the match expression.
     */
    public BooleanExpression likeMethod(ScalarExpression expr)
    {
        return new LikeExpression(this, OP_LIKE, expr);
    }

    /**
     * Method to handle trimming of both ends of a string.
     * @return The expression.
     */
    public StringExpression trimMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().trimMethod(this, true, true);
    }

    /**
     * Method to handle trimming of the left side of a string (leading spaces).
     * @return The expression.
     */
    public StringExpression trimLeftMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().trimMethod(this, true, false);
    }

    /**
     * Method to handle trimming of the right side of a string (trailing spaces).
     * @return The expression.
     */
    public StringExpression trimRightMethod()
    {
        return qs.getStoreManager().getDatastoreAdapter().trimMethod(this, false, true);
    }
}