/**********************************************************************
Copyright (c) 2007 Thomas Marti and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.awt.geom.Line2D;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreContainerObject;

/**
 * Mapping for java.awt.geom.Line2D.Float, maps the x1, y1, x2 and y2 values to float-precision 
 * datastore fields.
 */
public class Line2dFloatMapping extends SingleFieldMultiMapping
{

    private static final Line2D.Float sampleValue = new Line2D.Float(0, 0, 1, 1);

	/* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#initialize()
     */
    public void initialize(DatastoreAdapter dba, AbstractMemberMetaData fmd, DatastoreContainerObject container, ClassLoaderResolver clr)
    {
		super.initialize(dba, fmd, container, clr);

        addDatastoreField(ClassNameConstants.FLOAT); // X1
        addDatastoreField(ClassNameConstants.FLOAT); // Y1
        addDatastoreField(ClassNameConstants.FLOAT); // X2
        addDatastoreField(ClassNameConstants.FLOAT); // Y2
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getJavaType()
     */
    public Class getJavaType()
    {
        return Line2D.Float.class;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getSampleValue()
     */
    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return sampleValue;
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#setObject(org.datanucleus.ObjectManager, java.lang.Object, int[], java.lang.Object)
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
    	Line2D.Float line = (Line2D.Float)value;
        if (line == null)
        {
    		for (int i = 0; i < exprIndex.length; i++) 
    		{
    			getDataStoreMapping(i).setObject(preparedStatement, exprIndex[i], null);					
			}
        }
        else
        {
            getDataStoreMapping(0).setFloat(preparedStatement,exprIndex[0],line.x1);
            getDataStoreMapping(1).setFloat(preparedStatement,exprIndex[1],line.y1);
            getDataStoreMapping(2).setFloat(preparedStatement,exprIndex[2],line.x2);
            getDataStoreMapping(3).setFloat(preparedStatement,exprIndex[3],line.y2);
        }
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getObject(org.datanucleus.ObjectManager, java.lang.Object, int[])
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        // Check for null entries
        if (getDataStoreMapping(0).getObject(resultSet, exprIndex[0]) == null)
        {
            return null;
        }

        float x1 = getDataStoreMapping(0).getFloat(resultSet,exprIndex[0]); 
        float y1 = getDataStoreMapping(1).getFloat(resultSet,exprIndex[1]); 
        float x2 = getDataStoreMapping(2).getFloat(resultSet,exprIndex[2]); 
        float y2 = getDataStoreMapping(3).getFloat(resultSet,exprIndex[3]);
        return new Line2D.Float(x1, y1, x2, y2);
    }
}