/**********************************************************************
Copyright (c) 2009 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.store.query.cache;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import org.datanucleus.query.compiler.QueryCompilation;

/**
 * Representation of a query that is cached.
 * Stores the generic compilation of the query, together with the compilation(s) specific
 * to datastore(s). For example for RDBMS this will mean that this contains the generic compilation
 * of expression trees, and the SQL that is generated from the expression trees.
 */
public class CachedQuery implements Serializable
{
    /** The generic query compilation for this query. */
    protected final QueryCompilation compilation;

    /** Map of datastore-specific compilations for the query. */
    protected Map<String, Object> datastoreCompilations = null;

    public CachedQuery(QueryCompilation compilation)
    {
        this.compilation = compilation;
    }

    /**
     * Accessor for the query generic compilation.
     * @return The generic compilation
     */
    public QueryCompilation getCompilation()
    {
        return compilation;
    }

    /**
     * Method to store a datastore-specific compilation.
     * Will overwrite any existing compilation for that datastore.
     * @param datastoreKey The datastore key (e.g rdbms, odf, etc)
     * @param datastoreCompilation The compilation for the datastore
     */
    public void addDatastoreCompilation(String datastoreKey, Object datastoreCompilation)
    {
        if (datastoreCompilations == null)
        {
            datastoreCompilations = new HashMap<String, Object>();
        }
        datastoreCompilations.put(datastoreKey, datastoreCompilation);
    }

    /**
     * Method to remove a datastore-specific compilation.
     * @param datastoreKey The datastore key (e.g rdbms, odf, etc)
     */
    public void removeDatastoreCompilation(String datastoreKey)
    {
        if (datastoreCompilations != null)
        {
            datastoreCompilations.remove(datastoreKey);
        }
    }

    /**
     * Accessor for the datastore compilation for the specified datastore.
     * @param datastoreKey The datastore key (e.g rdbms, odf, etc)
     * @return The datastore compilation
     */
    public Object getDatastoreCompilation(String datastoreKey)
    {
        if (datastoreCompilations == null)
        {
            return null;
        }
        return datastoreCompilations.get(datastoreKey);
    }
}