/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.enhancer.asm.method;

import org.datanucleus.enhancer.ClassEnhancer;
import org.datanucleus.enhancer.asm.ASMClassMethod;
import org.datanucleus.enhancer.asm.ASMUtils;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.objectweb.asm.Label;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;

/**
 * Method to generate the method "jdoGetZZZ" using ASM for MEDIATE_READ fields.
 * <pre>
 * static YYY jdoGetZZZ(MyClass objPC)
 * {
 *     if (objPC.jdoStateManager != null && !objPC.jdoStateManager.isLoaded(objPC, 0))
 *         return (YYY) objPC.jdoStateManager.getObjectField(objPC, 0, objPC.ZZZ);
 *     if (objPC.jdoIsDetached() != false &&
 *         ((BitSet) objPC.jdoDetachedState[2]).get(0) != true &&
 *         ((BitSet) objPC.jdoDetachedState[3]).get(0) != true)
 *         throw new JDODetachedFieldAccessException(...);
 *     return objPC.ZZZ;
 * }
 * </pre>
 * with the last part only applying when Detachable
 */
public class JdoGetViaMediate extends ASMClassMethod
{
    /** Field that this jdoGetXXX is for. */
    protected AbstractMemberMetaData fmd;

    /**
     * Constructor.
     * @param enhancer ClassEnhancer
     * @param fmd MetaData for the field we are generating for
     */
    public JdoGetViaMediate(ClassEnhancer enhancer, AbstractMemberMetaData fmd)
    {
        super(enhancer, enhancer.getGetMethodPrefixMethodName() + fmd.getName(),
            (fmd.isPublic() ? Opcodes.ACC_PUBLIC : 0) | (fmd.isProtected() ? Opcodes.ACC_PROTECTED : 0) | 
            (fmd.isPrivate() ? Opcodes.ACC_PRIVATE : 0) | Opcodes.ACC_STATIC, fmd.getType(), null, null);

        // Set the arg type/name
        argTypes = new Class[] {getClassEnhancer().getClassEnhanced()};
        argNames = new String[] {"objPC"};

        this.fmd = fmd;
    }

    /**
     * Method to add the contents of the class method.
     */
    public void execute()
    {
        visitor.visitCode();

        String fieldTypeDesc = Type.getDescriptor(fmd.getType());

        Label startLabel = new Label();
        visitor.visitLabel(startLabel);

        visitor.visitVarInsn(Opcodes.ALOAD, 0);
        visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(),
            enhancer.getStateManagerFieldName(), "L" + enhancer.getStateManagerAsmClassName() + ";");
        Label l1 = new Label();
        visitor.visitJumpInsn(Opcodes.IFNULL, l1);
        visitor.visitVarInsn(Opcodes.ALOAD, 0);
        visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(),
            enhancer.getStateManagerFieldName(), "L" + enhancer.getStateManagerAsmClassName() + ";");
        visitor.visitVarInsn(Opcodes.ALOAD, 0);
        ASMUtils.addBIPUSHToMethod(visitor, fmd.getFieldId());
        if (enhancer.getClassMetaData().getPersistenceCapableSuperclass() != null)
        {
            visitor.visitFieldInsn(Opcodes.GETSTATIC, getClassEnhancer().getASMClassName(), 
                enhancer.getInheritedFieldCountFieldName(), "I");
            visitor.visitInsn(Opcodes.IADD);
        }
        visitor.visitMethodInsn(Opcodes.INVOKEINTERFACE, enhancer.getStateManagerAsmClassName(),
            "isLoaded", "(L" + enhancer.getPersistableAsmClassName() + ";I)Z");
        visitor.visitJumpInsn(Opcodes.IFNE, l1);

        visitor.visitVarInsn(Opcodes.ALOAD, 0);
        visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(),
            enhancer.getStateManagerFieldName(), "L" + enhancer.getStateManagerAsmClassName() + ";");
        visitor.visitVarInsn(Opcodes.ALOAD, 0);
        ASMUtils.addBIPUSHToMethod(visitor, fmd.getFieldId());
        if (enhancer.getClassMetaData().getPersistenceCapableSuperclass() != null)
        {
            visitor.visitFieldInsn(Opcodes.GETSTATIC, getClassEnhancer().getASMClassName(), 
                enhancer.getInheritedFieldCountFieldName(), "I");
            visitor.visitInsn(Opcodes.IADD);
        }
        visitor.visitVarInsn(Opcodes.ALOAD, 0);
        visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(),
            fmd.getName(), fieldTypeDesc);
        String jdoMethodName = "get" + ASMUtils.getTypeNameForJDOMethod(fmd.getType()) + "Field";
        String argTypeDesc = fieldTypeDesc;
        if (jdoMethodName.equals("getObjectField"))
        {
            argTypeDesc = ASMUtils.CD_Object;
        }
        visitor.visitMethodInsn(Opcodes.INVOKEINTERFACE, enhancer.getStateManagerAsmClassName(),
            jdoMethodName, "(L" + enhancer.getPersistableAsmClassName() + ";I" + argTypeDesc + ")" + argTypeDesc);
        if (jdoMethodName.equals("getObjectField"))
        {
            // Cast any object fields to the correct type
            visitor.visitTypeInsn(Opcodes.CHECKCAST, fmd.getTypeName().replace('.', '/'));
        }
        ASMUtils.addReturnForType(visitor, fmd.getType());

        visitor.visitLabel(l1);

        Label l4 = new Label();
        if (enhancer.getClassMetaData().isDetachable())
        {
            // "if (objPC.jdoIsDetached() != false && ((BitSet) objPC.jdoDetachedState[2]).get(5) != true)"
            visitor.visitVarInsn(Opcodes.ALOAD, 0);
            visitor.visitMethodInsn(Opcodes.INVOKEVIRTUAL, getClassEnhancer().getASMClassName(), 
                enhancer.getIsDetachedMethodName(), "()Z");
            visitor.visitJumpInsn(Opcodes.IFEQ, l4);
            visitor.visitVarInsn(Opcodes.ALOAD, 0);
            visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(),
                enhancer.getDetachedStateFieldName(), "[Ljava/lang/Object;");
            visitor.visitInsn(Opcodes.ICONST_2);
            visitor.visitInsn(Opcodes.AALOAD);
            visitor.visitTypeInsn(Opcodes.CHECKCAST, "java/util/BitSet");
            ASMUtils.addBIPUSHToMethod(visitor, fmd.getFieldId());
            if (enhancer.getClassMetaData().getPersistenceCapableSuperclass() != null)
            {
                visitor.visitFieldInsn(Opcodes.GETSTATIC, getClassEnhancer().getASMClassName(), 
                    enhancer.getInheritedFieldCountFieldName(), "I");
                visitor.visitInsn(Opcodes.IADD);
            }
            visitor.visitMethodInsn(Opcodes.INVOKEVIRTUAL, "java/util/BitSet", "get", "(I)Z");
            visitor.visitJumpInsn(Opcodes.IFNE, l4);
            visitor.visitVarInsn(Opcodes.ALOAD, 0);
            visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(),
                enhancer.getDetachedStateFieldName(), "[Ljava/lang/Object;");
            visitor.visitInsn(Opcodes.ICONST_3);
            visitor.visitInsn(Opcodes.AALOAD);
            visitor.visitTypeInsn(Opcodes.CHECKCAST, "java/util/BitSet");
            ASMUtils.addBIPUSHToMethod(visitor, fmd.getFieldId());
            if (enhancer.getClassMetaData().getPersistenceCapableSuperclass() != null)
            {
                visitor.visitFieldInsn(Opcodes.GETSTATIC, getClassEnhancer().getASMClassName(), 
                    enhancer.getInheritedFieldCountFieldName(), "I");
                visitor.visitInsn(Opcodes.IADD);
            }
            visitor.visitMethodInsn(Opcodes.INVOKEVIRTUAL, "java/util/BitSet", "get", "(I)Z");
            visitor.visitJumpInsn(Opcodes.IFNE, l4);

            if (enhancer.hasOption(ClassEnhancer.OPTION_GENERATE_DETACH_LISTENER))
            {
                visitor.visitMethodInsn(Opcodes.INVOKESTATIC, "org/datanucleus/util/DetachListener", 
                    "getInstance", "()Lorg/datanucleus/util/DetachListener;");
                visitor.visitVarInsn(Opcodes.ALOAD, 0);
                visitor.visitLdcInsn(fmd.getName());
                visitor.visitMethodInsn(Opcodes.INVOKEVIRTUAL, "org/datanucleus/util/DetachListener",
                    "undetachedFieldAccess", "(Ljava/lang/Object;Ljava/lang/String;)V");
            }
            else
            {
                // "throw new DetachedFieldAccessException(...)"
                visitor.visitTypeInsn(Opcodes.NEW, enhancer.getDetachedFieldAccessExceptionAsmClassName());
                visitor.visitInsn(Opcodes.DUP);
                visitor.visitLdcInsn(LOCALISER.msg("Enhancer.DetachedFieldAccess", fmd.getName()));
                visitor.visitMethodInsn(Opcodes.INVOKESPECIAL, enhancer.getDetachedFieldAccessExceptionAsmClassName(),
                    "<init>", "(Ljava/lang/String;)V");
                visitor.visitInsn(Opcodes.ATHROW);
            }
        }

        visitor.visitLabel(l4);
        visitor.visitVarInsn(Opcodes.ALOAD, 0);
        visitor.visitFieldInsn(Opcodes.GETFIELD, getClassEnhancer().getASMClassName(), fmd.getName(), fieldTypeDesc);
        ASMUtils.addReturnForType(visitor, fmd.getType());

        Label endLabel = new Label();
        visitor.visitLabel(endLabel);
        visitor.visitLocalVariable(argNames[0], getClassEnhancer().getClassDescriptor(), null, startLabel, endLabel, 0);
        visitor.visitMaxs(4, 1);

        visitor.visitEnd();
    }
}