/**********************************************************************
Copyright (c) 2004 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManager;
import org.datanucleus.ObjectManagerFactoryImpl;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.FieldRole;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.util.Localiser;

/**
 * Representation of the mapping of a Java type.
 * The java type maps to one or more datastore mappings. This means that a field/property
 * in a java class can be mapped to many columns in a table (in an RDBMS).
 * A JavaTypeMapping can exist in 2 forms
 * <ul>
 * <li>Constructed for a field/property managed by a datastore container, and so has metadata/container 
 * information</li>
 * <li>Constructed for the java type on its own to represent a literal in a JDOQL query, and consequently 
 * has no metadata/container information</li>
 * </ul>
 */
public abstract class JavaTypeMapping
{
    /** Localiser for messages */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.Localisation", ObjectManagerFactoryImpl.class.getClassLoader());

    /** MetaData for the field/property that we are mapping. When this is for a query literal will be NULL. */
    protected AbstractMemberMetaData mmd;

    /**
     * Role of the mapping for the field. Whether it is for the field as a whole, or element of
     * a collection field (in a join table), or key/value of a map field (in a join table).
     */
    protected int roleForMember = FieldRole.ROLE_NONE;

    /** The Datastore mappings for this Java type. */
    protected DatastoreMapping[] datastoreMappings = new DatastoreMapping[0];

    /** The Datastore Container storing this mapping. */
    protected DatastoreContainerObject datastoreContainer;

    /** Adapter for the datastore being used. */
    protected DatastoreAdapter dba;

    /** Actual type being mapped */
    protected String type;

    /**
     * Mapping of the reference on the end of a bidirectional association.
     * Only used when this mapping doesn't have datastore fields, but the other side has.
     */
    protected JavaTypeMapping referenceMapping;

    /**
     * Create a new empty JavaTypeMapping.
     * The caller must call one of the initialize() methods to initialise the instance with the 
     * DatastoreAdapter and its type.
     * The combination of this empty constructor and one of the initialize() methods is used instead of 
     * parameterised constructors for efficiency purpose, both in execution time and code maintainability.
     * See MappingFactory for how they are used.
     * Concrete subclasses must have a publicly accessible empty constructor. 
     */
    protected JavaTypeMapping()
    {
    }

    /**
     * Create a new Mapping with the given DatastoreAdapter for the given type.
     * Used by some querying process. TODO Remove this
     * @param dba The Datastore Adapter that this Mapping should use.
     * @param type The Class that this mapping maps to the database.
     * @param fmd FieldMetaData for the field to be mapped (if any)
     * @param container The datastore container storing this mapping (if any)
     */
    protected JavaTypeMapping(DatastoreAdapter dba, String type, AbstractMemberMetaData fmd, 
            DatastoreContainerObject container)
    {
        this.dba = dba;
        this.type = type;
        this.mmd = fmd;
        this.datastoreContainer = container;
    }

    /**
     * Initialize this JavaTypeMapping with the given DatastoreAdapter for the given type.
     * This will not set the "fmd" and "datastoreContainer" parameters. If these are required for
     * usage of the mapping then you should call
     * "setFieldInformation(AbstractMemberMetaData, DatastoreContainerObject)" below
     * @param dba The Datastore Adapter that this Mapping should use.
     * @param type The Class that this mapping maps to the database.
     * @see MappingFactory#createMapping(Class, DatastoreAdapter, String)
     */
    public void initialize(DatastoreAdapter dba, String type)
    {
        this.dba = dba;
        this.type = type;
    }

    /**
     * Initialize this JavaTypeMapping with the given DatastoreAdapter for the given FieldMetaData.
     * Subclasses should override this method to perform any datastore initialization operations.
     * @param dba The Datastore Adapter that this Mapping should use.
     * @param fmd FieldMetaData for the field to be mapped (if any)
     * @param container The datastore container storing this mapping (if any)
     * @param clr the ClassLoaderResolver
     */
    public void initialize(DatastoreAdapter dba, AbstractMemberMetaData fmd, DatastoreContainerObject container,
            ClassLoaderResolver clr)
    {
        this.dba = dba;
        this.mmd = fmd;

        // Set the type based on what role this mapping has for the field as a whole
        if (roleForMember == FieldRole.ROLE_ARRAY_ELEMENT)
        {
            this.type = mmd.getArray().getElementType();
        }
        else if (roleForMember == FieldRole.ROLE_COLLECTION_ELEMENT)
        {
            this.type = mmd.getCollection().getElementType();
        }
        else if (roleForMember == FieldRole.ROLE_MAP_KEY)
        {
            this.type = mmd.getMap().getKeyType();
        }
        else if (roleForMember == FieldRole.ROLE_MAP_VALUE)
        {
            this.type = mmd.getMap().getValueType();
        }
        else
        {
            this.type = mmd.getType().getName();
        }

        this.datastoreContainer = container;
    }

    /**
     * Method to set the metadata of the member for which this mapping applies.
     * Ffor use where the mapping was created for a particular type (using the 
     * initialize(DatastoreAdapter, String) and we now have the member that it applies for.
     * @param mmd Field/Property MetaData
     */
    public void setMemberMetaData(AbstractMemberMetaData mmd)
    {
        this.mmd = mmd;
    }

    public void setDatastoreContainer(DatastoreContainerObject table)
    {
        this.datastoreContainer = table;
    }

    /**
     * Method to set the role for the field/property.
     * Should be called before initialize().
     * @param role Role for field/property.
     */
    public void setRoleForMember(int role)
    {
        roleForMember = role;
    }

    /**
     * Accessor for the MetaData of the field/property to be mapped.
     * Will be null if this mapping is for a literal in a query.
     * @return Returns the metadata for the field or property
     */
    public AbstractMemberMetaData getMemberMetaData()
    {
        return mmd;
    }

    /**
     * Accessor for the role of this mapping for the field/property.
     * @return Role of this mapping for the field/property
     */
    public int getRoleForMember()
    {
        return roleForMember;
    }

    /**
     * Convenience method to return if the (part of the) field being represented by this mapping is serialised.
     * @return Whether to use Java serialisation
     */
    public boolean isSerialised()
    {
        if (roleForMember == FieldRole.ROLE_COLLECTION_ELEMENT)
        {
            if (mmd == null)
            {
                return false;
            }
            return (mmd.getCollection() != null ? mmd.getCollection().isSerializedElement() : false);
        }
        else if (roleForMember == FieldRole.ROLE_ARRAY_ELEMENT)
        { 
            if (mmd == null)
            {
                return false;
            }
            return (mmd.getArray() != null ? mmd.getArray().isSerializedElement() : false);
        }
        else if (roleForMember == FieldRole.ROLE_MAP_KEY)
        {
            if (mmd == null)
            {
                return false;
            }
            return (mmd.getMap() != null ? mmd.getMap().isSerializedKey() : false);
        }
        else if (roleForMember == FieldRole.ROLE_MAP_VALUE)
        {
            if (mmd == null)
            {
                return false;
            }
            return (mmd.getMap() != null ? mmd.getMap().isSerializedValue() : false);
        }
        else
        {
            return (mmd != null ? mmd.isSerialized() : false);
        }
    }

    /**
     * Accessor for whether this mapping is nullable
     * @return Whether it is nullable
     */
    public boolean isNullable()
    {
        for (int i=0; i<datastoreMappings.length; i++)
        {
            if (!datastoreMappings[i].isNullable())
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Whether the mapping has a simple (single column) datastore representation.
     * @return Whether it has a simple datastore representation (single column)
     */
    public boolean hasSimpleDatastoreRepresentation()
    {
        return true;
    }

    /**
     * Accessor for the datastore mappings for this java type
     * @return The datastore mapping(s)
     */
    public DatastoreMapping[] getDataStoreMappings()
    {
        return datastoreMappings;
    }

    /**
     * Accessor for the datastore class (e.g in an RDBMS context, the Table).
     * Will be null if this mapping is for a literal in a query.
     * @return The datastore class containing this mapped field.
     */
    public DatastoreContainerObject getDatastoreContainer()
    {
        return datastoreContainer;
    }

    /**
     * Accessor for a datastore mapping
     * @param index The id of the mapping
     * @return The datastore mapping
     */
    public DatastoreMapping getDataStoreMapping(int index)
    {
        return datastoreMappings[index];
    }

    /**
     * Accessor for the mapping at the other end of a relation when this field is
     * part of a 1-1, 1-N, M-N (bidirectional) relation. Will be null otherwise.
     * @return The mapping at the other end.
     */
    public JavaTypeMapping getReferenceMapping()
    {
        return referenceMapping; 
    }

    /**
     * Method to set the mapping at the other end of the relation.
     * Only used when part of a (bidirectional) relation.
     * @param referenceMapping The mapping at the other end
     */
    public void setReferenceMapping(JavaTypeMapping referenceMapping)
    {
        this.referenceMapping = referenceMapping; 
    }

    /**
     * Method to add a datastore mapping
     * @param datastoreMapping The datastore mapping
     */
    public void addDataStoreMapping(DatastoreMapping datastoreMapping)
    {
        DatastoreMapping[] dm = datastoreMappings;
        datastoreMappings = new DatastoreMapping[datastoreMappings.length+1];
        System.arraycopy(dm, 0, datastoreMappings, 0, dm.length);
        datastoreMappings[dm.length] = datastoreMapping;
    }

    /**
     * Acessor for the number of datastore fields (e.g. RDBMS number of columns)
     * @return the number of datastore fields
     */
    public int getNumberOfDatastoreFields()
    {
        return datastoreMappings.length;
    }

    /**
     * Accessor for the java type being mapped.
     * This is the java type that the mapping represents. Some examples :
     * <ul>
     * <li>if the field is of type "MyClass" then the mapping will be OIDMapping (or subclass)
     * the javaType will be OID, and the type will be MyClass.</li>
     * <li>if the field is of type "int" then the mapping will be IntegerMapping, the javaType will
     * be Integer, and the type will be int.</li>
     * </ul>
     * The "java type" is the java-type name used in the plugin.xml mapping file
     * @return The java type
     */
    public abstract Class getJavaType();

    /**
     * Accessor for the name of the java-type actually used when mapping the particular datastore
     * field. This java-type must have an entry in the datastore mappings.
     * The default implementation throws an UnsupportedOperationException.
     * TODO Merge this with getJavaType().
     *
     * @param index requested datastore field index.
     * @return the name of java-type for the requested datastore field.
     */
    public String getJavaTypeForDatastoreMapping(int index)
    {
        throw new UnsupportedOperationException("Datastore type mapping is not supported by: "+getClass());
    }

    /**
     * Accessor for the class name of the object that is being mapped here.
     * There are mainly two situations: 
     * <ul>
     * <li>For a JavaTypeMapping that maps a PersistentCapable class field, this will return
     * the type of the field. For example with a field of type "MyClass" this will return "MyClass"</li>
     * <li>For a JavaTypeMapping that maps a variable or parameter in a query, this will return
     * the type declared in the query.</li>
     * </ul>
     * @return The actual type that this Mapping maps.
     */
    public String getType()
    {
        return type;
    }

    /**
     * Return a sample value of the mapping type to be used for internal
     * evaluation of type and conversion.
     * @param clr TODO
     * @return The sample value. 
     */
    public abstract Object getSampleValue(ClassLoaderResolver clr);

    /**
     * Accessor for whether this mapping is to be included in any fetch statement.
     * @return Whether to include this mapping in a fetch statement
     */
    public boolean includeInFetchStatement()
    {
        return true;
    }

    /**
     * Accessor for whether this mapping is to be included in the update statement.
     * @return Whether to include in update statement
     */
    public boolean includeInUpdateStatement()
    {
        return true;
    }

    /**
     * Accessor for whether this mapping is to be included in the insert statement.
     * @return Whether to include in insert statement
     */
    public boolean includeInInsertStatement()
    {
        return true;
    }

    /**
     * Utility to output any error message.
     * @param method The method that failed.
     * @return the localised failure message
     **/
    protected String failureMessage(String method)
    {
        return LOCALISER.msg("041004",getClass().getName(),method);
    }

    // ------------------- Accessors & Mutators for datastore access -----------------------

    /**
     * Convenience setter to provide a default value for this field.
     * If the field is nullable, a null is provided, otherwise the sample value.
     * @param om The ObjectManager
     * @param datastoreStatement Prepared Statement
     * @param exprIndex The indices in the statement
     */
    public void setDefault(ObjectManager om, Object datastoreStatement, int[] exprIndex)
    {
        // TODO Dont just use the first datastore mapping and parameter number
        getDataStoreMapping(0).setObject(datastoreStatement, exprIndex[0], isNullable() ? null : getSampleValue(om.getClassLoaderResolver()));
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setBoolean(ObjectManager om, Object datastoreStatement, int[] exprIndex, boolean value)
    {
        throw new NucleusException(failureMessage("setBoolean")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public boolean getBoolean(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("setBoolean")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setChar(ObjectManager om, Object datastoreStatement, int[] exprIndex, char value)
    {
        throw new NucleusException(failureMessage("setChar")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public char getChar(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getChar")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setByte(ObjectManager om, Object datastoreStatement, int[] exprIndex, byte value)
    {
        throw new NucleusException(failureMessage("setByte")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public byte getByte(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getByte")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setShort(ObjectManager om, Object datastoreStatement, int[] exprIndex, short value)
    {
        throw new NucleusException(failureMessage("setShort")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public short getShort(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getShort")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setInt(ObjectManager om, Object datastoreStatement, int[] exprIndex, int value)
    {
        throw new NucleusException(failureMessage("setInt")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public int getInt(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getInt")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setLong(ObjectManager om, Object datastoreStatement, int[] exprIndex, long value)
    {
        throw new NucleusException(failureMessage("setLong")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public long getLong(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getLong")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setFloat(ObjectManager om, Object datastoreStatement, int[] exprIndex, float value)
    {
        throw new NucleusException(failureMessage("setFloat")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public float getFloat(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getFloat")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setDouble(ObjectManager om, Object datastoreStatement, int[] exprIndex, double value)
    {
        throw new NucleusException(failureMessage("setDouble")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public double getDouble(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getDouble")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setString(ObjectManager om, Object datastoreStatement, int[] exprIndex, String value)
    {
        throw new NucleusException(failureMessage("setString")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
    public String getString(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getString")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     * @param ownerSM the owner StateManager
     * @param ownerFieldNumber the owner absolute field number
     */    
    public void setObject(ObjectManager om, Object datastoreStatement, int[] exprIndex, Object value, StateManager ownerSM, int ownerFieldNumber)
    {
        throw new NucleusException(failureMessage("setObject")).setFatal();
    }

    /**
     * Sets a <code>value</code> into <code>datastoreStatement</code> 
     * at position specified by <code>exprIndex</code>.
     * @param om the ObjectManager 
     * @param datastoreStatement a datastore object that executes statements in the database 
     * @param exprIndex the position of the value in the statement
     * @param value the value
     */
    public void setObject(ObjectManager om, Object datastoreStatement, int[] exprIndex, Object value)
    {
        throw new NucleusException(failureMessage("setObject")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @param ownerSM the owner StateManager
     * @param ownerFieldNumber the owner absolute field number
     * @return the value
     */
    public Object getObject(ObjectManager om, Object datastoreResults, int[] exprIndex, StateManager ownerSM, int ownerFieldNumber)
    {
        throw new NucleusException(failureMessage("getObject")).setFatal();
    }

    /**
     * Obtains a value from <code>datastoreResults</code> 
     * at position specified by <code>exprIndex</code>. 
     * @param om the ObjectManager 
     * @param datastoreResults an object returned from the datastore with values 
     * @param exprIndex the position of the value in the result
     * @return the value
     */
   public Object getObject(ObjectManager om, Object datastoreResults, int[] exprIndex)
    {
        throw new NucleusException(failureMessage("getObject")).setFatal();
    }

   /**
    * Creates a literal from an value for use in TJDO-style JDOQL.
    * This implementation just throws a NucleusException.
    * @param qs The Query statement
    * @param value The object
    * @return A Scalar Expression
    */
   public ScalarExpression newLiteral(QueryExpression qs, Object value)
   {
       throw new NucleusException(failureMessage("newLiteral")).setFatal();
   }

   /**
    * Creates a expression from a field name/table. e.g. tablename.fieldname; tablealias.fieldalias
    * For use in TJDO-style JDOQL.
    * This implementation just throws a NucleusException.
    * @param qs The Query statement
    * @param te the alias for the table
    * @return A Scalar Expression
    */ 
   public ScalarExpression newScalarExpression(QueryExpression qs, LogicSetExpression te)
   {
       throw new NucleusException(failureMessage("newScalarExpression")).setFatal();
   }
}