/**********************************************************************
Copyright (c) 2009 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.api.jpa.criteria;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.persistence.Tuple;
import javax.persistence.criteria.CompoundSelection;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Order;
import javax.persistence.criteria.ParameterExpression;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Selection;
import javax.persistence.criteria.Subquery;
import javax.persistence.criteria.Predicate.BooleanOperator;

import org.datanucleus.api.jpa.metamodel.MetamodelImpl;
import org.datanucleus.query.expression.CreatorExpression;
import org.datanucleus.query.expression.DyadicExpression;
import org.datanucleus.query.expression.InvokeExpression;
import org.datanucleus.query.expression.Literal;
import org.datanucleus.query.expression.SubqueryExpression;
import org.datanucleus.query.expression.VariableExpression;

/**
 * Implementation of JPA2 Criteria "QueryBuilder".
 * Acts as a factory for expressions, and also for the criteria queries themselves.
 */
public class CriteriaBuilderImpl implements CriteriaBuilder
{
    MetamodelImpl model;

    public CriteriaBuilderImpl(MetamodelImpl model)
    {
        this.model = model;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#createQuery()
     */
    public CriteriaQuery<Object> createQuery()
    {
        return new CriteriaQueryImpl<Object>(model, null);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#createQuery(java.lang.Class)
     */
    public <T> CriteriaQuery<T> createQuery(Class<T> resultCls)
    {
        return new CriteriaQueryImpl<T>(model, resultCls);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#createTupleQuery()
     */
    public CriteriaQuery<Tuple> createTupleQuery()
    {
        return new CriteriaQueryImpl<Tuple>(model, Tuple.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#asc(javax.persistence.criteria.Expression)
     */
    public Order asc(Expression<?> expr)
    {
        return new OrderImpl(expr, true);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#desc(javax.persistence.criteria.Expression)
     */
    public Order desc(Expression<?> expr)
    {
        return new OrderImpl(expr, false);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#abs(javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> abs(Expression<N> expr)
    {
        ExpressionImpl<N> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "avg", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#avg(javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<Double> avg(Expression<N> expr)
    {
        ExpressionImpl<Double> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "avg", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#count(javax.persistence.criteria.Expression)
     */
    public Expression<Long> count(Expression<?> expr)
    {
        ExpressionImpl<Long> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "count", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#countDistinct(javax.persistence.criteria.Expression)
     */
    public Expression<Long> countDistinct(Expression<?> expr)
    {
        ExpressionImpl<Long> select = new ExpressionImpl(expr.getJavaType());
        DyadicExpression dyExpr =
            new DyadicExpression(org.datanucleus.query.expression.Expression.OP_DISTINCT,
                ((ExpressionImpl)expr).getQueryExpression());
        List args = new ArrayList();
        args.add(dyExpr);
        select.queryExpr = new InvokeExpression(null, "count", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#max(javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> max(Expression<N> expr)
    {
        ExpressionImpl<N> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "max", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#greatest(javax.persistence.criteria.Expression)
     */
    public <X extends Comparable<? super X>> Expression<X> greatest(Expression<X> expr)
    {
        ExpressionImpl<X> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "max", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#min(javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> min(Expression<N> expr)
    {
        ExpressionImpl<N> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "min", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#least(javax.persistence.criteria.Expression)
     */
    public <X extends Comparable<? super X>> Expression<X> least(Expression<X> expr)
    {
        ExpressionImpl<X> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "min", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#sqrt(javax.persistence.criteria.Expression)
     */
    public Expression<Double> sqrt(Expression<? extends Number> expr)
    {
        ExpressionImpl<Double> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "sqrt", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#sum(javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> sum(Expression<N> expr)
    {
        ExpressionImpl<N> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "sum", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.CriteriaBuilder#sumAsDouble(javax.persistence.criteria.Expression)
     */
    public Expression<Double> sumAsDouble(Expression<Float> expr)
    {
        ExpressionImpl<Double> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "sum", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.CriteriaBuilder#sumAsLong(javax.persistence.criteria.Expression)
     */
    public Expression<Long> sumAsLong(Expression<Integer> expr)
    {
        ExpressionImpl<Long> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        select.queryExpr = new InvokeExpression(null, "sum", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#and(javax.persistence.criteria.Predicate[])
     */
    public Predicate and(Predicate... preds)
    {
        PredicateImpl pred = new PredicateImpl();
        for (int i=0;i<preds.length;i++)
        {
            pred.append(preds[i]);
        }

        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#and(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate and(Expression<Boolean> expr0, Expression<Boolean> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_AND,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#or(javax.persistence.criteria.Predicate[])
     */
    public Predicate or(Predicate... preds)
    {
        PredicateImpl pred = new PredicateImpl(BooleanOperator.OR);
        for (int i=0;i<preds.length;i++)
        {
            pred.append(preds[i]);
        }

        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#or(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate or(Expression<Boolean> expr0, Expression<Boolean> expr1)
    {
        PredicateImpl pred = new PredicateImpl(BooleanOperator.OR);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_OR,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#equal(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate equal(Expression<?> expr0, Expression<?> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_EQ,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#equal(javax.persistence.criteria.Expression, java.lang.Object)
     */
    public Predicate equal(Expression<?> expr, Object obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_EQ,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#notEqual(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate notEqual(Expression<?> expr0, Expression<?> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_NOTEQ,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#notEqual(javax.persistence.criteria.Expression, java.lang.Object)
     */
    public Predicate notEqual(Expression<?> expr, Object obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_NOTEQ,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isNotNull(javax.persistence.criteria.Expression)
     */
    public Predicate isNotNull(Expression<?> expr)
    {
        return expr.isNotNull();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isNull(javax.persistence.criteria.Expression)
     */
    public Predicate isNull(Expression<?> expr)
    {
        return expr.isNull();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#ge(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate ge(Expression<? extends Number> expr0, Expression<? extends Number> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_GTEQ,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#ge(javax.persistence.criteria.Expression, java.lang.Number)
     */
    public Predicate ge(Expression<? extends Number> expr, Number obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_GTEQ,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#greaterThan(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <Y extends Comparable<? super Y>> Predicate greaterThan(Expression<? extends Y> expr0, Expression<? extends Y> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_GT,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#greaterThan(javax.persistence.criteria.Expression, java.lang.Comparable)
     */
    public <Y extends Comparable<? super Y>> Predicate greaterThan(Expression<? extends Y> expr, Y obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_GT,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#greaterThanOrEqualTo(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <Y extends Comparable<? super Y>> Predicate greaterThanOrEqualTo(Expression<? extends Y> expr0, Expression<? extends Y> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_GTEQ,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#greaterThanOrEqualTo(javax.persistence.criteria.Expression, java.lang.Comparable)
     */
    public <Y extends Comparable<? super Y>> Predicate greaterThanOrEqualTo(Expression<? extends Y> expr, Y obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_GTEQ,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#gt(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate gt(Expression<? extends Number> expr0, Expression<? extends Number> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_GT,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#gt(javax.persistence.criteria.Expression, java.lang.Number)
     */
    public Predicate gt(Expression<? extends Number> expr, Number obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_GT,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#le(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate le(Expression<? extends Number> expr0, Expression<? extends Number> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_LTEQ,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#le(javax.persistence.criteria.Expression, java.lang.Number)
     */
    public Predicate le(Expression<? extends Number> expr, Number obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_LTEQ,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#lessThan(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <Y extends Comparable<? super Y>> Predicate lessThan(Expression<? extends Y> expr0, Expression<? extends Y> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_LT,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#lessThan(javax.persistence.criteria.Expression, java.lang.Comparable)
     */
    public <Y extends Comparable<? super Y>> Predicate lessThan(Expression<? extends Y> expr, Y obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_LT,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#lessThanOrEqualTo(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <Y extends Comparable<? super Y>> Predicate lessThanOrEqualTo(Expression<? extends Y> expr0, Expression<? extends Y> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_LTEQ,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#lessThanOrEqualTo(javax.persistence.criteria.Expression, java.lang.Comparable)
     */
    public <Y extends Comparable<? super Y>> Predicate lessThanOrEqualTo(Expression<? extends Y> expr, Y obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_LTEQ,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#lt(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate lt(Expression<? extends Number> expr0, Expression<? extends Number> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_LT,
                ((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#lt(javax.persistence.criteria.Expression, java.lang.Number)
     */
    public Predicate lt(Expression<? extends Number> expr, Number obj)
    {
        PredicateImpl pred = new PredicateImpl();
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_LT,
                lit);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#sum(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> sum(Expression<? extends N> expr0, Expression<? extends N> expr1)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>) expr0.getJavaType());
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_ADD, 
                ((ExpressionImpl)expr1).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#sum(javax.persistence.criteria.Expression, java.lang.Number)
     */
    public <N extends Number> Expression<N> sum(Expression<? extends N> expr, N obj)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>) expr.getJavaType());
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_ADD, lit);
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#sum(java.lang.Number, javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> sum(N obj, Expression<? extends N> expr)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>) expr.getJavaType());
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(lit, org.datanucleus.query.expression.Expression.OP_ADD,
                ((ExpressionImpl)expr).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#quot(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<Number> quot(Expression<? extends Number> expr0, Expression<? extends Number> expr1)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<Number>(Number.class);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_DIV,
                ((ExpressionImpl)expr1).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#quot(javax.persistence.criteria.Expression, java.lang.Number)
     */
    public Expression<Number> quot(Expression<? extends Number> expr, Number obj)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<Number>(Number.class);
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_DIV, lit);
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#quot(java.lang.Number, javax.persistence.criteria.Expression)
     */
    public Expression<Number> quot(Number obj, Expression<? extends Number> expr)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<Number>(Number.class);
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(lit, org.datanucleus.query.expression.Expression.OP_DIV,
                ((ExpressionImpl)expr).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#diff(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> diff(Expression<? extends N> expr0, Expression<? extends N> expr1)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>)expr0.getJavaType());
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(), 
                org.datanucleus.query.expression.Expression.OP_SUB,
                ((ExpressionImpl)expr1).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#diff(javax.persistence.criteria.Expression, java.lang.Number)
     */
    public <N extends Number> Expression<N> diff(Expression<? extends N> expr, N obj)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>) expr.getJavaType());
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(), 
                org.datanucleus.query.expression.Expression.OP_SUB, lit);
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#diff(java.lang.Number, javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> diff(N obj, Expression<? extends N> expr)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>) expr.getJavaType());
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(lit, org.datanucleus.query.expression.Expression.OP_SUB,
                ((ExpressionImpl)expr).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#prod(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> prod(Expression<? extends N> expr0, Expression<? extends N> expr1)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>)expr0.getJavaType());
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(), 
                org.datanucleus.query.expression.Expression.OP_MUL,
                ((ExpressionImpl)expr1).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#prod(javax.persistence.criteria.Expression, java.lang.Number)
     */
    public <N extends Number> Expression<N> prod(Expression<? extends N> expr, N obj)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>) expr.getJavaType());
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(), 
                org.datanucleus.query.expression.Expression.OP_MUL, lit);
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#prod(java.lang.Number, javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> prod(N obj, Expression<? extends N> expr)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<N>((Class<N>) expr.getJavaType());
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(lit, org.datanucleus.query.expression.Expression.OP_MUL,
                ((ExpressionImpl)expr).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#mod(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<Integer> mod(Expression<Integer> expr0, Expression<Integer> expr1)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<Integer>(Integer.class);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).getQueryExpression(), 
                org.datanucleus.query.expression.Expression.OP_MOD,
                ((ExpressionImpl)expr1).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#mod(javax.persistence.criteria.Expression, java.lang.Integer)
     */
    public Expression<Integer> mod(Expression<Integer> expr, Integer obj)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<Integer>(Integer.class);
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(), 
                org.datanucleus.query.expression.Expression.OP_MOD, lit);
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#mod(java.lang.Integer, javax.persistence.criteria.Expression)
     */
    public Expression<Integer> mod(Integer obj, Expression<Integer> expr)
    {
        ExpressionImpl sumExpr = new ExpressionImpl<Integer>(Integer.class);
        Literal lit = new Literal(obj);
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(lit, org.datanucleus.query.expression.Expression.OP_MOD,
                ((ExpressionImpl)expr).getQueryExpression());
        sumExpr.queryExpr = queryExpr;
        return sumExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#between(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <Y extends Comparable<? super Y>> Predicate between(Expression<? extends Y> expr0, Expression<? extends Y> expr1,
            Expression<? extends Y> expr2)
    {
        PredicateImpl pred = new PredicateImpl();

        org.datanucleus.query.expression.Expression theExpr = ((ExpressionImpl)expr0).getQueryExpression();
        org.datanucleus.query.expression.Expression lowerExpr = ((ExpressionImpl)expr1).getQueryExpression();
        org.datanucleus.query.expression.Expression upperExpr = ((ExpressionImpl)expr2).getQueryExpression();
        DyadicExpression lowerDyadic =
            new DyadicExpression(theExpr, org.datanucleus.query.expression.Expression.OP_GTEQ, lowerExpr);
        DyadicExpression upperDyadic =
            new DyadicExpression(theExpr, org.datanucleus.query.expression.Expression.OP_LTEQ, upperExpr);
        DyadicExpression overallDyadic =
            new DyadicExpression(lowerDyadic, org.datanucleus.query.expression.Expression.OP_AND, upperDyadic);
        pred.queryExpr = overallDyadic;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#between(javax.persistence.criteria.Expression, java.lang.Comparable, java.lang.Comparable)
     */
    public <Y extends Comparable<? super Y>> Predicate between(Expression<? extends Y> expr, Y obj0, Y obj1)
    {
        PredicateImpl pred = new PredicateImpl();

        org.datanucleus.query.expression.Expression theExpr = ((ExpressionImpl)expr).getQueryExpression();
        Literal litLower = new Literal(obj0);
        Literal litUpper = new Literal(obj1);
        DyadicExpression lowerDyadic =
            new DyadicExpression(theExpr, org.datanucleus.query.expression.Expression.OP_GTEQ, litLower);
        DyadicExpression upperDyadic =
            new DyadicExpression(theExpr, org.datanucleus.query.expression.Expression.OP_LTEQ, litUpper);
        DyadicExpression overallDyadic =
            new DyadicExpression(lowerDyadic, org.datanucleus.query.expression.Expression.OP_AND, upperDyadic);
        pred.queryExpr = overallDyadic;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#coalesce()
     */
    public <T> Coalesce<T> coalesce()
    {
        return new CoalesceImpl(Object.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#coalesce(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <Y> Expression<Y> coalesce(Expression<? extends Y> expr0, Expression<? extends Y> expr1)
    {
        ExpressionImpl<Y> coalExpr = new ExpressionImpl<Y>((Class<Y>) expr0.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr0).getQueryExpression());
        args.add(((ExpressionImpl)expr1).getQueryExpression());
        coalExpr.queryExpr = new InvokeExpression(null, "COALESCE", args);
        return coalExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#coalesce(javax.persistence.criteria.Expression, java.lang.Object)
     */
    public <Y> Expression<Y> coalesce(Expression<? extends Y> expr, Y val)
    {
        ExpressionImpl<Y> coalExpr = new ExpressionImpl<Y>((Class<Y>)expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        args.add(new Literal(val));
        coalExpr.queryExpr = new InvokeExpression(null, "COALESCE", args);
        return coalExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#nullif(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <Y> Expression<Y> nullif(Expression<Y> expr0, Expression<?> expr1)
    {
        ExpressionImpl<Y> coalExpr = new ExpressionImpl<Y>((Class<Y>) expr0.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr0).getQueryExpression());
        args.add(((ExpressionImpl)expr1).getQueryExpression());
        coalExpr.queryExpr = new InvokeExpression(null, "NULLIF", args);
        return coalExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#nullif(javax.persistence.criteria.Expression, java.lang.Object)
     */
    public <Y> Expression<Y> nullif(Expression<Y> expr, Y val)
    {
        ExpressionImpl<Y> coalExpr = new ExpressionImpl<Y>((Class<Y>)expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        args.add(new Literal(val));
        coalExpr.queryExpr = new InvokeExpression(null, "NULLIF", args);
        return coalExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#conjunction()
     */
    public Predicate conjunction()
    {
        PredicateImpl pred = new PredicateImpl();
        pred.queryExpr = new Literal(Boolean.TRUE);
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#disjunction()
     */
    public Predicate disjunction()
    {
        PredicateImpl pred = new PredicateImpl();
        pred.queryExpr = new Literal(Boolean.FALSE);
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#construct(java.lang.Class, javax.persistence.criteria.Selection<?>[])
     */
    public <Y> CompoundSelection<Y> construct(Class<Y> cls, Selection<?>... args)
    {
        CompoundSelectionImpl<Y> select = new CompoundSelectionImpl<Y>(cls, args);
        List<String> clsNameComponents = new ArrayList();
        StringTokenizer tok = new StringTokenizer(cls.getName(), ".");
        while (tok.hasMoreTokens())
        {
            clsNameComponents.add(tok.nextToken());
        }
        List<org.datanucleus.query.expression.Expression> ctrArgs = new ArrayList();
        if (args != null)
        {
            for (int i=0;i<args.length;i++)
            {
                ExpressionImpl argExpr = (ExpressionImpl) args[i];
                ctrArgs.add(argExpr.queryExpr);
            }
        }
        select.queryExpr = new CreatorExpression(clsNameComponents, ctrArgs);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#currentDate()
     */
    public Expression<Date> currentDate()
    {
        ExpressionImpl<Date> select = new ExpressionImpl(Date.class);
        select.queryExpr = new InvokeExpression(null, "CURRENT_DATE", null);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#currentTime()
     */
    public Expression<Time> currentTime()
    {
        ExpressionImpl<Time> select = new ExpressionImpl(Time.class);
        select.queryExpr = new InvokeExpression(null, "CURRENT_TIME", null);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#currentTimestamp()
     */
    public Expression<Timestamp> currentTimestamp()
    {
        ExpressionImpl<Timestamp> select = new ExpressionImpl(Timestamp.class);
        select.queryExpr = new InvokeExpression(null, "CURRENT_TIMESTAMP", null);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#function(java.lang.String, java.lang.Class, javax.persistence.criteria.Expression<?>[])
     */
    public <T> Expression<T> function(String funcName, Class<T> returnType, Expression<?>... argExprs)
    {
        ExpressionImpl<T> funcExpr = new ExpressionImpl(returnType);
        List<org.datanucleus.query.expression.Expression> args = new ArrayList();
        if (argExprs != null)
        {
            for (int i=0;i<argExprs.length;i++)
            {
                args.add(((ExpressionImpl<?>)argExprs[i]).getQueryExpression());
            }
        }
        funcExpr.queryExpr = new InvokeExpression(null, funcName, args);
        return funcExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#all(javax.persistence.criteria.Subquery)
     */
    public <Y> Expression<Y> all(Subquery<Y> sub)
    {
        ExpressionImpl<Y> allExpr = new ExpressionImpl<Y>((Class<Y>) sub.getJavaType());
        org.datanucleus.query.expression.Expression subExpr = ((SubqueryImpl<Y>)sub).getQueryExpression();
        String varName = null;
        if (subExpr instanceof VariableExpression)
        {
            varName = ((VariableExpression)subExpr).getId();
        }
        else
        {
            varName = "SUB" + SubqueryImpl.random.nextInt();
        }
        allExpr.queryExpr = new SubqueryExpression("ALL", new VariableExpression(varName));
        return allExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#any(javax.persistence.criteria.Subquery)
     */
    public <Y> Expression<Y> any(Subquery<Y> sub)
    {
        ExpressionImpl<Y> allExpr = new ExpressionImpl<Y>((Class<Y>) sub.getJavaType());
        org.datanucleus.query.expression.Expression subExpr = ((SubqueryImpl<Y>)sub).getQueryExpression();
        String varName = null;
        if (subExpr instanceof VariableExpression)
        {
            varName = ((VariableExpression)subExpr).getId();
        }
        else
        {
            varName = "SUB" + SubqueryImpl.random.nextInt();
        }
        allExpr.queryExpr = new SubqueryExpression("ANY", new VariableExpression(varName));
        return allExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#some(javax.persistence.criteria.Subquery)
     */
    public <Y> Expression<Y> some(Subquery<Y> sub)
    {
        ExpressionImpl<Y> allExpr = new ExpressionImpl<Y>((Class<Y>) sub.getJavaType());
        org.datanucleus.query.expression.Expression subExpr = ((SubqueryImpl<Y>)sub).getQueryExpression();
        String varName = null;
        if (subExpr instanceof VariableExpression)
        {
            varName = ((VariableExpression)subExpr).getId();
        }
        else
        {
            varName = "SUB" + SubqueryImpl.random.nextInt();
        }
        allExpr.queryExpr = new SubqueryExpression("SOME", new VariableExpression(varName));
        return allExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#exists(javax.persistence.criteria.Subquery)
     */
    public Predicate exists(Subquery<?> sub)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression subExpr = ((SubqueryImpl<?>)sub).getQueryExpression();
        String varName = null;
        if (subExpr instanceof VariableExpression)
        {
            varName = ((VariableExpression)subExpr).getId();
        }
        else
        {
            varName = "SUB" + SubqueryImpl.random.nextInt();
        }
        pred.queryExpr = new SubqueryExpression("EXISTS", new VariableExpression(varName));
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#in(javax.persistence.criteria.Expression)
     */
    public <T> In<T> in(Expression<? extends T> expr)
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isEmpty(javax.persistence.criteria.Expression)
     */
    public <C extends Collection<?>> Predicate isEmpty(Expression<C> collExpr)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new InvokeExpression(((ExpressionImpl)collExpr).getQueryExpression(), "isEmpty", null);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isMember(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <E, C extends Collection<E>> Predicate isMember(Expression<E> expr, Expression<C> collExpr)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr).getQueryExpression());
        org.datanucleus.query.expression.Expression queryExpr =
            new InvokeExpression(((ExpressionImpl)collExpr).getQueryExpression(), "contains", args);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isMember(java.lang.Object, javax.persistence.criteria.Expression)
     */
    public <E, C extends Collection<E>> Predicate isMember(E val, Expression<C> collExpr)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        Literal lit = new Literal(val);
        args.add(lit);
        org.datanucleus.query.expression.Expression queryExpr =
            new InvokeExpression(((ExpressionImpl)collExpr).getQueryExpression(), "contains", args);
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isNotEmpty(javax.persistence.criteria.Expression)
     */
    public <C extends Collection<?>> Predicate isNotEmpty(Expression<C> collExpr)
    {
        Predicate pred = isEmpty(collExpr);
        return pred.not();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isNotMember(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public <E, C extends Collection<E>> Predicate isNotMember(Expression<E> expr, Expression<C> collExpr)
    {
        Predicate pred = isMember(expr, collExpr);
        return pred.not();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isNotMember(java.lang.Object, javax.persistence.criteria.Expression)
     */
    public <E, C extends Collection<E>> Predicate isNotMember(E val, Expression<C> collExpr)
    {
        Predicate pred = isMember(val, collExpr);
        return pred.not();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#size(javax.persistence.criteria.Expression)
     */
    public <C extends Collection<?>> Expression<Integer> size(Expression<C> expr)
    {
        ExpressionImpl<Integer> collSizeExpr = new ExpressionImpl(expr.getJavaType());
        collSizeExpr.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "size", null);
        return collSizeExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#size(java.util.Collection)
     */
    public <C extends Collection<?>> Expression<Integer> size(C coll)
    {
        // Strange method that seemingly just returns the size of the input collection, so why have it?
        ExpressionImpl<Integer> collSizeExpr = new ExpressionImpl<Integer>(Integer.class);
        collSizeExpr.queryExpr = new Literal(coll.size());
        return collSizeExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isFalse(javax.persistence.criteria.Expression)
     */
    public Predicate isFalse(Expression<Boolean> expr)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_EQ,
                new Literal(Boolean.FALSE));
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#isTrue(javax.persistence.criteria.Expression)
     */
    public Predicate isTrue(Expression<Boolean> expr)
    {
        PredicateImpl pred = new PredicateImpl();
        org.datanucleus.query.expression.Expression queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).getQueryExpression(),
                org.datanucleus.query.expression.Expression.OP_EQ,
                new Literal(Boolean.TRUE));
        pred.queryExpr = queryExpr;
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#tuple(javax.persistence.criteria.Selection<?>[])
     */
    public CompoundSelection<Tuple> tuple(Selection<?>... arg0)
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#array(javax.persistence.criteria.Selection<?>[])
     */
    public CompoundSelection<Object[]> array(Selection<?>... arg0)
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#keys(java.util.Map)
     */
    public <K, M extends Map<K, ?>> Expression<Set<K>> keys(M arg0)
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#values(java.util.Map)
     */
    public <V, M extends Map<?, V>> Expression<Collection<V>> values(M arg0)
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#like(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate like(Expression<String> expr, Expression<String> expr1)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr1).getQueryExpression());
        pred.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "matches", args);
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#like(javax.persistence.criteria.Expression, java.lang.String)
     */
    public Predicate like(Expression<String> expr, String regex)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        args.add(new Literal(regex));
        pred.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "matches", args);
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#like(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate like(Expression<String> expr, Expression<String> expr1, Expression<Character> escExpr)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr1).getQueryExpression());
        args.add(((ExpressionImpl)escExpr).getQueryExpression());
        pred.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "matches", args);
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#like(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression, char)
     */
    public Predicate like(Expression<String> expr, Expression<String> expr1, char escChr)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        args.add(((ExpressionImpl)expr1).getQueryExpression());
        args.add(new Literal(escChr));
        pred.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "matches", args);
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#like(javax.persistence.criteria.Expression, java.lang.String, javax.persistence.criteria.Expression)
     */
    public Predicate like(Expression<String> expr, String regex, Expression<Character> escExpr)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        args.add(new Literal(regex));
        args.add(((ExpressionImpl)escExpr).getQueryExpression());
        pred.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "matches", args);
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#like(javax.persistence.criteria.Expression, java.lang.String, char)
     */
    public Predicate like(Expression<String> expr, String regex, char escChr)
    {
        PredicateImpl pred = new PredicateImpl();
        List args = new ArrayList();
        args.add(new Literal(regex));
        args.add(new Literal(escChr));
        pred.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "matches", args);
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#neg(javax.persistence.criteria.Expression)
     */
    public <N extends Number> Expression<N> neg(Expression<N> expr)
    {
        ExpressionImpl<N> negExpr = new ExpressionImpl<N>((Class<N>) expr.getJavaType());
        negExpr.queryExpr = new DyadicExpression(org.datanucleus.query.expression.Expression.OP_NEG,
            ((ExpressionImpl)expr).getQueryExpression());
        return negExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#not(javax.persistence.criteria.Expression)
     */
    public Predicate not(Expression<Boolean> expr)
    {
        PredicateImpl pred = new PredicateImpl();
        pred.queryExpr = new DyadicExpression(org.datanucleus.query.expression.Expression.OP_NOT,
            ((ExpressionImpl)expr).getQueryExpression());
        return pred;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#notLike(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate notLike(Expression<String> expr, Expression<String> expr1)
    {
        Predicate pred = like(expr, expr1);
        return pred.not();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#notLike(javax.persistence.criteria.Expression, java.lang.String)
     */
    public Predicate notLike(Expression<String> expr, String regex)
    {
        Predicate pred = like(expr, regex);
        return pred.not();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#notLike(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Predicate notLike(Expression<String> expr, Expression<String> expr1, Expression<Character> escExpr)
    {
        Predicate pred = like(expr, expr1, escExpr);
        return pred.not();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#notLike(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression, char)
     */
    public Predicate notLike(Expression<String> expr, Expression<String> expr1, char escChr)
    {
        Predicate pred = like(expr, expr1, escChr);
        return pred.not();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#notLike(javax.persistence.criteria.Expression, java.lang.String, javax.persistence.criteria.Expression)
     */
    public Predicate notLike(Expression<String> expr, String regex, Expression<Character> escExpr)
    {
        Predicate pred = like(expr, regex, escExpr);
        return pred.not();
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#notLike(javax.persistence.criteria.Expression, java.lang.String, char)
     */
    public Predicate notLike(Expression<String> expr, String regex, char escChr)
    {
        Predicate pred = like(expr, regex, escChr);
        return pred.not();
    }

    private static long PARAM_NUMBER = 0;

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#parameter(java.lang.Class)
     */
    public <T> ParameterExpression<T> parameter(Class<T> cls)
    {
        // No name specified so add a dummy name
        String paramName = "DN_PARAM_" + (PARAM_NUMBER++);
        ParameterExpressionImpl<T> param = new ParameterExpressionImpl<T>(cls, paramName);
        param.queryExpr = new org.datanucleus.query.expression.ParameterExpression(paramName, cls);
        return param;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#parameter(java.lang.Class, java.lang.String)
     */
    public <T> ParameterExpression<T> parameter(Class<T> cls, String name)
    {
        ParameterExpressionImpl<T> param = new ParameterExpressionImpl<T>(cls, name);
        param.queryExpr = new org.datanucleus.query.expression.ParameterExpression(name, cls);
        return param;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#selectCase()
     */
    public <R> Case<R> selectCase()
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#selectCase(javax.persistence.criteria.Expression)
     */
    public <C, R> SimpleCase<C, R> selectCase(Expression<? extends C> expr)
    {
        // TODO Implement this
        throw new UnsupportedOperationException(
            "Not yet implemented. Provide a testcase that uses this and raise a JIRA attaching your testcase");
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#toBigDecimal(javax.persistence.criteria.Expression)
     */
    public Expression<BigDecimal> toBigDecimal(Expression<? extends Number> expr)
    {
        return expr.as(BigDecimal.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#toBigInteger(javax.persistence.criteria.Expression)
     */
    public Expression<BigInteger> toBigInteger(Expression<? extends Number> expr)
    {
        return expr.as(BigInteger.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#toDouble(javax.persistence.criteria.Expression)
     */
    public Expression<Double> toDouble(Expression<? extends Number> expr)
    {
        return expr.as(Double.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#toFloat(javax.persistence.criteria.Expression)
     */
    public Expression<Float> toFloat(Expression<? extends Number> expr)
    {
        return expr.as(Float.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#toInteger(javax.persistence.criteria.Expression)
     */
    public Expression<Integer> toInteger(Expression<? extends Number> expr)
    {
        return expr.as(Integer.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#toLong(javax.persistence.criteria.Expression)
     */
    public Expression<Long> toLong(Expression<? extends Number> expr)
    {
        return expr.as(Long.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#toString(javax.persistence.criteria.Expression)
     */
    public Expression<String> toString(Expression<Character> expr)
    {
        return expr.as(String.class);
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#concat(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<String> concat(Expression<String> expr0, Expression<String> expr1)
    {
        ExpressionImpl<String> concatExpr = new ExpressionImpl<String>(String.class);
        concatExpr.queryExpr =
            new DyadicExpression(((ExpressionImpl)expr0).queryExpr, 
                org.datanucleus.query.expression.Expression.OP_ADD, ((ExpressionImpl)expr1).queryExpr);
        return concatExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#concat(javax.persistence.criteria.Expression, java.lang.String)
     */
    public Expression<String> concat(Expression<String> expr, String val)
    {
        ExpressionImpl<String> concatExpr = new ExpressionImpl<String>(String.class);
        Literal lit = new Literal(val);
        concatExpr.queryExpr =
            new DyadicExpression(((ExpressionImpl)expr).queryExpr, org.datanucleus.query.expression.Expression.OP_ADD, lit);
        return concatExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#concat(java.lang.String, javax.persistence.criteria.Expression)
     */
    public Expression<String> concat(String val, Expression<String> expr)
    {
        ExpressionImpl<String> concatExpr = new ExpressionImpl<String>(String.class);
        Literal lit = new Literal(val);
        concatExpr.queryExpr =
            new DyadicExpression(lit, org.datanucleus.query.expression.Expression.OP_ADD, ((ExpressionImpl)expr).queryExpr);
        return concatExpr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#locate(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<Integer> locate(Expression<String> expr, Expression<String> exprSubstr)
    {
        ExpressionImpl<Integer> select = new ExpressionImpl(Integer.class);
        List args = new ArrayList();
        args.add(((ExpressionImpl)exprSubstr).getQueryExpression());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "indexOf", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#locate(javax.persistence.criteria.Expression, java.lang.String)
     */
    public Expression<Integer> locate(Expression<String> expr, String substr)
    {
        ExpressionImpl<Integer> select = new ExpressionImpl(Integer.class);
        List args = new ArrayList();
        Literal litStr = new Literal(substr);
        args.add(litStr);
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "indexOf", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#locate(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<Integer> locate(Expression<String> expr, Expression<String> exprSubstr, Expression<Integer> exprPos)
    {
        ExpressionImpl<Integer> select = new ExpressionImpl(Integer.class);
        List args = new ArrayList();
        args.add(((ExpressionImpl)exprSubstr).getQueryExpression());
        args.add(((ExpressionImpl)exprPos).getQueryExpression());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "indexOf", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#locate(javax.persistence.criteria.Expression, java.lang.String, int)
     */
    public Expression<Integer> locate(Expression<String> expr, String substr, int pos)
    {
        ExpressionImpl<Integer> select = new ExpressionImpl(Integer.class);
        List args = new ArrayList();
        Literal litStr = new Literal(substr);
        args.add(litStr);
        Literal litPos = new Literal(pos);
        args.add(litPos);
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "indexOf", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#substring(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<String> substring(Expression<String> expr, Expression<Integer> posExpr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)posExpr).getQueryExpression());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "substring", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#substring(javax.persistence.criteria.Expression, int)
     */
    public Expression<String> substring(Expression<String> expr, int pos)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        Literal lit = new Literal(pos);
        args.add(lit);
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "substring", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#substring(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<String> substring(Expression<String> expr, Expression<Integer> posExpr0, Expression<Integer> posExpr1)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        args.add(((ExpressionImpl)posExpr0).getQueryExpression());
        args.add(((ExpressionImpl)posExpr1).getQueryExpression());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "substring", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#substring(javax.persistence.criteria.Expression, int, int)
     */
    public Expression<String> substring(Expression<String> expr, int pos0, int pos1)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        List args = new ArrayList();
        Literal litStart = new Literal(pos0);
        Literal litEnd = new Literal(pos1);
        args.add(litStart);
        args.add(litEnd);
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "substring", args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#trim(javax.persistence.criteria.Expression)
     */
    public Expression<String> trim(Expression<String> expr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "trim", null);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#trim(javax.persistence.criteria.QueryBuilder.Trimspec, javax.persistence.criteria.Expression)
     */
    public Expression<String> trim(Trimspec spec, Expression<String> expr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        String method = "trim";
        if (spec == Trimspec.LEADING)
        {
            method = "trimLeft";
        }
        else if (spec == Trimspec.TRAILING)
        {
            method = "trimRight";
        }
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), method, null);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#trim(javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<String> trim(Expression<Character> chr, Expression<String> expr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        String method = "trim";
        List args = new ArrayList();
        args.add(((ExpressionImpl)chr).getQueryExpression());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), method, args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#trim(char, javax.persistence.criteria.Expression)
     */
    public Expression<String> trim(char chr, Expression<String> expr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        String method = "trim";
        List args = new ArrayList();
        args.add(new Literal(chr));
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), method, args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#trim(javax.persistence.criteria.QueryBuilder.Trimspec, javax.persistence.criteria.Expression, javax.persistence.criteria.Expression)
     */
    public Expression<String> trim(Trimspec spec, Expression<Character> chr, Expression<String> expr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        String method = "trim";
        if (spec == Trimspec.LEADING)
        {
            method = "trimLeft";
        }
        else if (spec == Trimspec.TRAILING)
        {
            method = "trimRight";
        }
        List args = new ArrayList();
        args.add(((ExpressionImpl)chr).getQueryExpression());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), method, args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#trim(javax.persistence.criteria.QueryBuilder.Trimspec, char, javax.persistence.criteria.Expression)
     */
    public Expression<String> trim(Trimspec spec, char chr, Expression<String> expr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        String method = "trim";
        if (spec == Trimspec.LEADING)
        {
            method = "trimLeft";
        }
        else if (spec == Trimspec.TRAILING)
        {
            method = "trimRight";
        }
        List args = new ArrayList();
        args.add(new Literal(chr));
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), method, args);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#lower(javax.persistence.criteria.Expression)
     */
    public Expression<String> lower(Expression<String> expr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "toLowerCase", null);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#upper(javax.persistence.criteria.Expression)
     */
    public Expression<String> upper(Expression<String> expr)
    {
        ExpressionImpl<String> select = new ExpressionImpl(expr.getJavaType());
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "toUpperCase", null);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#length(javax.persistence.criteria.Expression)
     */
    public Expression<Integer> length(Expression<String> expr)
    {
        ExpressionImpl<Integer> select = new ExpressionImpl(Integer.class);
        select.queryExpr = new InvokeExpression(((ExpressionImpl)expr).getQueryExpression(), "length", null);
        return select;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#literal(java.lang.Object)
     */
    public <T> Expression<T> literal(T obj)
    {
        ExpressionImpl expr = new ExpressionImpl<T>((Class<T>) obj.getClass());
        Literal lit = new Literal(obj);
        expr.queryExpr = lit;
        return expr;
    }

    /* (non-Javadoc)
     * @see javax.persistence.criteria.QueryBuilder#nullLiteral(java.lang.Class)
     */
    public <T> Expression<T> nullLiteral(Class<T> cls)
    {
        ExpressionImpl expr = new ExpressionImpl<T>(cls);
        Literal lit = new Literal(null);
        expr.queryExpr = lit;
        return expr;
    }
}
