/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.enhancer.asm;

import org.datanucleus.enhancer.ClassEnhancer;
import org.datanucleus.enhancer.ClassMethod;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Type;

/**
 * Representation of a method required by an enhanced class, for use by ASM.
 * @version $Revision: 1.6 $
 */
public abstract class ASMClassMethod extends ClassMethod
{
    /** Visitor for use in updating the method of the class (set in initialise). */
    protected MethodVisitor visitor;

    /**
     * Constructor.
     * @param enhancer Class Enhancer
     * @param name method name
     * @param access Access type for the method
     * @param returnType return type
     * @param argTypes argment types
     * @param argNames argment names
     */
    public ASMClassMethod(ClassEnhancer enhancer, String name, int access, Object returnType, 
            Object[] argTypes, String[] argNames)
    {
        super(enhancer, name, access, returnType, argTypes, argNames);
    }

    /**
     * Constructor.
     * @param enhancer Class Enhancer
     * @param name method name
     * @param access Access type for the method
     * @param returnType return type
     * @param argTypes argment types
     * @param argNames argment names
     * @param exceptions Exceptions that can be thrown
     */
    public ASMClassMethod(ClassEnhancer enhancer, String name, int access, Object returnType, 
            Object[] argTypes, String[] argNames, String[] exceptions)
    {
        super(enhancer, name, access, returnType, argTypes, argNames, exceptions);
    }

    /**
     * Convenience accessor for the ASMClassEnhancer
     * @return ClassEnhancer for ASM
     */
    protected ASMClassEnhancer getClassEnhancer()
    {
        return (ASMClassEnhancer)enhancer;
    }

    /**
     * Default implementation of initialise, specifying the method based on the ClassMethod info.
     */
    public void initialise()
    {
        // Do nothing. Already set up in initialise(ClassVisitor)
    }

    /**
     * Method to initialise the class method.
     * @param classVisitor Visitor for the class
     */
    public void initialise(ClassVisitor classVisitor)
    {
        // Add the method to the class using the ClassMethod info
        Type type = null;
        Type[] argtypes = null;
        if (returnType != null)
        {
            type = Type.getType((Class)returnType);
        }
        else
        {
            type = Type.VOID_TYPE;
        }
        if (argTypes != null)
        {
            argtypes = new Type[argTypes.length];
            for (int i=0;i<argTypes.length;i++)
            {
                argtypes[i] = Type.getType((Class)argTypes[i]);
            }
        }
        else
        {
            argtypes = new Type[0];
        }
        String methodDesc = Type.getMethodDescriptor(type, argtypes);
        this.visitor = classVisitor.visitMethod(access, methodName, methodDesc, null, exceptions);
    }

    /**
     * Accessor for the descriptor of the method.
     * @return The descriptor
     */
    public String getDescriptor()
    {
        StringBuffer str = new StringBuffer("(");
        if (argTypes != null && argTypes.length > 0)
        {
            for (int i=0;i<argTypes.length;i++)
            {
                str.append(Type.getDescriptor((Class)argTypes[i]));
            }
        }
        str.append(")");
        if (returnType != null)
        {
            str.append(Type.getDescriptor((Class)returnType));
        }
        else
        {
            str.append("V");
        }
        return str.toString();
    }
}