/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.query;

/**
 * JDOQL query helper class providing key information about the language etc.
 */
public class JDOQLQueryHelper
{
    /** Keywords used in single-string JDOQL. Uppercase variants specified here, but we allow the lowercase form. */
    public static final String[] SINGLE_STRING_KEYWORDS = {
            "SELECT", "UNIQUE", "INTO", "FROM", "EXCLUDE", "SUBCLASSES", "WHERE",
            "VARIABLES", "PARAMETERS", "GROUP", "ORDER", "BY", "RANGE"
            };
    /** Keywords in lowercase (we avoid calling toLowerCase() multiple times, which is expensive operation) **/
    public static final String[] SINGLE_STRING_KEYWORDS_LOWERCASE = {
            "select", "unique", "into", "from", "exclude", "subclasses", "where",
            "variables", "parameters", "group", "order", "by", "range"
            };

    /**
     * Convenience method returning if the supplied name is a keyword for this query language.
     * @param name Name to check
     * @return Whether it is a keyword
     */
    public static boolean isKeyword(String name)
    {
        for (int i=0;i<SINGLE_STRING_KEYWORDS.length;i++)
        {
            // JDOQL is case-sensitive - lowercase or UPPERCASE only
            if (name.equals(SINGLE_STRING_KEYWORDS[i]) || 
                name.equals(SINGLE_STRING_KEYWORDS_LOWERCASE[i]))
            {
                return true;
            }
        }
        if (name.equals("IMPORT") || name.equals("import"))
        {
            return true;
        }
        return false;
    }

    /**
     * Utility to check if a name is a valid Java identifier.
     * Used by JDOQL in validating the names of parameters/variables.
     * @param s The name
     * @return Whether it is a valid identifier in Java.
     **/
    public static boolean isValidJavaIdentifierForJDOQL(String s)
    {
        int len = s.length();
        if (len < 1)
        {
            return false;
        }

        if (s.equals("this"))
        {
            // Use of "this" is restricted in JDOQL
            return false;
        }

        char[] c = new char[len];
        s.getChars(0, len, c, 0);

        if (!Character.isJavaIdentifierStart(c[0]))
        {
            return false;
        }

        for (int i = 1; i < len; ++i)
        {
            if (!Character.isJavaIdentifierPart(c[i]))
            {
                return false;
            }
        }

        return true;
    }
}