/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of a Character literal in a Query.
 *
 * @version $Revision: 1.11 $
 **/
public class CharacterLiteral extends CharacterExpression implements Literal
{
    private final String value;

    /** Raw value that this literal represents. */
    Object rawValue;

    /**
     * Creates a char literal
     * @param qs the QueryExpression
     * @param mapping the mapping
     * @param value the char value
     */    
    public CharacterLiteral(QueryExpression qs, JavaTypeMapping mapping, char value)
    {
        super(qs);

        this.mapping = mapping;
        this.value = String.valueOf(value);
        st.append('\'').append(value).append('\'');
    }

    /**
     * Creates a char literal
     * @param qs the QueryExpression
     * @param mapping the mapping
     * @param value the char value
     */
    public CharacterLiteral(QueryExpression qs, JavaTypeMapping mapping, String value)
    {
        super(qs);

        this.mapping = mapping;
        this.value = value;
        st.appendParameter(mapping, value);
    }

    public Object getValue()
    {
        if( value == null )
        {
            return null;
        }
        return new Character(value.charAt(0));
    }
    
    public BooleanExpression eq(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.equals(((CharacterLiteral)expr).value));
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            return new BooleanLiteral(qs, mapping, !value.equals(((CharacterLiteral)expr).value));
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public BooleanExpression lt(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((CharacterLiteral)expr).value) < 0);
        }
        else
        {
            return super.lt(expr);
        }
    }

    public BooleanExpression lteq(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((CharacterLiteral)expr).value) <= 0);
        }
        else
        {
            return super.lteq(expr);
        }
    }

    public BooleanExpression gt(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((CharacterLiteral)expr).value) > 0);
        }
        else
        {
            return super.gt(expr);
        }
    }

    public BooleanExpression gteq(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            return new BooleanLiteral(qs, mapping, value.compareTo(((CharacterLiteral)expr).value) >= 0);
        }
        else
        {
            return super.gteq(expr);
        }
    }

    public ScalarExpression add(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            int v = value.charAt(0)+((CharacterLiteral)expr).value.charAt(0);
            return new IntegerLiteral(qs, mapping, new Integer(v));
        }
        else if (expr instanceof IntegerLiteral)
        {
            int v = value.charAt(0)+((Number)((IntegerLiteral)expr).getValue()).intValue();
            return new IntegerLiteral(qs, mapping, new Integer(v));
        }
        else
        {
            return super.add(expr);
        }
    }
    
    public ScalarExpression sub(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            int v = value.charAt(0)-((CharacterLiteral)expr).value.charAt(0);
            return new IntegerLiteral(qs, mapping, new Integer(v));
        }
        else if (expr instanceof IntegerLiteral)
        {
            int v = value.charAt(0)-((Number)((IntegerLiteral)expr).getValue()).intValue();
            return new IntegerLiteral(qs, mapping, new Integer(v));
        }
        else
        {
            return super.add(expr);
        }
    }    

    public ScalarExpression mod(ScalarExpression expr)
    {
        if (expr instanceof CharacterLiteral)
        {
            int v = value.charAt(0)%((CharacterLiteral)expr).value.charAt(0);
            return new IntegerLiteral(qs, mapping, new Integer(v));
        }
        else if (expr instanceof IntegerLiteral)
        {
            int v = value.charAt(0)%((Number)((IntegerLiteral)expr).getValue()).intValue();
            return new IntegerLiteral(qs, mapping, new Integer(v));
        }       
        else
        {
            return super.mod(expr);
        }
    }
    
    /**
     * Converts the Character Literal to lower case
     * @return the lowercased Character Literal
     */    
    public CharacterExpression toLowerCaseMethod()
    {
        return new CharacterLiteral(qs, mapping, value.toLowerCase());
    }

    /**
     * Converts the Character Literal to upper case
     * @return the uppercased Character Literal
     */    
    public CharacterExpression toUpperCaseMethod()
    {
        return new CharacterLiteral(qs, mapping, value.toUpperCase());
    }

    public ScalarExpression neg()
    {
        int v = -(value.charAt(0));
        return new IntegerLiteral(qs, mapping, new Integer(v));
    }    

    public ScalarExpression com()
    {
        int v = ~(value.charAt(0));
        return new IntegerLiteral(qs, mapping, new Integer(v));
    }

    /**
     * Method to save a "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @param val The raw value
     */
    public void setRawValue(Object val)
    {
        this.rawValue = val;
    }

    /**
     * Accessor for the "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @return The raw value
     */
    public Object getRawValue()
    {
        return rawValue;
    }
}