/**********************************************************************
Copyright (c) 2002 Kelly Grizzle (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2003 Andy Jefferson - coding standards
2005 Andy Jefferson - basic serialisation support
2005 Andy Jefferson - updated serialisation using SCOUtils methods
2005 Andy Jefferson - changed to allow the store to be set or list
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.util.Collection;

import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.sco.SCO;
import org.datanucleus.sco.SCOCollection;
import org.datanucleus.sco.SCOContainer;
import org.datanucleus.sco.SCOUtils;
import org.datanucleus.store.exceptions.ReachableObjectNotCascadedException;
import org.datanucleus.store.mapped.expression.CollectionExpression;
import org.datanucleus.store.mapped.expression.CollectionLiteral;
import org.datanucleus.store.mapped.expression.CollectionSubqueryExpression;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.Queryable;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.scostore.CollectionStore;
import org.datanucleus.util.NucleusLogger;

/**
 * Mapping for Collection/Set/List types.
 */
public class CollectionMapping extends AbstractContainerMapping implements MappingCallbacks
{
    /**
     * Equality operator.
     * @param obj Object to compare against
     * @return Whether they are equal
     */
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }

        if (!obj.getClass().equals(getClass()))
        {
            return false;
        }

        CollectionMapping sm = (CollectionMapping)obj;

        return mmd.equals(sm.mmd) && storeMgr.equals(sm.storeMgr);
    }

    /**
     * Accessor for the Java type represented here.
     * @return The java type
     */
    public Class getJavaType()
    {
        return Collection.class;
    }

    // --------------- Implementation of MappingCallbacks -------------------

    /**
     * Method to be called after the insert of the owner class element.
     * @param sm StateManager of the owner
     */
    public void postInsert(StateManager sm)
    {
        Collection value = (Collection) sm.provideField(mmd.getAbsoluteFieldNumber());
        if (containerIsStoredInSingleColumn())
        {
            // Make sure the elements are ok for proceeding
            SCOUtils.validateObjectsForWriting(sm.getObjectManager(), value);
            return;
        }

        if (value == null)
        {
            // replace null collections with an empty SCO wrapper
            replaceFieldWithWrapper(sm, null, false, false);
            return;
        }

        Object[] collElements = value.toArray();
        if (!mmd.isCascadePersist())
        {
            // Field doesnt support cascade-persist so no reachability
            if (NucleusLogger.REACHABILITY.isDebugEnabled())
            {
                NucleusLogger.REACHABILITY.debug(LOCALISER.msg("007006", mmd.getFullFieldName()));
            }

            // Check for any persistable elements that arent persistent
            for (int i=0;i<collElements.length;i++)
            {
                if (!sm.getObjectManager().getApiAdapter().isDetached(collElements[i]) &&
                    !sm.getObjectManager().getApiAdapter().isPersistent(collElements[i]))
                {
                    // Element is not persistent so throw exception
                    throw new ReachableObjectNotCascadedException(mmd.getFullFieldName(), collElements[i]);
                }
            }
        }
        else
        {
            // Reachability
            if (NucleusLogger.REACHABILITY.isDebugEnabled())
            {
                NucleusLogger.REACHABILITY.debug(LOCALISER.msg("007007", mmd.getFullFieldName()));
            }

            // Check if some elements need attaching
            // TODO Investigate if we can just use the attachCopy route below and skip off this check
            boolean needsAttaching = false;
            for (int i=0;i<collElements.length;i++)
            {
                if (sm.getObjectManager().getApiAdapter().isDetached(collElements[i]))
                {
                    needsAttaching = true;
                    break;
                }
            }

            if (needsAttaching)
            {
                // Create a wrapper and attach the elements (and add the others)
                SCO collWrapper = replaceFieldWithWrapper(sm, null, false, false);
                collWrapper.attachCopy(value);
            }
            else
            {
                if (value.size() > 0)
                {
                    // Add the elements direct to the datastore
                    ((CollectionStore) storeMgr.getBackingStoreForField(sm.getObjectManager().getClassLoaderResolver(),mmd, value.getClass())).addAll(sm, value, 0);

                    // Create a SCO wrapper with the elements loaded
                    replaceFieldWithWrapper(sm, value, false, false);
                }
                else
                {
                    // Create a SCO wrapper
                    replaceFieldWithWrapper(sm, null, false, false);
                }
            }
        }
    }

    /**
     * Method to be called after any update of the owner class element.
     * This method could be called in two situations
     * <ul>
     * <li>Update a collection field of an object by replacing the collection with a new collection, 
     * so UpdateRequest is called, which calls here</li>
     * <li>Persist a new object, and it needed to wait til the element was inserted so
     * goes into dirty state and then flush() triggers UpdateRequest, which comes here</li>
     * </ul>
     * @param sm StateManager of the owner
     */
    public void postUpdate(StateManager sm)
    {
        ObjectManager om = sm.getObjectManager();
        Collection value = (Collection) sm.provideField(mmd.getAbsoluteFieldNumber());
        if (containerIsStoredInSingleColumn())
        {
            // Make sure the elements are ok for proceeding
            SCOUtils.validateObjectsForWriting(om, value);
            return;
        }

        if (value == null)
        {
            // remove any elements in the collection and replace it with an empty SCO wrapper
            ((CollectionStore) storeMgr.getBackingStoreForField(om.getClassLoaderResolver(), mmd, null)).clear(sm);
            replaceFieldWithWrapper(sm, null, false, false);
            return;
        }

        if (value instanceof SCOContainer)
        {
            // Already have a SCO value
            SCOContainer sco = (SCOContainer) value;
            if (sm.getObject() == sco.getOwner() && fieldName.equals(sco.getFieldName()))
            {
                // Flush any outstanding updates
                sco.flush();

                return;
            }

            if (sco.getOwner() != null)
            {
                throw new NucleusException(LOCALISER.msg("CollectionMapping.WrongOwnerError")).setFatal();
            }
        }

        if (!mmd.isCascadeUpdate())
        {
            // TODO Should this throw exception if the element doesn't exist?
            // User doesn't want to update by reachability
            if (NucleusLogger.REACHABILITY.isDebugEnabled())
            {
                NucleusLogger.REACHABILITY.debug(LOCALISER.msg("007008", mmd.getFullFieldName()));
            }
            return;
        }
        if (NucleusLogger.REACHABILITY.isDebugEnabled())
        {
            NucleusLogger.REACHABILITY.debug(LOCALISER.msg("007009", mmd.getFullFieldName()));
        }

        CollectionStore backingStore = ((CollectionStore) storeMgr.getBackingStoreForField(
            om.getClassLoaderResolver(), mmd, value.getClass()));
        backingStore.update(sm, value);

        // Replace the field with a wrapper containing these elements
        replaceFieldWithWrapper(sm, value, false, false);
    }

    /**
     * Method to be called before any delete of the owner class element.
     * @param sm StateManager of the owner
     */
    public void preDelete(StateManager sm)
    {
        if (containerIsStoredInSingleColumn())
        {
            // Field is stored with the main object so nothing to clean up
            return;
        }

        // makes sure field is loaded
        sm.getObjectManager().getApiAdapter().isLoaded(sm, mmd.getAbsoluteFieldNumber());
        Collection value = (Collection) sm.provideField(mmd.getAbsoluteFieldNumber());
        if (value == null)
        {
            return;
        }

        boolean isDependentElement = mmd.getCollection().isDependentElement();
        boolean hasJoin = (mmd.getJoinMetaData() != null);
        boolean hasFK = false;
        if (!hasJoin)
        {
            if (mmd.getElementMetaData() != null && mmd.getElementMetaData().getForeignKeyMetaData() != null)
            {
                // FK collection, using <element> FK spec
                hasFK = true;
            }
            else if (mmd.getForeignKeyMetaData() != null)
            {
                // FK collection, using <field> FK spec
                hasFK = true;
            }
            AbstractMemberMetaData[] relatedMmds = mmd.getRelatedMemberMetaData(sm.getObjectManager().getClassLoaderResolver());
            if (relatedMmds != null && relatedMmds[0].getForeignKeyMetaData() != null)
            {
                // FK collection (bidir), using <field> FK spec at other end
                hasFK = true;
            }
        }
        if (sm.getObjectManager().getOMFContext().getPersistenceConfiguration().getStringProperty("datanucleus.deletionPolicy").equals("JDO2"))
        {
            // JDO2 doesnt currently (2.0 spec) take note of foreign-key
            hasFK = false;
        }

        // TODO Why dont we just do clear here always ? THe backing store should take care of if nulling or deleting etc
        if (isDependentElement || hasJoin || !hasFK)
        {
            // Elements are either dependent (in which case we need to delete them) or theres a join (in which case
            // we need to remove the join entries), or there are no FKs specified (in which case we need to clean up)
            if (!(value instanceof SCO))
            {
                value = (Collection)sm.wrapSCOField(mmd.getAbsoluteFieldNumber(), value, false, false, true);
            }
            value.clear();
            ((SCOCollection)value).flush();
        }
    }

    // ------------------------------- JDOQL Query Methods --------------------------------------

    /**
     * Accessor for a literal representing this type.
     * @param qs The Query
     * @param value the value of this object in the literal
     * @return The literal
     */
    public ScalarExpression newLiteral(QueryExpression qs, Object value)
    {
        if (containerIsStoredInSingleColumn())
        {
            throw new NucleusUserException(LOCALISER.msg("041025", mmd.getFullFieldName())).setFatal();
        }
        if (value instanceof Queryable)
        {
            return new CollectionSubqueryExpression(qs, ((Queryable)value).newQueryStatement());
        }
        else
        {
            return new CollectionLiteral(qs, this, (Collection)value);
        }
    }

    /**
     * Accessor for a scalar expression involving this object.
     * @param qs The Query
     * @param te The table holding this object.
     * @return The expression
     */
    public ScalarExpression newScalarExpression(QueryExpression qs, LogicSetExpression te)
    {
        if (containerIsStoredInSingleColumn())
        {
            throw new NucleusUserException(LOCALISER.msg("041025", mmd.getFullFieldName())).setFatal();
        }
        return new CollectionExpression(qs, datastoreContainer.getIDMapping(), te, ((CollectionStore) storeMgr.getBackingStoreForField(qs.getClassLoaderResolver(),mmd,null)), fieldName);
    }
}