/**********************************************************************
Copyright (c) 2002 Kelly Grizzle and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2002 Mike Martin - unknown changes
2003 Erik Bengtson - removed exist() operation
2004 Andy Jefferson - added getHighestFieldNumber()
2005 Andy Jefferson - javadocs
2007 Xuan Baldauf - Contrib of notifyMadePersistentClean()
2007 Xuan Baldauf - Contrib of internalXXXFieldXXX() methods
2008 Andy Jefferson - removed all deps on org.datanucleus.store.mapped
2011 Andy Jefferson - removed many methods added during merge with ObjectProvider
    ...
**********************************************************************/
package org.datanucleus.state;

import org.datanucleus.cache.CachedPC;
import org.datanucleus.store.FieldValues;
import org.datanucleus.store.ObjectProvider;

/**
 * StateManager for a persistable object. Based around the JDO PersistenceCapable contract.
 * Makes the assumption that a StateManager corresponds to ONE persistable object.
 */
public interface StateManager extends ObjectProvider
{
    /**
     * Initialises a state manager to manage a hollow instance having the given object ID and the given
     * (optional) field values. This constructor is used for creating new instances of existing persistent
     * objects, and consequently shouldnt be used when the StoreManager controls the creation of such objects
     * (such as in an ODBMS).
     * @param id the identity of the object.
     * @param fv the initial field values of the object (optional)
     * @param pcClass Class of the object that this will manage the state for
     */
    void initialiseForHollow(Object id, FieldValues fv, Class pcClass);

    /**
     * Initialises a state manager to manage a HOLLOW / P_CLEAN instance having the given FieldValues.
     * This constructor is used for creating new instances of existing persistent objects using application 
     * identity, and consequently shouldnt be used when the StoreManager controls the creation of such objects
     * (such as in an ODBMS).
     * @param fv the initial field values of the object.
     * @param pcClass Class of the object that this will manage the state for
     */
    void initialiseForHollowAppId(FieldValues fv, Class pcClass);

    /**
     * Initialises a state manager to manage the given hollow instance having the given object ID.
     * Unlike the {@link #initialiseForHollow} method, this method does not create a new instance and instead 
     * takes a pre-constructed instance.
     * @param id the identity of the object.
     * @param pc the object to be managed.
     */
    void initialiseForHollowPreConstructed(Object id, Object pc);

    /**
     * Initialises a state manager to manage the passed persistent instance having the given object ID.
     * Used where we have retrieved a PC object from a datastore directly (not field-by-field), for example on
     * an object datastore. This initialiser will not add StateManagers to all related PCs. This must be done by
     * any calling process. This simply adds the StateManager to the specified object and records the id, setting
     * all fields of the object as loaded.
     * @param id the identity of the object.
     * @param pc The object to be managed
     */
    void initialiseForPersistentClean(Object id, Object pc);

    /**
     * Initialises a state manager to manage a PersistenceCapable instance that will be EMBEDDED/SERIALISED 
     * into another PersistenceCapable object. The instance will not be assigned an identity in the process 
     * since it is a SCO.
     * @param pc The PersistenceCapable to manage (see copyPc also)
     * @param copyPc Whether the SM should manage a copy of the passed PC or that one
     */
    void initialiseForEmbedded(Object pc, boolean copyPc);

    /**
     * Initialises a state manager to manage a transient instance that is becoming newly persistent.
     * A new object ID for the instance is obtained from the store manager and the object is inserted
     * in the data store.
     * <p>This constructor is used for assigning state managers to existing
     * instances that are transitioning to a persistent state.
     * @param pc the instance being make persistent.
     * @param preInsertChanges Any changes to make before inserting
     */
    void initialiseForPersistentNew(Object pc, FieldValues preInsertChanges);

    /**
     * Initialises a state manager to manage a Transactional Transient instance.
     * A new object ID for the instance is obtained from the store manager and the object is inserted in the data store.
     * <p>
     * This constructor is used for assigning state managers to Transient
     * instances that are transitioning to a transient clean state.
     * @param pc the instance being make persistent.
     */
    void initialiseForTransactionalTransient(Object pc);

    /**
     * Initialises the StateManager to manage a PersistenceCapable object in detached state.
     * @param pc the detach object.
     * @param id the identity of the object.
     * @param version the detached version
     */
    void initialiseForDetached(Object pc, Object id, Object version);

    /**
     * Initialise to create a StateManager for a PersistenceCapable object, assigning the specified id to the object. 
     * This is used when getting objects out of the L2 Cache, where they have no StateManager assigned, and returning 
     * them as associated with a particular PM.
     * @param cachedPC Cache object for persistable object from L2 cache
     * @param id Id to assign to the PersistenceCapable object
     * @param pcClass Class of the object that this will manage the state for
     */
    void initialiseForCachedPC(CachedPC cachedPC, Object id, Class pcClass);

    /**
     * Initialises the StateManager to manage a PersistenceCapable object that is not persistent but that
     * is about to be deleted. The initial state will be P_NEW, but when the delete call comes in will be
     * P_NEW_DELETED. The object will not be enlisted in the transaction.
     * @param pc the object to delete
     */
    void initialiseForPNewToBeDeleted(Object pc);

    /**
     * Look to the database to determine which
     * class this object is. This parameter is a hint. Set false, if it's
     * already determined the correct pcClass for this pc "object" in a certain
     * level in the hierarchy. Set to true and it will look to the database.
     * @param fv the initial field values of the object.
     */
    void checkInheritance(FieldValues fv);
}