/**********************************************************************
Copyright (c) 2002 Kelly Grizzle and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2002 Mike Martin - unknown changes
2003 Andy Jefferson - commented
2003 Erik Bengtson - optimistic transaction
2008 Andy Jefferson - fixed transitionReadField, transitionWriteField exceptions
    ...
**********************************************************************/
package org.datanucleus.jdo.state;

import org.datanucleus.FetchPlan;
import org.datanucleus.StateManager;
import org.datanucleus.Transaction;
import org.datanucleus.jdo.exceptions.TransactionNotReadableException;
import org.datanucleus.jdo.exceptions.TransactionNotWritableException;
import org.datanucleus.state.IllegalStateTransitionException;
import org.datanucleus.state.LifeCycleState;

/**
 * Class representing the life cycle state of PersistentNontransactional.
 */
class PersistentNontransactional extends LifeCycleState
{
    /** Protected Constructor to prevent external instantiation. */
    protected PersistentNontransactional()
    {
		isPersistent = true;    	
        isDirty = false;
        isNew = false;
        isDeleted = false;
        isTransactional = false;

        stateType = P_NONTRANS;
    }

    /**
     * Method to transition to delete persistent.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionDeletePersistent(StateManager sm)
    {
        sm.clearLoadedFlags();
        return changeState(sm, P_DELETED);
    }

    /**
     * Method to transition to transactional.
     * @param sm StateManager.
     * @param refreshFields Whether to refresh loaded fields
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionMakeTransactional(StateManager sm, boolean refreshFields)
    {
        if (refreshFields)
        {
            sm.refreshLoadedFields();
        }
        return changeState(sm, P_CLEAN);
    }

    /**
     * Method to transition to transient.
     * @param sm StateManager.
     * @param useFetchPlan to make transient the fields in the fetch plan
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionMakeTransient(StateManager sm, boolean useFetchPlan, boolean detachAllOnCommit)
    {
        if (useFetchPlan)
        {
            sm.loadUnloadedFieldsInFetchPlan();
        }
        return changeState(sm, TRANSIENT);
    }

    /**
     * Method to transition to commit state.
     * @param sm StateManager.
     * @param tx the Transaction been committed.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionCommit(StateManager sm, Transaction tx)
    {
        throw new IllegalStateTransitionException(this, "commit", sm);
    }

    /**
     * Method to transition to rollback state.
     * @param sm StateManager.
     * @param tx The transaction
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRollback(StateManager sm,Transaction tx)
    {
        throw new IllegalStateTransitionException(this, "rollback", sm);
    }

    /**
     * Method to transition to refresh state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRefresh(StateManager sm)
    {
        // Refresh the FetchPlan fields and unload all others
        sm.refreshFieldsInFetchPlan();
        sm.unloadNonFetchPlanFields();

        return this;
    }
    
    /**
     * Method to transition to evict state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionEvict(StateManager sm)
    {
        sm.clearNonPrimaryKeyFields();
        sm.clearSavedFields();
        return changeState(sm, HOLLOW);
    }

    /**
     * Method to transition to read-field state.
     * @param sm StateManager.
     * @param isLoaded if the field was previously loaded.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionReadField(StateManager sm, boolean isLoaded)
    {
        Transaction tx = sm.getObjectManager().getTransaction();
		if (!tx.isActive() && !tx.getNontransactionalRead())
		{
	        throw new TransactionNotReadableException(LOCALISER.msg("027002"), sm.getInternalObjectId());
		}
        if (tx.isActive() && !tx.getOptimistic())
        {
            // Save the fields for rollback.
            sm.saveFields();
            sm.refreshLoadedFields();
            return changeState(sm, P_CLEAN);
        }
        else
        {
            return this;
        }
    }

    /**
     * Method to transition to write-field state.
     * @param sm StateManager.
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionWriteField(StateManager sm)
    {
        Transaction tx = sm.getObjectManager().getTransaction();
        if (!tx.isActive() && !tx.getNontransactionalWrite())
        {
            throw new TransactionNotWritableException(LOCALISER.msg("027001"), sm.getInternalObjectId());
        }
        if (tx.isActive())
        {
            // Save the fields for rollback.
            sm.saveFields();
            return changeState(sm, P_DIRTY);
        }
        else
        {
            // Save the fields for rollback.
            sm.saveFields();
            return changeState(sm, P_NONTRANS_DIRTY);
        }
    }

    /**
     * Method to transition to retrieve state.
     * @param sm StateManager.
	 * @param fgOnly only the current fetch group fields
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRetrieve(StateManager sm, boolean fgOnly)
    {
        Transaction tx = sm.getObjectManager().getTransaction();
        if (tx.isActive() && !tx.getOptimistic())
        {
            // Save the fields for rollback.
            sm.saveFields();
    		if (fgOnly)
            {
                sm.loadUnloadedFieldsInFetchPlan();
            }
    		else
            {
    			sm.loadUnloadedFields();
            }             
            return changeState(sm, P_CLEAN);
        }
        else if (tx.isActive() && tx.getOptimistic())
        {
            // Save the fields for rollback.
            sm.saveFields(); //TODO this is wrong... saving all the time, retrieve is asked... side effects besides performance?
    		if (fgOnly)
            {
                sm.loadUnloadedFieldsInFetchPlan();
            }
    		else
            {
    			sm.loadUnloadedFields();
            }
    		return this;
        }
        else
        {
    		if (fgOnly)
            {
                sm.loadUnloadedFieldsInFetchPlan();
            }
    		else
            {
    			sm.loadUnloadedFields();
            }
    		return this;
        }
    }

    /**
     * Method to transition to retrieve state.
     * @param sm StateManager.
     * @param fetchPlan the fetch plan to load fields
     * @return new LifeCycle state.
     **/
    public LifeCycleState transitionRetrieve(StateManager sm, FetchPlan fetchPlan)
    {
        Transaction tx = sm.getObjectManager().getTransaction();
        if (tx.isActive() && !tx.getOptimistic())
        {
            // Save the fields for rollback.
            sm.saveFields();
            sm.loadUnloadedFieldsOfClassInFetchPlan(fetchPlan);
            return changeState(sm, P_CLEAN);
        }
        else if (tx.isActive() && tx.getOptimistic())
        {
            // Save the fields for rollback.
            sm.saveFields(); //TODO this is wrong... saving all the time, retrieve is asked... side effects besides performance?
            sm.loadUnloadedFieldsOfClassInFetchPlan(fetchPlan);
            return this;
        }
        else
        {
            sm.loadUnloadedFieldsOfClassInFetchPlan(fetchPlan);
            return this;
        }
    }

    /**
     * Method to transition when serialised.
     * @param sm State Manager
     * @return The new LifeCycle state
     */
    public LifeCycleState transitionSerialize(StateManager sm)
    {
        Transaction tx = sm.getObjectManager().getTransaction();
        if (tx.isActive() && !tx.getOptimistic())
        {
            return changeState(sm, P_CLEAN);
        }
        return this;
    }

    /**
     * Method to transition to detached-clean.
     * @param sm StateManager.
     * @return new LifeCycle state.
     */
    public LifeCycleState transitionDetach(StateManager sm)
    {
        return changeState(sm, DETACHED_CLEAN);
    }

    /**
     * Method to return a string version of this object.
     * @return The string "P_NONTRANS".
     **/
    public String toString()
    {
        return "P_NONTRANS";
    }
}