/**********************************************************************
Copyright (c) 2006 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.lang.reflect.Array;

import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of array expression. Either backed by ScalarExpressions, or by an array field.
 */
public class ArrayLiteral extends ArrayExpression implements Literal
{
    /** value of the array **/
    Object value;

    /**
     * Constructor
     * @param qs The query statement
     */
    protected ArrayLiteral(QueryExpression qs)
    {
        super(qs);
    }

    /**
     * Constructor
     * @param qs The query statement
     * @param expr the array of expressions
     */
    public ArrayLiteral(QueryExpression qs, ScalarExpression[] expr)
    {
        super(qs);
        this.exprs = expr;
    }
    
    /**
     * Constructor
     * @param qs The query statement
     * @param mapping the {@link JavaTypeMapping}
     * @param literal the array value
     */
    public ArrayLiteral(QueryExpression qs, JavaTypeMapping mapping, Object literal)
    {
        super(qs);
        this.mapping = mapping;
        this.value = literal;
        if (value == null || !value.getClass().isArray())
        {
            throw new NucleusUserException("Invalid argument literal : " + value);
        }

        this.exprs = new ScalarExpression[Array.getLength(value)];
        for( int i=0; i<Array.getLength(value); i++ )
        {
            JavaTypeMapping m = qs.getStoreManager().getMappingManager().getMappingWithDatastoreMapping(
                Array.get(value, i).getClass(), false, false, qs.getClassLoaderResolver());
            this.exprs[i] = m.newLiteral(qs, Array.get(value, i));
        }
    }    

    public Object getValue()
    {
        return value;
    }

    /**
     * Method to save a "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @param val The raw value
     */
    public void setRawValue(Object val)
    {
        // Dont support raw value for binary literals
    }

    /**
     * Accessor for the "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @return The raw value
     */
    public Object getRawValue()
    {
        return null; // Dont support raw value for binary literals
    }
}