/****************************************************************************
*																			*
*					  cryptlib Generic Crypto HW Routines					*
*						Copyright Peter Gutmann 1998-2019					*
*																			*
****************************************************************************/

#define PKC_CONTEXT		/* Indicate that we're working with PKC contexts */
#if defined( INC_ALL )
  #include "crypt.h"
  #include "context.h"
  #include "device.h"
  #include "hardware.h"
#else
  #include "crypt.h"
  #include "context/context.h"
  #include "device/device.h"
  #include "device/hardware.h"
#endif /* Compiler-specific includes */

#ifdef USE_HARDWARE

/****************************************************************************
*																			*
*						 		Utility Routines							*
*																			*
****************************************************************************/

/* Get a reference to the cryptographic HAL object that underlies a native 
   cryptlib object.  This is used to connect a reference from a PKCS #15 
   storage object to the corresponding HAL object via the hardware storageID 
   that's recorded in the PKCS #15 storage object */

CHECK_RETVAL STDC_NONNULL_ARG( ( 2 ) ) \
static int getHardwareReference( IN_HANDLE const CRYPT_CONTEXT iCryptContext,
								 OUT_INT_Z int *storageRef )
	{
	MESSAGE_DATA msgData;
	BYTE storageID[ KEYID_SIZE + 8 ];
	int status;

	assert( isWritePtr( storageRef, sizeof( int ) ) );

	REQUIRES( isHandleRangeValid( iCryptContext ) );

	/* Clear return value */
	*storageRef = CRYPT_ERROR;

	setMessageData( &msgData, storageID, KEYID_SIZE );
	status = krnlSendMessage( iCryptContext, IMESSAGE_GETATTRIBUTE_S,
							  &msgData, CRYPT_IATTRIBUTE_DEVICESTORAGEID );
	if( cryptStatusOK( status ) )
		status = hwLookupItem( storageID, msgData.length, storageRef );
	if( cryptStatusError( status ) )
		{
		/* In theory this is an internal error but in practice we shouldn't
		   treat this as too fatal, what it really means is that the crypto
		   hardware (which we don't control and therefore can't do too much
		   about) is out of sync with the PKCS #15 storage object.  This can 
		   happen for example during the development process when the 
		   hardware is reinitialised but the storage object isn't, or from
		   any one of a number of other circumstances beyond our control.  
		   To deal with this we return a standard notfound error but also 
		   output a diagnostic message for developers to let them know that
		   they need to check hardware/storage object synchronisation */
		DEBUG_DIAG(( "Object held in PKCS #15 object store doesn't "
					 "correspond to anything known to the crypto HAL" ));
		return( CRYPT_ERROR_NOTFOUND );
		}

	return( CRYPT_OK );
	}

/* Open and close the PKCS #15 storage object associated with this device.  
   This is either mapped to storage inside the hardware device or stored on 
   disk if the device doesn't provide its own storage */

#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
static int getCryptoStorageObject( OUT_HANDLE_OPT CRYPT_KEYSET *iCryptKeyset )
	{
	CRYPT_KEYSET iHWKeyset;
	int status;

	assert( isWritePtr( iCryptKeyset, sizeof( CRYPT_KEYSET ) ) );

	/* Clear return value */
	*iCryptKeyset = CRYPT_ERROR;

	/* Get a reference to the crypto storage object from the crypto hardware
	   device */
	status = krnlSendMessage( CRYPTO_OBJECT_HANDLE, IMESSAGE_GETATTRIBUTE, 
							  &iHWKeyset, CRYPT_IATTRIBUTE_HWSTORAGE );
	if( cryptStatusOK( status ) )
		status = krnlSendNotifier( iHWKeyset, IMESSAGE_INCREFCOUNT );
	if( cryptStatusError( status ) )
		{
		/* Rather than returning some possible low-level permssion error or 
		   similar we report the problem as a CRYPT_ERROR_NOTINITED since 
		   the most likely issue is that the storage object isn't set up for 
		   use */
		return( CRYPT_ERROR_NOTINITED );
		}

	*iCryptKeyset = iHWKeyset;
	return( CRYPT_OK );
	}
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1, 4 ) ) \
static int openStorageObject( OUT_HANDLE_OPT CRYPT_KEYSET *iCryptKeyset,
							  IN_ENUM( CRYPT_KEYOPT ) \
									const CRYPT_KEYOPT_TYPE options,
							  IN_HANDLE const CRYPT_DEVICE iCryptDevice,
							  INOUT ERROR_INFO *errorInfo )
	{
	CRYPT_KEYOPT_TYPE createOptions = CRYPT_KEYOPT_NONE;
	MESSAGE_CREATEOBJECT_INFO createInfo;
	ERROR_INFO localErrorInfo;
	void *storageObjectAddr;
	BOOLEAN isFileKeyset = FALSE;
	int storageObjectSize, status;

	assert( isWritePtr( iCryptKeyset, sizeof( CRYPT_KEYSET ) ) );
	assert( isWritePtr( errorInfo, sizeof( ERROR_INFO ) ) );

	REQUIRES( options == CRYPT_KEYOPT_NONE || \
			  options == CRYPT_KEYOPT_CREATE );
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	REQUIRES( iCryptDevice == CRYPTO_OBJECT_HANDLE || \
			  isHandleRangeValid( iCryptDevice ) );
#else
	REQUIRES( isHandleRangeValid( iCryptDevice ) );
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

	/* Clear return value */
	*iCryptKeyset = CRYPT_ERROR;

	/* If we're working with a crypto HAL then the HAL will have opened the
	   storage object when it was initialised, in which case we don't want
	   to open it again but merely obtain a reference to the existing
	   storage object */
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	if( iCryptDevice != CRYPTO_OBJECT_HANDLE )
		return( getCryptoStorageObject( iCryptKeyset ) );
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

	/* Try and open/create the PKCS #15 storage object.  If the hardware 
	   device provides secure storage for this then we use that, otherwise 
	   we make it a plain file */
	clearErrorString( &localErrorInfo );
	status = hwGetStorage( &storageObjectAddr, &storageObjectSize );
	if( status == OK_SPECIAL )
		{
		/* If the device provides its own storage but this hasn't been 
		   initialised yet, indicated by a return value of OK_SPECIAL, then 
		   we can't open it as a storage object until it's explicitly 
		   initialised.  If the open option is CRYPT_KEYOPT_CREATE then
		   we're expecting to initialise anyway, but if then not we switch 
		   the open option to CRYPT_KEYOPT_CREATE now */
		if( options == CRYPT_KEYOPT_NONE )
			{
			DEBUG_DIAG(( "Built-in device storage is zeroised, cryptlib "
						 "will initialise the storage object" ));
			createOptions = CRYPT_KEYOPT_CREATE;
			}
		status = CRYPT_OK;
		}
	if( cryptStatusOK( status ) )
		{
		setMessageCreateObjectIndirectInfo( &createInfo, storageObjectAddr, 
											storageObjectSize, 
											CRYPT_KEYSET_FILE, 
											&localErrorInfo );
		if( createOptions != CRYPT_KEYOPT_NONE )
			createInfo.arg2 = createOptions;
		status = krnlSendMessage( CRYPTO_OBJECT_HANDLE, 
								  IMESSAGE_DEV_CREATEOBJECT_INDIRECT,
								  &createInfo, OBJECT_TYPE_KEYSET );
		}
	else
		{
		char storageFilePath[ MAX_PATH_LENGTH + 8 ];
		int storageFilePathLen;

		/* There's no in-memory storage provided by the HAL, use an on-disk 
		   file */
		status = fileBuildCryptlibPath( storageFilePath, MAX_PATH_LENGTH, 
										&storageFilePathLen, "CLKEYS", 6, 
										( options == CRYPT_KEYOPT_CREATE ) ? \
										  BUILDPATH_CREATEPATH : \
										  BUILDPATH_GETPATH );
		if( cryptStatusError( status ) )
			return( status );
		isFileKeyset = TRUE;
		setMessageCreateObjectInfo( &createInfo, CRYPT_KEYSET_FILE );
		createInfo.strArg1 = storageFilePath;
		createInfo.strArgLen1 = storageFilePathLen;
		if( options != CRYPT_KEYOPT_NONE )
			createInfo.arg2 = options;
		status = krnlSendMessage( SYSTEM_OBJECT_HANDLE, 
								  IMESSAGE_DEV_CREATEOBJECT,
								  &createInfo, OBJECT_TYPE_KEYSET );
		}
	if( cryptStatusError( status ) )
		{
		retExtErr( status,
				   ( status, errorInfo, &localErrorInfo,
					 "Couldn't open device storage object" ) );
		}

	/* Now that we've got the storage object we have to perform a somewhat 
	   awkward backreference-update of the keyset to give it the handle of 
	   the owning device since we need to create any contexts for keys 
	   fetched from the storage object via the hardware device rather than 
	   the default system device.  In theory we could also do this via a new 
	   get-owning-object message but we still need to signal to the keyset 
	   that it's a storage object rather than a standard keyset so this 
	   action serves a second purpose anyway and we may as well use it to 
	   explicitly set the owning-device handle at the same time.

	   Note that we don't set the storage object as a dependent object of 
	   the device because it's not necessarily constant across device 
	   sessions.  In particular if we initialise or zeroise the device then 
	   the storage object will be reset, but there's no way to switch 
	   dependent objects without destroying and recreating the parent.  In
	   addition it's not certain whether the storage-object keyset should
	   really be a dependent object or not, in theory it's nice because it
	   allows keyset-specific messages/accesses to be sent to the device and
	   automatically routed to the keyset (standard accesses will still go 
	   to the device, so for example a getItem() will be handled as a 
	   device-get rather than a keyset-get) but such unmediated access to 
	   the underlying keyset probably isn't a good idea anyway */
	status = krnlSendMessage( createInfo.cryptHandle, IMESSAGE_SETATTRIBUTE,
							  ( MESSAGE_CAST ) &iCryptDevice, 
							  CRYPT_IATTRIBUTE_HWDEVICE );
	if( cryptStatusError( status ) )
		{
		krnlSendNotifier( createInfo.cryptHandle, IMESSAGE_DECREFCOUNT );
		return( status );
		}
	*iCryptKeyset = createInfo.cryptHandle;

	return( isFileKeyset ? OK_SPECIAL : CRYPT_OK );
	}

CHECK_RETVAL \
static int deleteStorageObject( const BOOLEAN updateBackingStore,
								const BOOLEAN isFileKeyset )
	{
	int status;

	REQUIRES( updateBackingStore == TRUE || updateBackingStore == FALSE );
	REQUIRES( isFileKeyset == TRUE || isFileKeyset == FALSE );
	REQUIRES( !updateBackingStore == !!isFileKeyset );
			  /* The awkward expression is necessary to convert the safe 
			     boolean into a 0/1 boolean */

	/* Delete the storage object */
	if( !isFileKeyset )
		{
		void *storageObjectAddr;
		int storageObjectSize;

		/* Clear the storage and notify the HAL of the change if required */
		status = hwGetStorage( &storageObjectAddr, &storageObjectSize );
		if( cryptStatusError( status ) && status != OK_SPECIAL )
			{
			/* Another shouldn't-occur situation, see the comment in
			   getHardwareReference() */
			DEBUG_DIAG(( "Reference to secure hardware storage not "
						 "available from HAL" ));
			return( CRYPT_ERROR_NOTFOUND );
			}
		zeroise( storageObjectAddr, storageObjectSize );
		if( updateBackingStore )
			hwStorageUpdateNotify( 0 );
		}
	else
		{
		char storageFilePath[ MAX_PATH_LENGTH + 8 ];
		int storageFilePathLen;

		status = fileBuildCryptlibPath( storageFilePath, MAX_PATH_LENGTH, 
										&storageFilePathLen, "CLKEYS", 6, 
										BUILDPATH_GETPATH );
		if( cryptStatusError( status ) )
			return( status );
		fileErase( storageFilePath );
		}

	return( CRYPT_OK );
	}

/* Initialise/zeroise the device.  This gets a bit complicated because the 
   device that we're initialising could be the internal crypto hardware 
   device if this is in use, so we have to perform a sleight-of-hand that
   allows it to be reset during use */

#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )

CHECK_RETVAL_BOOL STDC_NONNULL_ARG( ( 1 ) ) \
static BOOLEAN isCryptoHardwareDevice( const DEVICE_INFO *deviceInfoPtr )
	{					   
	CRYPT_KEYSET iHWKeyset;
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	int status;

	assert( isReadPtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );

	/* If this is the crypto hardware device, there's no special handling
	   required */
	if( deviceInfoPtr->objectHandle == CRYPTO_OBJECT_HANDLE )
		return( FALSE );

	/* It's not the crypto hardware device, check whether it's an alias for
	   it by comparing the keyset handle, which for an aliased device will
	   be a reference to the one used by the crypto hardware device */
	status = krnlSendMessage( CRYPTO_OBJECT_HANDLE, IMESSAGE_GETATTRIBUTE, 
							  &iHWKeyset, CRYPT_IATTRIBUTE_HWSTORAGE );
	if( cryptStatusError( status ) )
		return( FALSE );
	
	return( iHWKeyset == hardwareInfo->iCryptKeyset ? TRUE : FALSE );
	}
#else
  #define isCryptoHardwareDevice( deviceInfoPtr )		FALSE
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1 ) ) \
static int initDevice( INOUT DEVICE_INFO *deviceInfoPtr,
					   const BOOLEAN isCryptoDeviceAlias,	
					   const BOOLEAN isZeroise )
	{
	CRYPT_KEYSET iHWKeyset;
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	int status;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );

	REQUIRES( isCryptoDeviceAlias == TRUE || isCryptoDeviceAlias == FALSE );
	REQUIRES( isZeroise == TRUE || isZeroise == FALSE );

	/* If this is an alias to the crypto hardware device then we need to 
	   handle it specially since we're about to reinitialise the hardware
	   out from underneath it.  We perform this sleight-of-hand trick by 
	   notifying the crypto hardware device to reset its state, which 
	   reinitialises the storage, and then get a reference to the newly-
	   intialised storage from it */
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	if( isCryptoDeviceAlias )
		{
		status = krnlSendMessage( CRYPTO_OBJECT_HANDLE, 
								  IMESSAGE_SETATTRIBUTE, MESSAGE_VALUE_TRUE, 
								  CRYPT_IATTRIBUTE_RESETNOTIFY );
		if( cryptStatusOK( status ) )
			status = getCryptoStorageObject( &hardwareInfo->iCryptKeyset );
		if( cryptStatusError( status ) )
			return( status );
		SET_FLAG( deviceInfoPtr->flags, DEVICE_FLAG_LOGGEDIN );

		return( CRYPT_OK );
		}
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

	/* The only real difference between a zeroise and an initialise is that 
	   the zeroise only clears existing state and exits while the initialise 
	   resets the state with the device ready to be used again */
	if( isZeroise )
		return( deleteStorageObject( TRUE, hardwareInfo->isFileKeyset ) );

	/* Initialise the device.  In addition to the logged-in flag we set the 
	   needs-cleanup flag to indicate that if the device is closed before 
	   performing further initialisation then the storage object should be 
	   removed */
	status = openStorageObject( &iHWKeyset, CRYPT_KEYOPT_CREATE,
								deviceInfoPtr->objectHandle, 
								DEVICE_ERRINFO );
	if( cryptStatusError( status ) && status != OK_SPECIAL )
		return( status );
	hardwareInfo->iCryptKeyset = iHWKeyset;
	if( status == OK_SPECIAL )
		hardwareInfo->isFileKeyset = TRUE;
	SET_FLAG( deviceInfoPtr->flags, 
			  DEVICE_FLAG_LOGGEDIN | DEVICE_FLAG_NEEDSCLEANUP );

	return( CRYPT_OK );
	}

/****************************************************************************
*																			*
*					Device Init/Shutdown/Device Control Routines			*
*																			*
****************************************************************************/

/* Close a previously-opened session with the device.  We have to have this
   before the initialisation function since it may be called by it if the 
   initialisation process fails */

STDC_NONNULL_ARG( ( 1 ) ) \
static void shutdownFunction( INOUT DEVICE_INFO *deviceInfoPtr )
	{
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );

	/* Shut down access to the storage object */
	if( hardwareInfo->iCryptKeyset != CRYPT_ERROR )
		{
		/* Close the storage object and, if there's cleanup required, delete 
		   it.  We don't force the change through to the backing store because
		   the cleanup flag indicates that the initialistion process wasn't
		   completed before it could be written to backing store, so there's
		   nothing to update in the backing store */
		krnlSendNotifier( hardwareInfo->iCryptKeyset, IMESSAGE_DECREFCOUNT );
		if( TEST_FLAG( deviceInfoPtr->flags, DEVICE_FLAG_NEEDSCLEANUP ) )
			( void ) deleteStorageObject( FALSE, hardwareInfo->isFileKeyset );
		hardwareInfo->iCryptKeyset = CRYPT_ERROR;
		}
	CLEAR_FLAG( deviceInfoPtr->flags, DEVICE_FLAG_ACTIVE | \
									  DEVICE_FLAG_LOGGEDIN | \
									  DEVICE_FLAG_NEEDSCLEANUP );
	}

/* Open a session with the device */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1 ) ) \
static int completeInit( INOUT DEVICE_INFO *deviceInfoPtr )
	{
	CRYPT_KEYSET iCryptKeyset;
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	const DEV_GETRANDOMFUNCTION getRandomFunction = \
					( DEV_GETRANDOMFUNCTION ) \
					FNPTR_GET( deviceInfoPtr->getRandomFunction );
	MESSAGE_DATA msgData;
	BYTE buffer[ 32 + 8 ];
	const int quality = 95;
	int status;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );

	REQUIRES( getRandomFunction != NULL );

	/* Grab 256 bits of entropy and send it to the system device.  Since 
	   we're using a crypto hardware device we assume that it's good-quality 
	   entropy */
	status = getRandomFunction( deviceInfoPtr, buffer, 32, NULL );
	ENSURES( cryptStatusOK( status ) );
	setMessageData( &msgData, buffer, 32 );
	status = krnlSendMessage( SYSTEM_OBJECT_HANDLE, 
							  IMESSAGE_SETATTRIBUTE_S, &msgData, 
							  CRYPT_IATTRIBUTE_ENTROPY );
	if( cryptStatusOK( status ) )
		{
		( void ) krnlSendMessage( SYSTEM_OBJECT_HANDLE, 
								  IMESSAGE_SETATTRIBUTE,
								  ( MESSAGE_CAST ) &quality,
								  CRYPT_IATTRIBUTE_ENTROPY_QUALITY );
		}
	zeroise( buffer, 32 );

	/* Try and open the PKCS #15 storage object, either in the hardware if 
	   it provides secure storage or in a disk file if not.  If we can't 
	   open it it means that either it doesn't exist (i.e. persistent key 
	   storage isn't supported) or the device hasn't been initialised yet.  
	   This isn't a fatal error, although it does mean that some public-key 
	   operations will be restricted if they depend on storing key metadata 
	   in the storage object */
	status = openStorageObject( &iCryptKeyset, CRYPT_KEYOPT_NONE,
								deviceInfoPtr->objectHandle,
								DEVICE_ERRINFO );
	if( cryptStatusOK( status ) || status == OK_SPECIAL )
		{
		hardwareInfo->iCryptKeyset = iCryptKeyset;
		if( status == OK_SPECIAL )
			hardwareInfo->isFileKeyset = TRUE;
		status = CRYPT_OK;
		}

	return( status );
	}

CHECK_RETVAL STDC_NONNULL_ARG( ( 1 ) ) \
static int initFunction( INOUT DEVICE_INFO *deviceInfoPtr, 
						 STDC_UNUSED const char *name,
						 STDC_UNUSED const int nameLength )
	{
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	int status;

	UNUSED_ARG( name );

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );

	/* Set up any internal objects to contain invalid handles */
	hardwareInfo->iCryptKeyset = CRYPT_ERROR;

	/* In some configurations the hardware device isn't a standard crypto
	   device but a HAL used to access custom or non-public algorithms and
	   mechanisms.  In this case we're being initialised at boot time and
	   won't be visible externally, so we don't perform any further 
	   initialisation beyond this point.  In particular we don't try and
	   open the PKCS #15 storage object since keyset access won't be 
	   available yet */
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	if( deviceInfoPtr->objectHandle == CRYPTO_OBJECT_HANDLE )
		{
		deviceInfoPtr->label = "Crypto HAL";
		deviceInfoPtr->labelLen = 10;
		SET_FLAG( deviceInfoPtr->flags, DEVICE_FLAG_ACTIVE );

		ENSURES( sanityCheckDevice( deviceInfoPtr ) );

		return( CRYPT_OK );
		}
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

	/* Set up the device information.  Since this is a built-in hardware 
	   device it's always present and available */
	deviceInfoPtr->label = "Cryptographic hardware device";
	deviceInfoPtr->labelLen = 29;
	SET_FLAG( deviceInfoPtr->flags, 
			  DEVICE_FLAG_ACTIVE | DEVICE_FLAG_LOGGEDIN );

	/* Complete the initialisation process */
	status = completeInit( deviceInfoPtr );
	if( cryptStatusError( status ) )
		return( status );

	ENSURES( sanityCheckDevice( deviceInfoPtr ) );

	return( CRYPT_OK );
	}

/* Handle device control functions */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1 ) ) \
static int controlFunction( INOUT DEVICE_INFO *deviceInfoPtr,
							IN_ATTRIBUTE const CRYPT_ATTRIBUTE_TYPE type,
							IN_BUFFER_OPT( dataLength ) void *data, 
							IN_LENGTH_SHORT_Z const int dataLength,
							INOUT_OPT MESSAGE_FUNCTION_EXTINFO *messageExtInfo )
	{
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );
	assert( data == NULL || isReadPtrDynamic( data, dataLength ) );

	REQUIRES( sanityCheckDevice( deviceInfoPtr ) );
	REQUIRES( isAttribute( type ) || isInternalAttribute( type ) );

	UNUSED_ARG( hardwareInfo );
	UNUSED_ARG( messageExtInfo );

	/* Handle user authorisation.  Since this is a built-in hardware device 
	   it's always available for use so these are just dummy routines, 
	   although they can be expanded to call down into the HAL for actual
	   authentication if any hardware with such a facility is ever used */
	if( type == CRYPT_DEVINFO_AUTHENT_USER || \
		type == CRYPT_DEVINFO_AUTHENT_SUPERVISOR )
		{
		/* Authenticate the user */
		/* ... */

		/* The device is now ready for use */
		SET_FLAG( deviceInfoPtr->flags, DEVICE_FLAG_LOGGEDIN );		
		return( CRYPT_OK );
		}

	/* Handle authorisation value changes */
	if( type == CRYPT_DEVINFO_SET_AUTHENT_SUPERVISOR )
		{
		/* Set SO PIN */
		/* ... */

		return( CRYPT_OK );
		}
	if( type == CRYPT_DEVINFO_SET_AUTHENT_USER )
		{
		/* Set user PIN */
		/* ... */

		return( CRYPT_OK );
		}

	/* Handle initialisation and zeroisation */
	if( type == CRYPT_DEVINFO_INITIALISE || \
		type == CRYPT_DEVINFO_ZEROISE )
		{
		const BOOLEAN isCryptoDeviceAlias = \
							isCryptoHardwareDevice( deviceInfoPtr );
							/* Get device information before we clear it */

		/* Shut down any existing state if necessary in preparation for the 
		   zeroise/initialise.  Since this clears all state we manually 
		   reset the device-active flag since we're still active, just with
		   all information cleared */
		shutdownFunction( deviceInfoPtr );
		hwInitialise();
		SET_FLAG( deviceInfoPtr->flags, DEVICE_FLAG_ACTIVE );

		/* Initialise the device */
		return( initDevice( deviceInfoPtr, isCryptoDeviceAlias,
							( type == CRYPT_DEVINFO_ZEROISE ) ? \
							  TRUE : FALSE ) );
		}

	/* Complete the object initialisation process.  This is handled as a
	   distinct operation for the system crypto object, which can't be
	   completely initialised when it's created because the cryptlib 
	   initialisation process hasn't completed yet */
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	if( type == CRYPT_IATTRIBUTE_COMPLETEINIT )
		{
		int status;

		status = completeInit( deviceInfoPtr );
		if( cryptStatusError( status ) )
			{
			/* Since this function is called on object creation, and in this 
			   case on system init, if it fails there's no object to get 
			   extended error information from so we dump the error 
			   information as a diagnostic for debugging purposes */
			DEBUG_DIAG_ERRMSG(( "Hardware device init completion failed, "
								"status %s, error string:\n  '%s'", 
								getStatusName( status ),
								getErrorInfoString( DEVICE_ERRINFO ) ));
			}
		return( status );
		}
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

	/* Handle high-reliability time */
	if( type == CRYPT_IATTRIBUTE_TIME )
		{
		time_t *timePtr = ( time_t * ) data;

		UNUSED_ARG( timePtr );

		return( CRYPT_ERROR_NOTAVAIL );
		}

	/* Handle the commit notification for data held in the underlying 
	   storage object, indicating that the data has changed.  This is used 
	   for the system crypto object, which can have in-memory key data 
	   associated with it, to tell the crypto HAL to update its view of the
	   storage object */
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	if( type == CRYPT_IATTRIBUTE_COMMITNOTIFY )
		{
		REQUIRES( dataLength == CRYPT_UNUSED || dataLength == 0 || \
				  rangeCheck( dataLength, MIN_CRYPT_OBJECTSIZE, \
							  MAX_INTLENGTH ) );

		/* If we're in the middle of a reset/zeroise, we don't commit any 
		   data to storage but instead clear it */
		if( hardwareInfo->discardData )
			{
			return( deleteStorageObject( TRUE, 
										 hardwareInfo->isFileKeyset ) );
			}

		hwStorageUpdateNotify( dataLength );

		return( CRYPT_OK );
		}
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

	/* Handle the reset notification when the device is initialised.  This
	   just closes the keyset, discarding what's written, and reloads it
	   from the storage object */
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	if( type == CRYPT_IATTRIBUTE_RESETNOTIFY )
		{
		/* Close the keyset, discarding any data from it and clearing the 
		   underlying storage */
		hardwareInfo->discardData = TRUE;
		krnlSendNotifier( hardwareInfo->iCryptKeyset, 
						  IMESSAGE_DECREFCOUNT );
		hardwareInfo->discardData = FALSE;

		/* Redo the initialisation */
		return( completeInit( deviceInfoPtr ) );
		}
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

	retIntError();
	}

/* Get random data from the device.  The messageExtInfo parameter is used
   to handle recursive messages sent to the system device during the 
   randomness-polling process and isn't used here */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1, 2 ) ) \
static int getRandomFunction( INOUT DEVICE_INFO *deviceInfoPtr, 
							  OUT_BUFFER_FIXED( length ) void *buffer,
							  IN_LENGTH_SHORT const int length, 
							  INOUT_OPT MESSAGE_FUNCTION_EXTINFO *messageExtInfo )
	{
	UNUSED_ARG( messageExtInfo );

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );
	assert( isWritePtrDynamic( buffer, length ) );

	REQUIRES( sanityCheckDevice( deviceInfoPtr ) );
	REQUIRES( isShortIntegerRangeNZ( length ) );

	/* Clear the return value and make sure that we fail the FIPS 140 tests
	   on the output if there's a problem */
	REQUIRES( isShortIntegerRangeNZ( length ) ); 
	zeroise( buffer, length );

	/* Fill the buffer with random data */
	return( hwGetRandom( buffer, length ) );
	}

/* Query the custom crypto HAL for encoding information for custom 
   algorithms and mechanisms that may be provided by the HAL.  This is just
   a wrapper that passes the call throug to the HAL */

#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )

CHECK_RETVAL STDC_NONNULL_ARG( ( 1, 2 ) ) \
static int catalogQueryFunction( INOUT DEVICE_INFO *deviceInfoPtr, 
								 INOUT MESSAGE_CATALOGQUERY_INFO *queryInfo, 
								 IN_ENUM( CATALOGQUERY_ITEM ) \
											const CATALOGQUERY_ITEM_TYPE itemType )
	{
	return( hwCatalogQuery( queryInfo, itemType ) );
	}
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */

/****************************************************************************
*																			*
*						Get/Set/Delete Item Routines						*
*																			*
****************************************************************************/

/* Instantiate an object in a device.  This works like the create-context
   function but instantiates a cryptlib object using data already contained
   in the device, for example a stored private key or a certificate.  If 
   we're not using a crypto HAL (in other words cryptlib's native crypto is
   enabled) and the value being read is a public key and there's a 
   certificate attached then the instantiated object is a native cryptlib 
   object rather than a device object with a native certificate object 
   attached because there doesn't appear to be any good reason to create the 
   public-key object in the device, and the cryptlib native object will 
   probably be faster anyway */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1, 2, 5, 8 ) ) \
static int getItemFunction( INOUT DEVICE_INFO *deviceInfoPtr,
							OUT_HANDLE_OPT CRYPT_HANDLE *iCryptContext,
							IN_ENUM( KEYMGMT_ITEM ) \
								const KEYMGMT_ITEM_TYPE itemType,
							IN_KEYID const CRYPT_KEYID_TYPE keyIDtype,
							IN_BUFFER( keyIDlength ) const void *keyID, 
							IN_LENGTH_KEYID const int keyIDlength,
							IN_OPT void *auxInfo, 
							INOUT_LENGTH_SHORT_Z int *auxInfoLength,
							IN_FLAGS_Z( KEYMGMT ) const int flags )
	{
	CRYPT_CONTEXT iLocalContext;
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	MESSAGE_KEYMGMT_INFO getkeyInfo;
	int storageRef, status;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );
	assert( isWritePtr( iCryptContext, sizeof( CRYPT_CONTEXT ) ) );
	assert( isReadPtrDynamic( keyID, keyIDlength ) );

	REQUIRES( sanityCheckDevice( deviceInfoPtr ) );
	REQUIRES( itemType == KEYMGMT_ITEM_PUBLICKEY || \
			  itemType == KEYMGMT_ITEM_PRIVATEKEY );
	REQUIRES( keyIDtype == CRYPT_KEYID_NAME || \
			  keyIDtype == CRYPT_KEYID_URI || \
			  keyIDtype == CRYPT_IKEYID_KEYID || \
			  keyIDtype == CRYPT_IKEYID_PGPKEYID || \
			  keyIDtype == CRYPT_IKEYID_ISSUERANDSERIALNUMBER );
	REQUIRES( keyIDlength >= MIN_NAME_LENGTH && \
			  keyIDlength < MAX_ATTRIBUTE_SIZE );
	REQUIRES( auxInfo == NULL && *auxInfoLength == 0 );
	REQUIRES( isFlagRangeZ( flags, KEYMGMT ) );

	/* Clear return value */
	*iCryptContext = CRYPT_ERROR;

	/* Redirect the fetch down to the PKCS #15 storage object, which will
	   create either a dummy context that we have to connect to the actual
	   hardware or a native public-key/certificate object if it's a non-
	   private-key item and we're not using a crypto HAL for our crypto */
	if( hardwareInfo->iCryptKeyset == CRYPT_ERROR )
		{
		retExt( CRYPT_ERROR_NOTINITED, 
				( CRYPT_ERROR_NOTINITED, DEVICE_ERRINFO,
				  "No storage object associated with this device" ) );
		}
	setMessageKeymgmtInfo( &getkeyInfo, keyIDtype, keyID, keyIDlength,
						   NULL, 0, flags );
	status = krnlSendMessage( hardwareInfo->iCryptKeyset,
							  IMESSAGE_KEY_GETKEY, &getkeyInfo,
							  itemType );
	if( cryptStatusError( status ) )
		{
		retExtObjDirect( status, DEVICE_ERRINFO, 
						 hardwareInfo->iCryptKeyset );
		}
	iLocalContext = getkeyInfo.cryptHandle;

	/* If it's a public-key fetch and we're not using a crypto HAL, we've 
	   created a cryptlib native object and we're done */
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	if( deviceInfoPtr->objectHandle != CRYPTO_OBJECT_HANDLE && \
		itemType != KEYMGMT_ITEM_PRIVATEKEY )
#else
	if( itemType != KEYMGMT_ITEM_PRIVATEKEY )
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */
		{
		*iCryptContext = iLocalContext;
		return( CRYPT_OK );
		}

	/* Connect the dummy context that was created with the underlying 
	   hardware.  When this final step has been completed we can move the 
	   context to the initialised state */
	status = getHardwareReference( iLocalContext, &storageRef );
	if( cryptStatusError( status ) )
		{
		krnlSendNotifier( iLocalContext, IMESSAGE_DECREFCOUNT );
		retExt( status,
				( status, DEVICE_ERRINFO,
				  "Fetched item doesn't correspond to anything known to "
				  "the crypto HAL" ) );
		}
	status = krnlSendMessage( iLocalContext, IMESSAGE_SETATTRIBUTE,
							  &storageRef, CRYPT_IATTRIBUTE_DEVICEOBJECT );
	if( cryptStatusOK( status ) )
		{
		status = krnlSendMessage( iLocalContext, IMESSAGE_SETATTRIBUTE, 
								  MESSAGE_VALUE_UNUSED, 
								  CRYPT_IATTRIBUTE_INITIALISED );
		}
	if( cryptStatusError( status ) )
		{
		krnlSendNotifier( iLocalContext, IMESSAGE_DECREFCOUNT );
		return( status );
		}

	*iCryptContext = iLocalContext;
	return( CRYPT_OK );
	}

/* Add an object to a device.  This can only ever add a certificate
   (enforced by the kernel ACLs) so we don't have to perform any 
   special-case handling */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1 ) ) \
static int setItemFunction( INOUT DEVICE_INFO *deviceInfoPtr, 
							IN_HANDLE const CRYPT_HANDLE iCryptHandle )
	{
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	MESSAGE_KEYMGMT_INFO setkeyInfo;
	int status;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );

	REQUIRES( sanityCheckDevice( deviceInfoPtr ) );
	REQUIRES( isHandleRangeValid( iCryptHandle ) );

	/* Redirect the add down to the PKCS #15 storage object */
	if( hardwareInfo->iCryptKeyset == CRYPT_ERROR )
		{
		retExt( CRYPT_ERROR_NOTINITED, 
				( CRYPT_ERROR_NOTINITED, DEVICE_ERRINFO,
				  "No storage object associated with this device" ) );
		}
	setMessageKeymgmtInfo( &setkeyInfo, CRYPT_KEYID_NONE, NULL, 0,
						   NULL, 0, KEYMGMT_FLAG_NONE );
	setkeyInfo.cryptHandle = iCryptHandle;
	status = krnlSendMessage( hardwareInfo->iCryptKeyset,
							  IMESSAGE_KEY_SETKEY, &setkeyInfo,
							  KEYMGMT_ITEM_PUBLICKEY );
	if( cryptStatusError( status ) )
		{
		retExtObjDirect( status, DEVICE_ERRINFO, 
						 hardwareInfo->iCryptKeyset );
		}

	return( CRYPT_OK );
	}

/* Delete an object in a device */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1, 4 ) ) \
static int deleteItemFunction( INOUT DEVICE_INFO *deviceInfoPtr,
							   IN_ENUM( KEYMGMT_ITEM ) \
									const KEYMGMT_ITEM_TYPE itemType,
							   IN_KEYID const CRYPT_KEYID_TYPE keyIDtype,
							   IN_BUFFER( keyIDlength ) const void *keyID, 
							   IN_LENGTH_KEYID const int keyIDlength )
	{
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	MESSAGE_KEYMGMT_INFO getkeyInfo, deletekeyInfo;
	int status;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );
	assert( isReadPtrDynamic( keyID, keyIDlength ) );

	REQUIRES( sanityCheckDevice( deviceInfoPtr ) );
	REQUIRES( itemType == KEYMGMT_ITEM_PUBLICKEY || \
			  itemType == KEYMGMT_ITEM_PRIVATEKEY );
	REQUIRES( keyIDtype == CRYPT_KEYID_NAME );
	REQUIRES( keyIDlength >= MIN_NAME_LENGTH && \
			  keyIDlength < MAX_ATTRIBUTE_SIZE );

	/* Perform the delete both from the PKCS #15 storage object and the
	   native storage.  This gets a bit complicated because all that we have
	   to identify the item is one of several types of keyID and the 
	   hardware device needs a storageID to identify it.  To deal with this 
	   we have to instantiate a dummy object via the keyID which then 
	   contains the storageID, from which we can get the storageRef.  
	   
	   In addition if we're not using a crypto HAL and the object that's 
	   stored isn't a private-key object then there's no associated 
	   cryptographic hardware object.  To handle this we try and instantiate 
	   a dummy private-key object in order to get the storageID, and if 
	   we're using a crypto HAL we fall back to trying for a public-key 
	   object if that fails.  If this succeeds, we use it to locate the 
	   underlying hardware object and delete it.  Finally, we delete the 
	   original PKCS #15 object */
	if( hardwareInfo->iCryptKeyset == CRYPT_ERROR )
		{
		retExt( CRYPT_ERROR_NOTINITED, 
				( CRYPT_ERROR_NOTINITED, DEVICE_ERRINFO,
				  "No storage object associated with this device" ) );
		}
	setMessageKeymgmtInfo( &getkeyInfo, keyIDtype, keyID, keyIDlength,
						   NULL, 0, KEYMGMT_FLAG_NONE );
	status = krnlSendMessage( hardwareInfo->iCryptKeyset,
							  IMESSAGE_KEY_GETKEY, &getkeyInfo,
							  KEYMGMT_ITEM_PRIVATEKEY );
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	if( cryptStatusError( status ) && \
		deviceInfoPtr->objectHandle == CRYPTO_OBJECT_HANDLE )
		{
		/* It's not a private-key object, try again with a public-key 
		   object */
		status = krnlSendMessage( hardwareInfo->iCryptKeyset,
								  IMESSAGE_KEY_GETKEY, &getkeyInfo,
								  KEYMGMT_ITEM_PUBLICKEY );
		}
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */
	if( cryptStatusOK( status ) )
		{
		int storageRef;

		/* We've located the hardware object, get its hardware reference and 
		   delete it (we destroy the cryptlib-level object before we do this
		   since we're about to delete the corresponding hardware object out
		   from underneath it).  If this fails we continue anyway because we 
		   know that there's also a PKCS #15 object to delete */
		status = getHardwareReference( getkeyInfo.cryptHandle, &storageRef );
		krnlSendNotifier( getkeyInfo.cryptHandle, IMESSAGE_DECREFCOUNT );
		if( cryptStatusOK( status ) )
			( void ) hwDeleteItem( storageRef );
		}
	setMessageKeymgmtInfo( &deletekeyInfo, keyIDtype, keyID, keyIDlength,
						   NULL, 0, KEYMGMT_FLAG_NONE );
	status = krnlSendMessage( hardwareInfo->iCryptKeyset,
							  IMESSAGE_KEY_DELETEKEY, &deletekeyInfo,
							  itemType );
	if( cryptStatusError( status ) )
		{
		retExtObjDirect( status, DEVICE_ERRINFO, 
						 hardwareInfo->iCryptKeyset );
		}

	return( CRYPT_OK );
	}

/* Get the sequence of certificates in a chain from a device.  Since these 
   functions operate only on certificates we can redirect them straight down 
   to the underlying storage object */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1, 2, 3, 5 ) ) \
static int getFirstItemFunction( INOUT DEVICE_INFO *deviceInfoPtr, 
								 OUT_HANDLE_OPT CRYPT_CERTIFICATE *iCertificate,
								 OUT int *stateInfo,
								 IN_KEYID const CRYPT_KEYID_TYPE keyIDtype,
								 IN_BUFFER( keyIDlength ) const void *keyID, 
								 IN_LENGTH_KEYID const int keyIDlength,
								 IN_ENUM( KEYMGMT_ITEM ) \
									const KEYMGMT_ITEM_TYPE itemType,
								 IN_FLAGS_Z( KEYMGMT ) const int options )
	{
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	MESSAGE_KEYMGMT_INFO getnextcertInfo;
	int status;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );
	assert( isWritePtr( iCertificate, sizeof( CRYPT_CERTIFICATE ) ) );
	assert( isReadPtrDynamic( keyID, keyIDlength ) );
	assert( isWritePtr( stateInfo, sizeof( int ) ) );

	REQUIRES( sanityCheckDevice( deviceInfoPtr ) );
	REQUIRES( keyIDtype == CRYPT_IKEYID_KEYID );
	REQUIRES( isShortIntegerRangeMin( keyIDlength, 4 ) );
	REQUIRES( itemType == KEYMGMT_ITEM_PUBLICKEY );
	REQUIRES( isFlagRangeZ( options, KEYMGMT ) );

	/* Clear return values */
	*iCertificate = CRYPT_ERROR;
	*stateInfo = CRYPT_ERROR;

	/* Make sure that there's somewhere to fetch the item from */
	if( hardwareInfo->iCryptKeyset == CRYPT_ERROR )
		{
		retExt( CRYPT_ERROR_NOTINITED, 
				( CRYPT_ERROR_NOTINITED, DEVICE_ERRINFO,
				  "No storage object associated with this device" ) );
		}

	/* Get the first certificate */
	setMessageKeymgmtInfo( &getnextcertInfo, keyIDtype, keyID, keyIDlength, 
						   stateInfo, sizeof( int ), options );
	status = krnlSendMessage( hardwareInfo->iCryptKeyset, 
							  IMESSAGE_KEY_GETFIRSTCERT, &getnextcertInfo, 
							  KEYMGMT_ITEM_PUBLICKEY );
	if( cryptStatusError( status ) )
		{
		retExtObjDirect( status, DEVICE_ERRINFO, 
						 hardwareInfo->iCryptKeyset );
		}

	return( CRYPT_OK );
	}

CHECK_RETVAL STDC_NONNULL_ARG( ( 1, 2, 3 ) ) \
static int getNextItemFunction( INOUT DEVICE_INFO *deviceInfoPtr, 
								OUT_HANDLE_OPT CRYPT_CERTIFICATE *iCertificate,
								INOUT int *stateInfo, 
								IN_FLAGS_Z( KEYMGMT ) const int options )
	{
	HARDWARE_INFO *hardwareInfo = deviceInfoPtr->deviceHardware;
	MESSAGE_KEYMGMT_INFO getnextcertInfo;
	int status;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );
	assert( isWritePtr( iCertificate, sizeof( CRYPT_CERTIFICATE ) ) );
	assert( isWritePtr( stateInfo, sizeof( int ) ) );

	REQUIRES( sanityCheckDevice( deviceInfoPtr ) );
	REQUIRES( isHandleRangeValid( *stateInfo ) || *stateInfo == CRYPT_ERROR );
	REQUIRES( isFlagRangeZ( options, KEYMGMT ) );

	UNUSED_ARG( hardwareInfo );

	/* Clear return value */
	*iCertificate = CRYPT_ERROR;

	/* Make sure that there's somewhere to fetch the item from.  This can 
	   happen if the device is cleared/zeroised/reinitalised after the 
	   getFirstItem() call */
	if( hardwareInfo->iCryptKeyset == CRYPT_ERROR )
		{
		retExt( CRYPT_ERROR_NOTINITED, 
				( CRYPT_ERROR_NOTINITED, DEVICE_ERRINFO,
				  "No storage object associated with this device" ) );
		}

	/* If the previous certificate was the last one, there's nothing left to 
	   fetch */
	if( *stateInfo == CRYPT_ERROR )
		return( CRYPT_ERROR_NOTFOUND );

	/* Get the next certificate */
	setMessageKeymgmtInfo( &getnextcertInfo, CRYPT_KEYID_NONE, NULL, 0, 
						   stateInfo, sizeof( int ), options );
	status = krnlSendMessage( hardwareInfo->iCryptKeyset, 
							  IMESSAGE_KEY_GETNEXTCERT, &getnextcertInfo, 
							  KEYMGMT_ITEM_PUBLICKEY );
	if( cryptStatusError( status ) )
		{
		retExtObjDirect( status, DEVICE_ERRINFO, 
						 hardwareInfo->iCryptKeyset );
		}

	return( CRYPT_OK );
	}

/****************************************************************************
*																			*
*						 	Device Access Routines							*
*																			*
****************************************************************************/

/* Mechanisms for use with the hardware if it doesn't provide its own 
   mechanisms.  These aren't the full set supported by the system device 
   since functions like private key export aren't available.  The list is 
   sorted in order of frequency of use in order to make lookups a bit 
   faster */

static const MECHANISM_FUNCTION_INFO defaultMechanismFunctions[] = {
	{ MESSAGE_DEV_EXPORT, MECHANISM_ENC_PKCS1, ( MECHANISM_FUNCTION ) exportPKCS1 },
	{ MESSAGE_DEV_IMPORT, MECHANISM_ENC_PKCS1, ( MECHANISM_FUNCTION ) importPKCS1 },
	{ MESSAGE_DEV_SIGN, MECHANISM_SIG_PKCS1, ( MECHANISM_FUNCTION ) signPKCS1 },
	{ MESSAGE_DEV_SIGCHECK, MECHANISM_SIG_PKCS1, ( MECHANISM_FUNCTION ) sigcheckPKCS1 },
#if defined( USE_SSL ) && defined( USE_RSA_SUITES )
	{ MESSAGE_DEV_EXPORT, MECHANISM_ENC_PKCS1_RAW, ( MECHANISM_FUNCTION ) exportPKCS1 },
	{ MESSAGE_DEV_IMPORT, MECHANISM_ENC_PKCS1_RAW, ( MECHANISM_FUNCTION ) importPKCS1 },
#endif /* USE_SSL && USE_RSA_SUITES */
#ifdef USE_PGP
	{ MESSAGE_DEV_EXPORT, MECHANISM_ENC_PKCS1_PGP, ( MECHANISM_FUNCTION ) exportPKCS1PGP },
	{ MESSAGE_DEV_IMPORT, MECHANISM_ENC_PKCS1_PGP, ( MECHANISM_FUNCTION ) importPKCS1PGP },
#endif /* USE_PGP */
	{ MESSAGE_DEV_EXPORT, MECHANISM_ENC_CMS, ( MECHANISM_FUNCTION ) exportCMS },
	{ MESSAGE_DEV_IMPORT, MECHANISM_ENC_CMS, ( MECHANISM_FUNCTION ) importCMS },
	{ MESSAGE_DEV_DERIVE, MECHANISM_DERIVE_PBKDF2, ( MECHANISM_FUNCTION ) derivePBKDF2 },
#if defined( USE_ENVELOPES ) && defined( USE_CMS )
	{ MESSAGE_DEV_KDF, MECHANISM_DERIVE_PBKDF2, ( MECHANISM_FUNCTION ) kdfPBKDF2 },
#endif /* USE_ENVELOPES && USE_CMS */
	{ MESSAGE_DEV_KDF, MECHANISM_DERIVE_HKDF, ( MECHANISM_FUNCTION ) kdfHKDF },
#if defined( USE_PGP ) || defined( USE_PGPKEYS )
	{ MESSAGE_DEV_DERIVE, MECHANISM_DERIVE_PGP, ( MECHANISM_FUNCTION ) derivePGP },
#endif /* USE_PGP || USE_PGPKEYS */
#if defined( USE_SSL ) || defined( USE_SSH )
	{ MESSAGE_DEV_DERIVE, MECHANISM_DERIVE_HOTP, ( MECHANISM_FUNCTION ) deriveHOTP },
#endif /* USE_SSL || USE_SSH */
#ifdef USE_SSL
	{ MESSAGE_DEV_DERIVE, MECHANISM_DERIVE_TLS, ( MECHANISM_FUNCTION ) deriveSSL },
	{ MESSAGE_DEV_DERIVE, MECHANISM_DERIVE_SSL, ( MECHANISM_FUNCTION ) deriveTLS },
	{ MESSAGE_DEV_SIGN, MECHANISM_SIG_SSL, ( MECHANISM_FUNCTION ) signSSL },
	{ MESSAGE_DEV_SIGCHECK, MECHANISM_SIG_SSL, ( MECHANISM_FUNCTION ) sigcheckSSL },
#endif /* USE_SSL */
#ifdef USE_CMP
	{ MESSAGE_DEV_DERIVE, MECHANISM_DERIVE_CMP, ( MECHANISM_FUNCTION ) deriveCMP },
#endif /* USE_CMP */
#ifdef USE_OAEP
	{ MESSAGE_DEV_EXPORT, MECHANISM_ENC_OAEP, ( MECHANISM_FUNCTION ) exportOAEP },
	{ MESSAGE_DEV_IMPORT, MECHANISM_ENC_OAEP, ( MECHANISM_FUNCTION ) importOAEP },
#endif /* USE_OAEP */
#ifdef USE_PSS
	{ MESSAGE_DEV_SIGN, MECHANISM_SIG_PSS, ( MECHANISM_FUNCTION ) signPSS},
	{ MESSAGE_DEV_SIGCHECK, MECHANISM_SIG_PSS, ( MECHANISM_FUNCTION ) sigcheckPSS },
#endif /* USE_PSS */
#ifdef USE_PKCS12
	{ MESSAGE_DEV_DERIVE, MECHANISM_DERIVE_PKCS12, ( MECHANISM_FUNCTION ) derivePKCS12 },
#endif /* USE_PKCS12 */
	{ MESSAGE_NONE, MECHANISM_NONE, NULL }, { MESSAGE_NONE, MECHANISM_NONE, NULL }
	};

/* Initialise the cryptographic hardware and its crypto capability 
   interface */

#define MAX_DEVICE_CAPABILITIES		32

static CAPABILITY_INFO_LIST capabilityInfoList[ MAX_DEVICE_CAPABILITIES ];

CHECK_RETVAL \
int deviceInitHardware( void )
	{
	const CAPABILITY_INFO *capabilityInfo;
	LOOP_INDEX i;
	int noCapabilities, status;

	/* Get the hardware capability information */
	status = hwGetCapabilities( &capabilityInfo, &noCapabilities );
	if( cryptStatusError( status ) )
		{
		DEBUG_DIAG_ERRMSG(( "Couldn't get native hardware capabilities, "
							"status %s", getStatusName( status ) ));
		return( status );
		}
	ENSURES( noCapabilities > 0 && \
			 noCapabilities <= MAX_DEVICE_CAPABILITIES );

	/* Build the list of available capabilities */
	memset( capabilityInfoList, 0, \
			sizeof( CAPABILITY_INFO_LIST ) * MAX_DEVICE_CAPABILITIES );
	LOOP_MED( i = 0, i < noCapabilities && \
					 capabilityInfo[ i ].cryptAlgo != CRYPT_ALGO_NONE, i++ )
		{
		const CAPABILITY_INFO *capabilityInfoPtr;

		ENSURES( LOOP_INVARIANT_MED( i, 0, noCapabilities - 1 ) );

		capabilityInfoPtr = &capabilityInfo[ i ];

		DEBUG_DIAG(( "Hardware driver reports support for %s algorithm, key "
					 "size %d...%d", capabilityInfoPtr->algoName, 
					 capabilityInfoPtr->minKeySize, 
					 capabilityInfoPtr->maxKeySize ));

		REQUIRES( sanityCheckCapability( capabilityInfoPtr ) );
		
		DATAPTR_SET( capabilityInfoList[ i ].info, \
					 ( void * ) capabilityInfoPtr );
		DATAPTR_SET( capabilityInfoList[ i ].next, NULL );
		if( i > 0 )
			{
			DATAPTR_SET( capabilityInfoList[ i - 1 ].next, 
						 &capabilityInfoList[ i ] );
			}
		}
	ENSURES( LOOP_BOUND_OK );
	ENSURES( i < noCapabilities );

	/* Finally, patch in the generic-secret capability.  This is a bit of an 
	   odd capability that doesn't represent any encryption algorithm but is
	   used for authenticated encryption as an intermediate step when 
	   generating distinct encryption and authentication keys from a single 
	   shared-secret key */
	if( i < MAX_DEVICE_CAPABILITIES )
		{
		const CAPABILITY_INFO *capabilityInfoPtr = \
										getGenericSecretCapability();

		DEBUG_DIAG(( "Adding generic secret capability, key size %d...%d, "
					 "for authenticated encryption support", 
					 capabilityInfoPtr->minKeySize, 
					 capabilityInfoPtr->maxKeySize ));

		REQUIRES( sanityCheckCapability( capabilityInfoPtr ) );

		DATAPTR_SET( capabilityInfoList[ i ].info, \
					 ( void * ) capabilityInfoPtr );
		DATAPTR_SET( capabilityInfoList[ i ].next, NULL );
		DATAPTR_SET( capabilityInfoList[ i - 1 ].next, 
					 &capabilityInfoList[ i ] );
		}

	return( CRYPT_OK );
	}

void deviceEndHardware( void )
	{
	}

/* Set up the function pointers to the device methods.  This is called when 
   a hardware device object is created */

CHECK_RETVAL STDC_NONNULL_ARG( ( 1 ) ) \
int setDeviceHardware( INOUT DEVICE_INFO *deviceInfoPtr )
	{
	const MECHANISM_FUNCTION_INFO *mechanismFunctions;
	int mechanismFunctionCount, status;

	assert( isWritePtr( deviceInfoPtr, sizeof( DEVICE_INFO ) ) );

	FNPTR_SET( deviceInfoPtr->initFunction, initFunction );
	FNPTR_SET( deviceInfoPtr->shutdownFunction, shutdownFunction );
	FNPTR_SET( deviceInfoPtr->controlFunction, controlFunction );
	FNPTR_SET( deviceInfoPtr->getItemFunction, getItemFunction );
	FNPTR_SET( deviceInfoPtr->setItemFunction, setItemFunction );
	FNPTR_SET( deviceInfoPtr->deleteItemFunction, deleteItemFunction );
	FNPTR_SET( deviceInfoPtr->getFirstItemFunction, getFirstItemFunction );
	FNPTR_SET( deviceInfoPtr->getNextItemFunction, getNextItemFunction );
#ifndef CONFIG_NO_SELFTEST
	FNPTR_SET( deviceInfoPtr->selftestFunction, selftestDevice );
#endif /* !CONFIG_NO_SELFTEST */
	FNPTR_SET( deviceInfoPtr->getRandomFunction, getRandomFunction );
#if defined( CONFIG_CRYPTO_HW1 ) || defined( CONFIG_CRYPTO_HW2 )
	FNPTR_SET( deviceInfoPtr->catalogQueryFunction, catalogQueryFunction );
#endif /* CONFIG_CRYPTO_HW1 || CONFIG_CRYPTO_HW2 */
	DATAPTR_SET( deviceInfoPtr->capabilityInfoList, capabilityInfoList );
	status = hwGetMechanisms( &mechanismFunctions, 
							  &mechanismFunctionCount );
	if( cryptStatusOK( status ) )
		{
		DATAPTR_SET( deviceInfoPtr->mechanismFunctions, 
					 ( void * ) mechanismFunctions );
		deviceInfoPtr->mechanismFunctionCount = mechanismFunctionCount;
		}
	else
		{
		/* The hardware doesn't support mechanism-level operations, fall 
		   back to the built-in cryptlib ones */
		DATAPTR_SET( deviceInfoPtr->mechanismFunctions, \
					 ( void * ) defaultMechanismFunctions );
		deviceInfoPtr->mechanismFunctionCount = \
						FAILSAFE_ARRAYSIZE( defaultMechanismFunctions, \
											MECHANISM_FUNCTION_INFO );
		}

	return( CRYPT_OK );
	}
#endif /* USE_HARDWARE */
