#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#

from dataclasses import dataclass

from plc4py.api.exceptions.exceptions import PlcRuntimeException
from plc4py.api.exceptions.exceptions import SerializationException
from plc4py.api.messages.PlcMessage import PlcMessage
from plc4py.protocols.modbus.readwrite.ModbusPDU import ModbusPDU
from plc4py.spi.generation.ReadBuffer import ReadBuffer
from plc4py.spi.generation.WriteBuffer import WriteBuffer
from plc4py.utils.ConnectionStringHandling import strtobool
from typing import ClassVar
import math


@dataclass
class ModbusPDUMaskWriteHoldingRegisterResponse(ModbusPDU):
    reference_address: int
    and_mask: int
    or_mask: int
    # Accessors for discriminator values.
    error_flag: ClassVar[bool] = False
    function_flag: ClassVar[int] = 0x16
    response: ClassVar[bool] = True

    def serialize_modbus_pdu_child(self, write_buffer: WriteBuffer):
        write_buffer.push_context("ModbusPDUMaskWriteHoldingRegisterResponse")

        # Simple Field (referenceAddress)
        write_buffer.write_unsigned_short(
            self.reference_address, bit_length=16, logical_name="referenceAddress"
        )

        # Simple Field (andMask)
        write_buffer.write_unsigned_short(
            self.and_mask, bit_length=16, logical_name="andMask"
        )

        # Simple Field (orMask)
        write_buffer.write_unsigned_short(
            self.or_mask, bit_length=16, logical_name="orMask"
        )

        write_buffer.pop_context("ModbusPDUMaskWriteHoldingRegisterResponse")

    def length_in_bytes(self) -> int:
        return int(math.ceil(float(self.length_in_bits() / 8.0)))

    def length_in_bits(self) -> int:
        length_in_bits: int = super().length_in_bits()
        _value: ModbusPDUMaskWriteHoldingRegisterResponse = self

        # Simple field (referenceAddress)
        length_in_bits += 16

        # Simple field (andMask)
        length_in_bits += 16

        # Simple field (orMask)
        length_in_bits += 16

        return length_in_bits

    @staticmethod
    def static_parse_builder(read_buffer: ReadBuffer, response: bool):
        read_buffer.push_context("ModbusPDUMaskWriteHoldingRegisterResponse")

        if isinstance(response, str):
            response = bool(strtobool(response))

        reference_address: int = read_buffer.read_unsigned_short(
            logical_name="reference_address", bit_length=16, response=response
        )

        and_mask: int = read_buffer.read_unsigned_short(
            logical_name="and_mask", bit_length=16, response=response
        )

        or_mask: int = read_buffer.read_unsigned_short(
            logical_name="or_mask", bit_length=16, response=response
        )

        read_buffer.pop_context("ModbusPDUMaskWriteHoldingRegisterResponse")
        # Create the instance
        return ModbusPDUMaskWriteHoldingRegisterResponseBuilder(
            reference_address, and_mask, or_mask
        )

    def equals(self, o: object) -> bool:
        if self == o:
            return True

        if not isinstance(o, ModbusPDUMaskWriteHoldingRegisterResponse):
            return False

        that: ModbusPDUMaskWriteHoldingRegisterResponse = (
            ModbusPDUMaskWriteHoldingRegisterResponse(o)
        )
        return (
            (self.reference_address == that.reference_address)
            and (self.and_mask == that.and_mask)
            and (self.or_mask == that.or_mask)
            and super().equals(that)
            and True
        )

    def hash_code(self) -> int:
        return hash(self)

    def __str__(self) -> str:
        # TODO:- Implement a generic python object to probably json convertor or something.
        return ""


@dataclass
class ModbusPDUMaskWriteHoldingRegisterResponseBuilder:
    reference_address: int
    and_mask: int
    or_mask: int

    def build(
        self,
    ) -> ModbusPDUMaskWriteHoldingRegisterResponse:
        modbus_pdumask_write_holding_register_response: (
            ModbusPDUMaskWriteHoldingRegisterResponse
        ) = ModbusPDUMaskWriteHoldingRegisterResponse(
            self.reference_address, self.and_mask, self.or_mask
        )
        return modbus_pdumask_write_holding_register_response
