/**************************************************************************
**
** This file is part of Qt Creator
**
** Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
**
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** Commercial Usage
**
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
**
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at http://qt.nokia.com/contact.
**
**************************************************************************/

#include "vcsbaseoutputwindow.h"

#include <utils/qtcassert.h>

#include <QtGui/QPlainTextEdit>
#include <QtGui/QTextCharFormat>
#include <QtGui/QContextMenuEvent>
#include <QtGui/QMenu>
#include <QtGui/QAction>

#include <QtCore/QPointer>
#include <QtCore/QTextCodec>
#include <QtCore/QTime>

namespace VCSBase {

namespace Internal {

// A plain text edit with a special context menu containing "Clear" and
// and functions to append specially formatted entries.
class OutputWindowPlainTextEdit : public QPlainTextEdit {
public:
    explicit OutputWindowPlainTextEdit(QWidget *parent);

    void appendLines(QString s);
    // Append red error text and pop up.
    void appendError(const QString &text);
    // Append warning error text and pop up.
    void appendWarning(const QString &text);
    // Append a bold command "10:00 " + "Executing: vcs -diff"
    void appendCommand(const QString &text);

protected:
    virtual void contextMenuEvent(QContextMenuEvent *event);

private:
    const QTextCharFormat m_defaultFormat;
    QTextCharFormat m_errorFormat;
    QTextCharFormat m_warningFormat;
    QTextCharFormat m_commandFormat;
};

OutputWindowPlainTextEdit::OutputWindowPlainTextEdit(QWidget *parent) :
    QPlainTextEdit(parent),
    m_defaultFormat(currentCharFormat()),
    m_errorFormat(m_defaultFormat),
    m_warningFormat(m_defaultFormat),
    m_commandFormat(m_defaultFormat)
{
    setReadOnly(true);
    setFrameStyle(QFrame::NoFrame);
    m_errorFormat.setForeground(Qt::red);
    m_warningFormat.setForeground(Qt::darkYellow);
    m_commandFormat.setFontWeight(QFont::Bold);
}

void OutputWindowPlainTextEdit::contextMenuEvent(QContextMenuEvent *event)
{
    QMenu *menu = createStandardContextMenu();
    menu->addSeparator();
    QAction *clearAction = menu->addAction(VCSBaseOutputWindow::tr("Clear"));
    connect(clearAction, SIGNAL(triggered()), this, SLOT(clear()));
    menu->exec(event->globalPos());
    delete menu;
}

void OutputWindowPlainTextEdit::appendLines(QString s)
{
    if (s.isEmpty())
        return;
    // Avoid additional new line character generated by appendPlainText
    if (s.endsWith(QLatin1Char('\n')))
        s.truncate(s.size() - 1);
    appendPlainText(s);
    // Scroll down
    moveCursor(QTextCursor::End);
    ensureCursorVisible();
}

void OutputWindowPlainTextEdit::appendError(const QString &text)
{
    setCurrentCharFormat(m_errorFormat);
    appendLines(text);
    setCurrentCharFormat(m_defaultFormat);
}

void OutputWindowPlainTextEdit::appendWarning(const QString &text)
{
    setCurrentCharFormat(m_warningFormat);
    appendLines(text);
    setCurrentCharFormat(m_defaultFormat);
}

// Append command with new line and log time stamp
void OutputWindowPlainTextEdit::appendCommand(const QString &text)
{
    setCurrentCharFormat(m_commandFormat);
    const QString timeStamp = QTime::currentTime().toString(QLatin1String("\nHH:mm "));
    appendLines(timeStamp + text);
    setCurrentCharFormat(m_defaultFormat);
}

} // namespace Internal

// ------------------- VCSBaseOutputWindowPrivate
struct VCSBaseOutputWindowPrivate {
    static VCSBaseOutputWindow *instance;
    QPointer<Internal::OutputWindowPlainTextEdit> plainTextEdit;
};

VCSBaseOutputWindow *VCSBaseOutputWindowPrivate::instance = 0;

VCSBaseOutputWindow::VCSBaseOutputWindow() :
    d(new VCSBaseOutputWindowPrivate)
{
    VCSBaseOutputWindowPrivate::instance = this;
}

VCSBaseOutputWindow::~VCSBaseOutputWindow()
{
    VCSBaseOutputWindowPrivate::instance = 0;
    delete d;
}

QWidget *VCSBaseOutputWindow::outputWidget(QWidget *parent)
{
    if (!d->plainTextEdit)
        d->plainTextEdit = new Internal::OutputWindowPlainTextEdit(parent);
    return d->plainTextEdit;
}

QWidgetList VCSBaseOutputWindow::toolBarWidgets() const
{
    return QWidgetList();
}

QString VCSBaseOutputWindow::name() const
{
    return tr("Version Control");
}

int VCSBaseOutputWindow::priorityInStatusBar() const
{
    return -1;
}

void VCSBaseOutputWindow::clearContents()
{
    QTC_ASSERT(d->plainTextEdit, return)
    d->plainTextEdit->clear();
}

void VCSBaseOutputWindow::visibilityChanged(bool visible)
{
    if (visible && d->plainTextEdit)
        d->plainTextEdit->setFocus();
}

void VCSBaseOutputWindow::setFocus()
{
}

bool VCSBaseOutputWindow::hasFocus()
{
    return false;
}

bool VCSBaseOutputWindow::canFocus()
{
    return false;
}

bool VCSBaseOutputWindow::canNavigate()
{
    return false;
}

bool VCSBaseOutputWindow::canNext()
{
    return false;
}

bool VCSBaseOutputWindow::canPrevious()
{
    return false;
}

void VCSBaseOutputWindow::goToNext()
{
}

void VCSBaseOutputWindow::goToPrev()
{
}

void VCSBaseOutputWindow::setText(const QString &text)
{
    QTC_ASSERT(d->plainTextEdit, return)
    d->plainTextEdit->setPlainText(text);
}

void VCSBaseOutputWindow::setData(const QByteArray &data)
{
    setText(QTextCodec::codecForLocale()->toUnicode(data));
}

void VCSBaseOutputWindow::appendSilently(const QString &text)
{
    QTC_ASSERT(d->plainTextEdit, return)
    d->plainTextEdit->appendLines(text);
}

void VCSBaseOutputWindow::append(const QString &text)
{
    QTC_ASSERT(d->plainTextEdit, return)
    appendSilently(text);
    // Pop up without focus
    if (!d->plainTextEdit->isVisible())
        popup(false);
}

void VCSBaseOutputWindow::appendError(const QString &text)
{
    QTC_ASSERT(d->plainTextEdit, return)
    d->plainTextEdit->appendError(text);
    if (!d->plainTextEdit->isVisible())
        popup(false); // Pop up without focus
}

void VCSBaseOutputWindow::appendWarning(const QString &text)
{
    QTC_ASSERT(d->plainTextEdit, return)
    d->plainTextEdit->appendWarning(text);
    if (!d->plainTextEdit->isVisible())
        popup(false); // Pop up without focus
}

void VCSBaseOutputWindow::appendCommand(const QString &text)
{
    QTC_ASSERT(d->plainTextEdit, return)
    d->plainTextEdit->appendCommand(text);
}

void VCSBaseOutputWindow::appendData(const QByteArray &data)
{
    QTC_ASSERT(d->plainTextEdit, return)
    appendDataSilently(data);
    if (!d->plainTextEdit->isVisible())
        popup(false); // Pop up without focus
}

void VCSBaseOutputWindow::appendDataSilently(const QByteArray &data)
{
    append(QTextCodec::codecForLocale()->toUnicode(data));
}

VCSBaseOutputWindow *VCSBaseOutputWindow::instance()
{
    if (!VCSBaseOutputWindowPrivate::instance) {
        VCSBaseOutputWindow *w = new VCSBaseOutputWindow;
        Q_UNUSED(w)
    }
    return VCSBaseOutputWindowPrivate::instance;
}

} // namespace VCSBase
