package openapi3filter_test

import (
	"bytes"
	"context"
	"encoding/json"
	"math"
	"math/big"
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/getkin/kin-openapi/openapi3"
	"github.com/getkin/kin-openapi/openapi3filter"
	"github.com/getkin/kin-openapi/routers/gorillamux"
)

func TestIntMax(t *testing.T) {
	spec := `
openapi: 3.0.0
info:
  version: 1.0.0
  title: test large integer value
paths:
  /test:
    post:
      requestBody:
        content:
          application/json:
            schema:
              type: object
              properties:
                testInteger:
                  type: integer
                  format: int64
                testDefault:
                  type: boolean
                  default: false
      responses:
        '200':
          description: Successful response
`[1:]

	loader := openapi3.NewLoader()

	doc, err := loader.LoadFromData([]byte(spec))
	require.NoError(t, err)

	err = doc.Validate(loader.Context)
	require.NoError(t, err)

	router, err := gorillamux.NewRouter(doc)
	require.NoError(t, err)

	testOne := func(value *big.Int, pass bool) {
		valueString := value.String()

		req, err := http.NewRequest(http.MethodPost, "/test", bytes.NewReader([]byte(`{"testInteger":`+valueString+`}`)))
		require.NoError(t, err)
		req.Header.Set("Content-Type", "application/json")

		route, pathParams, err := router.FindRoute(req)
		require.NoError(t, err)

		err = openapi3filter.ValidateRequest(
			context.Background(),
			&openapi3filter.RequestValidationInput{
				Request:    req,
				PathParams: pathParams,
				Route:      route,
			})
		if pass {
			require.NoError(t, err)

			dec := json.NewDecoder(req.Body)
			dec.UseNumber()
			var jsonAfter map[string]any
			err = dec.Decode(&jsonAfter)
			require.NoError(t, err)

			valueAfter := jsonAfter["testInteger"]
			require.IsType(t, json.Number(""), valueAfter)
			assert.Equal(t, valueString, string(valueAfter.(json.Number)))
		} else {
			if assert.Error(t, err) {
				var serr *openapi3.SchemaError
				if assert.ErrorAs(t, err, &serr) {
					assert.Equal(t, "number must be an int64", serr.Reason)
				}
			}
		}
	}

	bigMaxInt64 := big.NewInt(math.MaxInt64)
	bigMaxInt64Plus1 := new(big.Int).Add(bigMaxInt64, big.NewInt(1))
	bigMinInt64 := big.NewInt(math.MinInt64)
	bigMinInt64Minus1 := new(big.Int).Sub(bigMinInt64, big.NewInt(1))

	testOne(bigMaxInt64, true)
	// XXX not yet fixed
	// testOne(bigMaxInt64Plus1, false)
	testOne(bigMaxInt64Plus1, true)
	testOne(bigMinInt64, true)
	// XXX not yet fixed
	// testOne(bigMinInt64Minus1, false)
	testOne(bigMinInt64Minus1, true)
}
