---
id: csrf
title: CSRF
---

CSRF middleware for [Fiber](https://github.com/gofiber/fiber) that provides [Cross-site request forgery](https://en.wikipedia.org/wiki/Cross-site_request_forgery) protection by passing a csrf token via cookies. This cookie value will be used to compare against the client csrf token on requests, other than those defined as "safe" by RFC7231 \(GET, HEAD, OPTIONS, or TRACE\). When the csrf token is invalid, this middleware will return the `fiber.ErrForbidden` error. 

CSRF Tokens are generated on GET requests. You can retrieve the CSRF token with `c.Locals(contextKey)`, where `contextKey` is the string you set in the config (see Custom Config below).

When no `csrf_` cookie is set, or the token has expired, a new token will be generated and `csrf_` cookie set.

:::note
This middleware uses our [Storage](https://github.com/gofiber/storage) package to support various databases through a single interface. The default configuration for this middleware saves data to memory, see the examples below for other databases.
:::

## Signatures

```go
func New(config ...Config) fiber.Handler
```

## Examples

Import the middleware package that is part of the Fiber web framework

```go
import (
    "github.com/gofiber/fiber/v2"
    "github.com/gofiber/fiber/v2/middleware/csrf"
)
```

After you initiate your Fiber app, you can use the following possibilities:

```go
// Initialize default config
app.Use(csrf.New())

// Or extend your config for customization
app.Use(csrf.New(csrf.Config{
    KeyLookup:      "header:X-Csrf-Token",
    CookieName:     "csrf_",
	CookieSameSite: "Lax",
    Expiration:     1 * time.Hour,
    KeyGenerator:   utils.UUID,
    Extractor:      func(c *fiber.Ctx) (string, error) { ... },
}))
```

:::note
KeyLookup will be ignored if Extractor is explicitly set.
:::

## Config

```go
// Config defines the config for middleware.
type Config struct {
    // Next defines a function to skip this middleware when returned true.
    //
    // Optional. Default: nil
    Next func(c *fiber.Ctx) bool

    // KeyLookup is a string in the form of "<source>:<key>" that is used
    // to create an Extractor that extracts the token from the request.
    // Possible values:
    // - "header:<name>"
    // - "query:<name>"
    // - "param:<name>"
    // - "form:<name>"
    // - "cookie:<name>"
    //
    // Ignored if an Extractor is explicitly set.
    //
    // Optional. Default: "header:X-CSRF-Token"
    KeyLookup string

    // Name of the session cookie. This cookie will store session key.
	// Optional. Default value "csrf_".
    CookieName string

    // Domain of the CSRF cookie.
    // Optional. Default value "".
    CookieDomain string

    // Path of the CSRF cookie.
    // Optional. Default value "".
    CookiePath string

    // Indicates if CSRF cookie is secure.
    // Optional. Default value false.
    CookieSecure bool

    // Indicates if CSRF cookie is HTTP only.
    // Optional. Default value false.
    CookieHTTPOnly bool

    // Indicates if CSRF cookie is requested by SameSite.
    // Optional. Default value "Lax".
    CookieSameSite string

    // Decides whether cookie should last for only the browser sesison.
    // Ignores Expiration if set to true
    CookieSessionOnly bool

    // Expiration is the duration before csrf token will expire
    //
    // Optional. Default: 1 * time.Hour
    Expiration time.Duration

    // Store is used to store the state of the middleware
    //
    // Optional. Default: memory.New()
    Storage fiber.Storage

    // Context key to store generated CSRF token into context.
    // If left empty, token will not be stored in context.
    //
    // Optional. Default: ""
    ContextKey string

    // KeyGenerator creates a new CSRF token
    //
    // Optional. Default: utils.UUID
    KeyGenerator func() string

    // Extractor returns the csrf token
    //
    // If set this will be used in place of an Extractor based on KeyLookup.
    //
    // Optional. Default will create an Extractor based on KeyLookup.
    Extractor func(c *fiber.Ctx) (string, error)
}
```

## Default Config

```go
var ConfigDefault = Config{
	KeyLookup:      "header:" + HeaderName,
	CookieName:     "csrf_",
	CookieSameSite: "Lax",
	Expiration:     1 * time.Hour,
	KeyGenerator:   utils.UUID,
	ErrorHandler:   defaultErrorHandler,
	Extractor:      CsrfFromHeader(HeaderName),
}
```

## Constants

```go
const (
    HeaderName = "X-Csrf-Token"
)
```

### Custom Storage/Database

You can use any storage from our [storage](https://github.com/gofiber/storage/) package.

```go
storage := sqlite3.New() // From github.com/gofiber/storage/sqlite3
app.Use(csrf.New(csrf.Config{
	Storage: storage,
}))
```
