// Copyright 2024 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package css provides access to the CSS API.
//
// For product documentation, see: https://developers.google.com/comparison-shopping-services/api/overview
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/css/v1"
//	...
//	ctx := context.Background()
//	cssService, err := css.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	cssService, err := css.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	cssService, err := css.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package css // import "google.golang.org/api/css/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version

const apiId = "css:v1"
const apiName = "css"
const apiVersion = "v1"
const basePath = "https://css.googleapis.com/"
const basePathTemplate = "https://css.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://css.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// Manage your product listings and accounts for Google Shopping
	ContentScope = "https://www.googleapis.com/auth/content"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/content",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s, err := New(client)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	s := &Service{client: client, BasePath: basePath}
	s.Accounts = NewAccountsService(s)
	return s, nil
}

type Service struct {
	client    *http.Client
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Accounts *AccountsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAccountsService(s *Service) *AccountsService {
	rs := &AccountsService{s: s}
	rs.CssProductInputs = NewAccountsCssProductInputsService(s)
	rs.CssProducts = NewAccountsCssProductsService(s)
	rs.Labels = NewAccountsLabelsService(s)
	return rs
}

type AccountsService struct {
	s *Service

	CssProductInputs *AccountsCssProductInputsService

	CssProducts *AccountsCssProductsService

	Labels *AccountsLabelsService
}

func NewAccountsCssProductInputsService(s *Service) *AccountsCssProductInputsService {
	rs := &AccountsCssProductInputsService{s: s}
	return rs
}

type AccountsCssProductInputsService struct {
	s *Service
}

func NewAccountsCssProductsService(s *Service) *AccountsCssProductsService {
	rs := &AccountsCssProductsService{s: s}
	return rs
}

type AccountsCssProductsService struct {
	s *Service
}

func NewAccountsLabelsService(s *Service) *AccountsLabelsService {
	rs := &AccountsLabelsService{s: s}
	return rs
}

type AccountsLabelsService struct {
	s *Service
}

// Account: Information about CSS/MC account.
type Account struct {
	// AccountType: Output only. The type of this account.
	//
	// Possible values:
	//   "ACCOUNT_TYPE_UNSPECIFIED" - Unknown account type.
	//   "CSS_GROUP" - CSS group account.
	//   "CSS_DOMAIN" - CSS domain account.
	//   "MC_PRIMARY_CSS_MCA" - MC Primary CSS MCA account.
	//   "MC_CSS_MCA" - MC CSS MCA account.
	//   "MC_MARKETPLACE_MCA" - MC Marketplace MCA account.
	//   "MC_OTHER_MCA" - MC Other MCA account.
	//   "MC_STANDALONE" - MC Standalone account.
	//   "MC_MCA_SUBACCOUNT" - MC MCA sub-account.
	AccountType string `json:"accountType,omitempty"`
	// AutomaticLabelIds: Automatically created label IDs assigned to the MC
	// account by CSS Center.
	AutomaticLabelIds googleapi.Int64s `json:"automaticLabelIds,omitempty"`
	// DisplayName: The CSS/MC account's short display name.
	DisplayName string `json:"displayName,omitempty"`
	// FullName: Output only. Immutable. The CSS/MC account's full name.
	FullName string `json:"fullName,omitempty"`
	// HomepageUri: Output only. Immutable. The CSS/MC account's homepage.
	HomepageUri string `json:"homepageUri,omitempty"`
	// LabelIds: Manually created label IDs assigned to the CSS/MC account by a CSS
	// parent account.
	LabelIds googleapi.Int64s `json:"labelIds,omitempty"`
	// Name: The label resource name. Format: accounts/{account}
	Name string `json:"name,omitempty"`
	// Parent: The CSS/MC account's parent resource. CSS group for CSS domains; CSS
	// domain for MC accounts. Returned only if the user has access to the parent
	// account.
	Parent string `json:"parent,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccountType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccountType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Account) MarshalJSON() ([]byte, error) {
	type NoMethod Account
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccountLabel: Label assigned by CSS domain or CSS group to one of its
// sub-accounts.
type AccountLabel struct {
	// AccountId: Output only. The ID of account this label belongs to.
	AccountId int64 `json:"accountId,omitempty,string"`
	// Description: The description of this label.
	Description string `json:"description,omitempty"`
	// DisplayName: The display name of this label.
	DisplayName string `json:"displayName,omitempty"`
	// LabelId: Output only. The ID of the label.
	LabelId int64 `json:"labelId,omitempty,string"`
	// LabelType: Output only. The type of this label.
	//
	// Possible values:
	//   "LABEL_TYPE_UNSPECIFIED" - Unknown label type.
	//   "MANUAL" - Indicates that the label was created manually.
	//   "AUTOMATIC" - Indicates that the label was created automatically by CSS
	// Center.
	LabelType string `json:"labelType,omitempty"`
	// Name: The resource name of the label. Format:
	// accounts/{account}/labels/{label}
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccountId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccountId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccountLabel) MarshalJSON() ([]byte, error) {
	type NoMethod AccountLabel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Attributes: Attributes for CSS Product.
type Attributes struct {
	// AdditionalImageLinks: Additional URL of images of the item.
	AdditionalImageLinks []string `json:"additionalImageLinks,omitempty"`
	// Adult: Set to true if the item is targeted towards adults.
	Adult bool `json:"adult,omitempty"`
	// AgeGroup: Target age group of the item.
	AgeGroup string `json:"ageGroup,omitempty"`
	// Brand: Product Related Attributes.[14-36] Brand of the item.
	Brand string `json:"brand,omitempty"`
	// Certifications: A list of certificates claimed by the CSS for the given
	// product.
	Certifications []*Certification `json:"certifications,omitempty"`
	// Color: Color of the item.
	Color string `json:"color,omitempty"`
	// CppAdsRedirect: Allows advertisers to override the item URL when the product
	// is shown within the context of Product Ads.
	CppAdsRedirect string `json:"cppAdsRedirect,omitempty"`
	// CppLink: URL directly linking to your the Product Detail Page of the CSS.
	CppLink string `json:"cppLink,omitempty"`
	// CppMobileLink: URL for the mobile-optimized version of the Product Detail
	// Page of the CSS.
	CppMobileLink string `json:"cppMobileLink,omitempty"`
	// CustomLabel0: Custom label 0 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel0 string `json:"customLabel0,omitempty"`
	// CustomLabel1: Custom label 1 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel1 string `json:"customLabel1,omitempty"`
	// CustomLabel2: Custom label 2 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel2 string `json:"customLabel2,omitempty"`
	// CustomLabel3: Custom label 3 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel3 string `json:"customLabel3,omitempty"`
	// CustomLabel4: Custom label 4 for custom grouping of items in a Shopping
	// campaign.
	CustomLabel4 string `json:"customLabel4,omitempty"`
	// Description: Description of the item.
	Description string `json:"description,omitempty"`
	// ExcludedDestinations: The list of destinations to exclude for this target
	// (corresponds to unchecked check boxes in Merchant Center).
	ExcludedDestinations []string `json:"excludedDestinations,omitempty"`
	// ExpirationDate: Date on which the item should expire, as specified upon
	// insertion, in ISO 8601 (http://en.wikipedia.org/wiki/ISO_8601) format. The
	// actual expiration date is exposed in `productstatuses` as
	// googleExpirationDate (https://support.google.com/merchants/answer/6324499)
	// and might be earlier if `expirationDate` is too far in the future. Note: It
	// may take 2+ days from the expiration date for the item to actually get
	// deleted.
	ExpirationDate string `json:"expirationDate,omitempty"`
	// Gender: Target gender of the item.
	Gender string `json:"gender,omitempty"`
	// GoogleProductCategory: Google's category of the item (see Google product
	// taxonomy (https://support.google.com/merchants/answer/1705911)). When
	// querying products, this field will contain the user provided value. There is
	// currently no way to get back the auto assigned google product categories
	// through the API.
	GoogleProductCategory string `json:"googleProductCategory,omitempty"`
	// Gtin: Global Trade Item Number (GTIN
	// (https://support.google.com/merchants/answer/188494#gtin)) of the item.
	Gtin string `json:"gtin,omitempty"`
	// HeadlineOfferCondition: Condition of the headline offer.
	HeadlineOfferCondition string `json:"headlineOfferCondition,omitempty"`
	// HeadlineOfferLink: Link to the headline offer.
	HeadlineOfferLink string `json:"headlineOfferLink,omitempty"`
	// HeadlineOfferMobileLink: Mobile Link to the headline offer.
	HeadlineOfferMobileLink string `json:"headlineOfferMobileLink,omitempty"`
	// HeadlineOfferPrice: Headline Price of the aggregate offer.
	HeadlineOfferPrice *Price `json:"headlineOfferPrice,omitempty"`
	// HeadlineOfferShippingPrice: Headline Price of the aggregate offer.
	HeadlineOfferShippingPrice *Price `json:"headlineOfferShippingPrice,omitempty"`
	// HighPrice: High Price of the aggregate offer.
	HighPrice *Price `json:"highPrice,omitempty"`
	// ImageLink: URL of an image of the item.
	ImageLink string `json:"imageLink,omitempty"`
	// IncludedDestinations: The list of destinations to include for this target
	// (corresponds to checked check boxes in Merchant Center). Default
	// destinations are always included unless provided in `excludedDestinations`.
	IncludedDestinations []string `json:"includedDestinations,omitempty"`
	// IsBundle: Whether the item is a merchant-defined bundle. A bundle is a
	// custom grouping of different products sold by a merchant for a single price.
	IsBundle bool `json:"isBundle,omitempty"`
	// ItemGroupId: Shared identifier for all variants of the same product.
	ItemGroupId string `json:"itemGroupId,omitempty"`
	// LowPrice: Low Price of the aggregate offer.
	LowPrice *Price `json:"lowPrice,omitempty"`
	// Material: The material of which the item is made.
	Material string `json:"material,omitempty"`
	// Mpn: Manufacturer Part Number (MPN
	// (https://support.google.com/merchants/answer/188494#mpn)) of the item.
	Mpn string `json:"mpn,omitempty"`
	// Multipack: The number of identical products in a merchant-defined multipack.
	Multipack int64 `json:"multipack,omitempty,string"`
	// NumberOfOffers: The number of aggregate offers.
	NumberOfOffers int64 `json:"numberOfOffers,omitempty,string"`
	// Pattern: The item's pattern (e.g. polka dots).
	Pattern string `json:"pattern,omitempty"`
	// Pause: Publication of this item will be temporarily paused.
	Pause string `json:"pause,omitempty"`
	// ProductDetails: Technical specification or additional product details.
	ProductDetails []*ProductDetail `json:"productDetails,omitempty"`
	// ProductHeight: The height of the product in the units provided. The value
	// must be between 0 (exclusive) and 3000 (inclusive).
	ProductHeight *ProductDimension `json:"productHeight,omitempty"`
	// ProductHighlights: Bullet points describing the most relevant highlights of
	// a product.
	ProductHighlights []string `json:"productHighlights,omitempty"`
	// ProductLength: The length of the product in the units provided. The value
	// must be between 0 (exclusive) and 3000 (inclusive).
	ProductLength *ProductDimension `json:"productLength,omitempty"`
	// ProductTypes: Categories of the item (formatted as in products data
	// specification (https://support.google.com/merchants/answer/6324406)).
	ProductTypes []string `json:"productTypes,omitempty"`
	// ProductWeight: The weight of the product in the units provided. The value
	// must be between 0 (exclusive) and 2000 (inclusive).
	ProductWeight *ProductWeight `json:"productWeight,omitempty"`
	// ProductWidth: The width of the product in the units provided. The value must
	// be between 0 (exclusive) and 3000 (inclusive).
	ProductWidth *ProductDimension `json:"productWidth,omitempty"`
	// Size: Size of the item. Only one value is allowed. For variants with
	// different sizes, insert a separate product for each size with the same
	// `itemGroupId` value (see https://support.google.com/merchants/answer/6324492
	// (size definition)).
	Size string `json:"size,omitempty"`
	// SizeSystem: System in which the size is specified. Recommended for apparel
	// items.
	SizeSystem string `json:"sizeSystem,omitempty"`
	// SizeTypes: The cut of the item. It can be used to represent combined size
	// types for apparel items. Maximum two of size types can be provided (see
	// https://support.google.com/merchants/answer/6324497 (size type)).
	SizeTypes []string `json:"sizeTypes,omitempty"`
	// Title: Title of the item.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalImageLinks") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalImageLinks") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Attributes) MarshalJSON() ([]byte, error) {
	type NoMethod Attributes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Certification: The certification for the product. Use the this attribute to
// describe certifications, such as energy efficiency ratings, associated with
// a product.
type Certification struct {
	// Authority: The authority or certification body responsible for issuing the
	// certification. At this time, the most common value is "EC" or
	// “European_Commission” for energy labels in the EU.
	Authority string `json:"authority,omitempty"`
	// Code: The code of the certification. For example, for the EPREL certificate
	// with the link
	// https://eprel.ec.europa.eu/screen/product/dishwashers2019/123456 the code is
	// 123456. The code is required for European Energy Labels.
	Code string `json:"code,omitempty"`
	// Name: The name of the certification. At this time, the most common value is
	// "EPREL", which represents energy efficiency certifications in the EU
	// European Registry for Energy Labeling (EPREL) database.
	Name string `json:"name,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Authority") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Authority") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Certification) MarshalJSON() ([]byte, error) {
	type NoMethod Certification
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CssProduct: The processed CSS Product(a.k.a Aggregate Offer internally).
type CssProduct struct {
	// Attributes: Output only. A list of product attributes.
	Attributes *Attributes `json:"attributes,omitempty"`
	// ContentLanguage: Output only. The two-letter ISO 639-1
	// (http://en.wikipedia.org/wiki/ISO_639-1) language code for the product.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// CssProductStatus: Output only. The status of a product, data validation
	// issues, that is, information about a product computed asynchronously.
	CssProductStatus *CssProductStatus `json:"cssProductStatus,omitempty"`
	// CustomAttributes: Output only. A list of custom (CSS-provided) attributes.
	// It can also be used to submit any attribute of the feed specification in its
	// generic form (for example, `{ "name": "size type", "value": "regular" }`).
	// This is useful for submitting attributes not explicitly exposed by the API,
	// such as additional attributes used for Buy on Google.
	CustomAttributes []*CustomAttribute `json:"customAttributes,omitempty"`
	// FeedLabel: Output only. The feed label for the product.
	FeedLabel string `json:"feedLabel,omitempty"`
	// Name: The name of the CSS Product. Format:
	// "accounts/{account}/cssProducts/{css_product}"
	Name string `json:"name,omitempty"`
	// RawProvidedId: Output only. Your unique raw identifier for the product.
	RawProvidedId string `json:"rawProvidedId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Attributes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Attributes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CssProduct) MarshalJSON() ([]byte, error) {
	type NoMethod CssProduct
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CssProductInput: This resource represents input data you submit for a CSS
// Product, not the processed CSS Product that you see in CSS Center, in
// Shopping Ads, or across Google surfaces.
type CssProductInput struct {
	// Attributes: A list of CSS Product attributes.
	Attributes *Attributes `json:"attributes,omitempty"`
	// ContentLanguage: Required. The two-letter ISO 639-1
	// (http://en.wikipedia.org/wiki/ISO_639-1) language code for the CSS Product.
	ContentLanguage string `json:"contentLanguage,omitempty"`
	// CustomAttributes: A list of custom (CSS-provided) attributes. It can also be
	// used for submitting any attribute of the feed specification in its generic
	// form (for example: `{ "name": "size type", "value": "regular" }`). This is
	// useful for submitting attributes not explicitly exposed by the API, such as
	// additional attributes used for Buy on Google.
	CustomAttributes []*CustomAttribute `json:"customAttributes,omitempty"`
	// FeedLabel: Required. The feed label
	// (https://developers.google.com/shopping-content/guides/products/feed-labels)
	// for the CSS Product. Feed Label is synonymous to "target country" and hence
	// should always be a valid region code. For example: 'DE' for Germany, 'FR'
	// for France.
	FeedLabel string `json:"feedLabel,omitempty"`
	// FinalName: Output only. The name of the processed CSS Product. Format:
	// `accounts/{account}/cssProducts/{css_product}` "
	FinalName string `json:"finalName,omitempty"`
	// FreshnessTime: Represents the existing version (freshness) of the CSS
	// Product, which can be used to preserve the right order when multiple updates
	// are done at the same time. This field must not be set to the future time. If
	// set, the update is prevented if a newer version of the item already exists
	// in our system (that is the last update time of the existing CSS products is
	// later than the freshness time set in the update). If the update happens, the
	// last update time is then set to this freshness time. If not set, the update
	// will not be prevented and the last update time will default to when this
	// request was received by the CSS API. If the operation is prevented, the
	// aborted exception will be thrown.
	FreshnessTime string `json:"freshnessTime,omitempty"`
	// Name: The name of the CSS Product input. Format:
	// `accounts/{account}/cssProductInputs/{css_product_input}`
	Name string `json:"name,omitempty"`
	// RawProvidedId: Required. Your unique identifier for the CSS Product. This is
	// the same for the CSS Product input and processed CSS Product. We only allow
	// ids with alphanumerics, underscores and dashes. See the products feed
	// specification (https://support.google.com/merchants/answer/188494#id) for
	// details.
	RawProvidedId string `json:"rawProvidedId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Attributes") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Attributes") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CssProductInput) MarshalJSON() ([]byte, error) {
	type NoMethod CssProductInput
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CssProductStatus: The status of the Css Product, data validation issues,
// that is, information about the Css Product computed asynchronously.
type CssProductStatus struct {
	// CreationDate: Date on which the item has been created, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	CreationDate string `json:"creationDate,omitempty"`
	// DestinationStatuses: The intended destinations for the product.
	DestinationStatuses []*DestinationStatus `json:"destinationStatuses,omitempty"`
	// GoogleExpirationDate: Date on which the item expires, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	GoogleExpirationDate string `json:"googleExpirationDate,omitempty"`
	// ItemLevelIssues: A list of all issues associated with the product.
	ItemLevelIssues []*ItemLevelIssue `json:"itemLevelIssues,omitempty"`
	// LastUpdateDate: Date on which the item has been last updated, in ISO 8601
	// (http://en.wikipedia.org/wiki/ISO_8601) format.
	LastUpdateDate string `json:"lastUpdateDate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CreationDate") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreationDate") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CssProductStatus) MarshalJSON() ([]byte, error) {
	type NoMethod CssProductStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CustomAttribute: A message that represents custom attributes. Exactly one of
// `value` or `group_values` must not be empty.
type CustomAttribute struct {
	// GroupValues: Subattributes within this attribute group. If `group_values` is
	// not empty, `value` must be empty.
	GroupValues []*CustomAttribute `json:"groupValues,omitempty"`
	// Name: The name of the attribute.
	Name string `json:"name,omitempty"`
	// Value: The value of the attribute. If `value` is not empty, `group_values`
	// must be empty.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "GroupValues") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "GroupValues") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CustomAttribute) MarshalJSON() ([]byte, error) {
	type NoMethod CustomAttribute
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DestinationStatus: The destination status of the product status.
type DestinationStatus struct {
	// ApprovedCountries: List of country codes (ISO 3166-1 alpha-2) where the
	// aggregate offer is approved.
	ApprovedCountries []string `json:"approvedCountries,omitempty"`
	// Destination: The name of the destination
	Destination string `json:"destination,omitempty"`
	// DisapprovedCountries: List of country codes (ISO 3166-1 alpha-2) where the
	// aggregate offer is disapproved.
	DisapprovedCountries []string `json:"disapprovedCountries,omitempty"`
	// PendingCountries: List of country codes (ISO 3166-1 alpha-2) where the
	// aggregate offer is pending approval.
	PendingCountries []string `json:"pendingCountries,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ApprovedCountries") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApprovedCountries") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DestinationStatus) MarshalJSON() ([]byte, error) {
	type NoMethod DestinationStatus
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// ItemLevelIssue: The ItemLevelIssue of the product status.
type ItemLevelIssue struct {
	// ApplicableCountries: List of country codes (ISO 3166-1 alpha-2) where issue
	// applies to the aggregate offer.
	ApplicableCountries []string `json:"applicableCountries,omitempty"`
	// Attribute: The attribute's name, if the issue is caused by a single
	// attribute.
	Attribute string `json:"attribute,omitempty"`
	// Code: The error code of the issue.
	Code string `json:"code,omitempty"`
	// Description: A short issue description in English.
	Description string `json:"description,omitempty"`
	// Destination: The destination the issue applies to.
	Destination string `json:"destination,omitempty"`
	// Detail: A detailed issue description in English.
	Detail string `json:"detail,omitempty"`
	// Documentation: The URL of a web page to help with resolving this issue.
	Documentation string `json:"documentation,omitempty"`
	// Resolution: Whether the issue can be resolved by the merchant.
	Resolution string `json:"resolution,omitempty"`
	// Servability: How this issue affects serving of the aggregate offer.
	Servability string `json:"servability,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ApplicableCountries") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApplicableCountries") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ItemLevelIssue) MarshalJSON() ([]byte, error) {
	type NoMethod ItemLevelIssue
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListAccountLabelsResponse: Response message for the `ListAccountLabels`
// method.
type ListAccountLabelsResponse struct {
	// AccountLabels: The labels from the specified account.
	AccountLabels []*AccountLabel `json:"accountLabels,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccountLabels") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccountLabels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListAccountLabelsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListAccountLabelsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListChildAccountsResponse: Response message for the `ListChildAccounts`
// method.
type ListChildAccountsResponse struct {
	// Accounts: The CSS/MC accounts returned for the specified CSS parent account.
	Accounts []*Account `json:"accounts,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Accounts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Accounts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListChildAccountsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListChildAccountsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListCssProductsResponse: Response message for the ListCssProducts method.
type ListCssProductsResponse struct {
	// CssProducts: The processed CSS products from the specified account. These
	// are your processed CSS products after applying rules and supplemental feeds.
	CssProducts []*CssProduct `json:"cssProducts,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CssProducts") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CssProducts") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListCssProductsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListCssProductsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Price: The price represented as a number and currency.
type Price struct {
	// AmountMicros: The price represented as a number in micros (1 million micros
	// is an equivalent to one's currency standard unit, for example, 1 USD =
	// 1000000 micros).
	AmountMicros int64 `json:"amountMicros,omitempty,string"`
	// CurrencyCode: The currency of the price using three-letter acronyms
	// according to ISO 4217 (http://en.wikipedia.org/wiki/ISO_4217).
	CurrencyCode string `json:"currencyCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AmountMicros") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AmountMicros") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Price) MarshalJSON() ([]byte, error) {
	type NoMethod Price
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductDetail: The product details.
type ProductDetail struct {
	// AttributeName: The name of the product detail.
	AttributeName string `json:"attributeName,omitempty"`
	// AttributeValue: The value of the product detail.
	AttributeValue string `json:"attributeValue,omitempty"`
	// SectionName: The section header used to group a set of product details.
	SectionName string `json:"sectionName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AttributeName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AttributeName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductDetail) MarshalJSON() ([]byte, error) {
	type NoMethod ProductDetail
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ProductDimension: The dimension of the product.
type ProductDimension struct {
	// Unit: Required. The dimension units. Acceptable values are: * "in" *
	// "cm"
	Unit string `json:"unit,omitempty"`
	// Value: Required. The dimension value represented as a number. The value can
	// have a maximum precision of four decimal places.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductDimension) MarshalJSON() ([]byte, error) {
	type NoMethod ProductDimension
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductDimension) UnmarshalJSON(data []byte) error {
	type NoMethod ProductDimension
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// ProductWeight: The weight of the product.
type ProductWeight struct {
	// Unit: Required. The weight unit. Acceptable values are: * "g" * "kg" *
	// "oz" * "lb"
	Unit string `json:"unit,omitempty"`
	// Value: Required. The weight represented as a number. The weight can have a
	// maximum precision of four decimal places.
	Value float64 `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Unit") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Unit") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ProductWeight) MarshalJSON() ([]byte, error) {
	type NoMethod ProductWeight
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *ProductWeight) UnmarshalJSON(data []byte) error {
	type NoMethod ProductWeight
	var s1 struct {
		Value gensupport.JSONFloat64 `json:"value"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Value = float64(s1.Value)
	return nil
}

// UpdateAccountLabelsRequest: The request message for the `UpdateLabels`
// method.
type UpdateAccountLabelsRequest struct {
	// LabelIds: The list of label IDs to overwrite the existing account label IDs.
	// If the list is empty, all currently assigned label IDs will be deleted.
	LabelIds googleapi.Int64s `json:"labelIds,omitempty"`
	// Parent: Optional. Only required when updating MC account labels. The CSS
	// domain that is the parent resource of the MC account. Format:
	// accounts/{account}
	Parent string `json:"parent,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LabelIds") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LabelIds") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UpdateAccountLabelsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod UpdateAccountLabelsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AccountsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves a single CSS/MC account by ID.
//
// - name: The name of the managed CSS/MC account. Format: accounts/{account}.
func (r *AccountsService) Get(name string) *AccountsGetCall {
	c := &AccountsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Parent sets the optional parameter "parent": Only required when retrieving
// MC account information. The CSS domain that is the parent resource of the MC
// account. Format: accounts/{account}
func (c *AccountsGetCall) Parent(parent string) *AccountsGetCall {
	c.urlParams_.Set("parent", parent)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsGetCall) Fields(s ...googleapi.Field) *AccountsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsGetCall) IfNoneMatch(entityTag string) *AccountsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsGetCall) Context(ctx context.Context) *AccountsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Account.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsGetCall) Do(opts ...googleapi.CallOption) (*Account, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Account{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsListChildAccountsCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ListChildAccounts: Lists all the accounts under the specified CSS account
// ID, and optionally filters by label ID and account name.
//
//   - parent: The parent account. Must be a CSS group or domain. Format:
//     accounts/{account}.
func (r *AccountsService) ListChildAccounts(parent string) *AccountsListChildAccountsCall {
	c := &AccountsListChildAccountsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// FullName sets the optional parameter "fullName": If set, only the MC
// accounts with the given name (case sensitive) will be returned.
func (c *AccountsListChildAccountsCall) FullName(fullName string) *AccountsListChildAccountsCall {
	c.urlParams_.Set("fullName", fullName)
	return c
}

// LabelId sets the optional parameter "labelId": If set, only the MC accounts
// with the given label ID will be returned.
func (c *AccountsListChildAccountsCall) LabelId(labelId int64) *AccountsListChildAccountsCall {
	c.urlParams_.Set("labelId", fmt.Sprint(labelId))
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// accounts to return. The service may return fewer than this value. If
// unspecified, at most 50 accounts will be returned. The maximum value is
// 1000; values above 1000 will be coerced to 1000.
func (c *AccountsListChildAccountsCall) PageSize(pageSize int64) *AccountsListChildAccountsCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListChildAccounts` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListChildAccounts` must match the call that provided the page token.
func (c *AccountsListChildAccountsCall) PageToken(pageToken string) *AccountsListChildAccountsCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsListChildAccountsCall) Fields(s ...googleapi.Field) *AccountsListChildAccountsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsListChildAccountsCall) IfNoneMatch(entityTag string) *AccountsListChildAccountsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsListChildAccountsCall) Context(ctx context.Context) *AccountsListChildAccountsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsListChildAccountsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsListChildAccountsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}:listChildAccounts")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.listChildAccounts" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListChildAccountsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsListChildAccountsCall) Do(opts ...googleapi.CallOption) (*ListChildAccountsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListChildAccountsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsListChildAccountsCall) Pages(ctx context.Context, f func(*ListChildAccountsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsUpdateLabelsCall struct {
	s                          *Service
	name                       string
	updateaccountlabelsrequest *UpdateAccountLabelsRequest
	urlParams_                 gensupport.URLParams
	ctx_                       context.Context
	header_                    http.Header
}

// UpdateLabels: Updates labels assigned to CSS/MC accounts by a CSS domain.
//
// - name: The label resource name. Format: accounts/{account}.
func (r *AccountsService) UpdateLabels(name string, updateaccountlabelsrequest *UpdateAccountLabelsRequest) *AccountsUpdateLabelsCall {
	c := &AccountsUpdateLabelsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.updateaccountlabelsrequest = updateaccountlabelsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsUpdateLabelsCall) Fields(s ...googleapi.Field) *AccountsUpdateLabelsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsUpdateLabelsCall) Context(ctx context.Context) *AccountsUpdateLabelsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsUpdateLabelsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsUpdateLabelsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.updateaccountlabelsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:updateLabels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.updateLabels" call.
// Any non-2xx status code is an error. Response headers are in either
// *Account.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsUpdateLabelsCall) Do(opts ...googleapi.CallOption) (*Account, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Account{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsCssProductInputsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a CSS Product input from your CSS Center account. After a
// delete it may take several minutes until the input is no longer available.
//
//   - name: The name of the CSS product input resource to delete. Format:
//     accounts/{account}/cssProductInputs/{css_product_input}.
func (r *AccountsCssProductInputsService) Delete(name string) *AccountsCssProductInputsDeleteCall {
	c := &AccountsCssProductInputsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// SupplementalFeedId sets the optional parameter "supplementalFeedId": The
// Content API Supplemental Feed ID. The field must not be set if the action
// applies to a primary feed. If the field is set, then product action applies
// to a supplemental feed instead of primary Content API feed.
func (c *AccountsCssProductInputsDeleteCall) SupplementalFeedId(supplementalFeedId int64) *AccountsCssProductInputsDeleteCall {
	c.urlParams_.Set("supplementalFeedId", fmt.Sprint(supplementalFeedId))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsCssProductInputsDeleteCall) Fields(s ...googleapi.Field) *AccountsCssProductInputsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsCssProductInputsDeleteCall) Context(ctx context.Context) *AccountsCssProductInputsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsCssProductInputsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsCssProductInputsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.cssProductInputs.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsCssProductInputsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsCssProductInputsInsertCall struct {
	s               *Service
	parent          string
	cssproductinput *CssProductInput
	urlParams_      gensupport.URLParams
	ctx_            context.Context
	header_         http.Header
}

// Insert: Uploads a CssProductInput to your CSS Center account. If an input
// with the same contentLanguage, identity, feedLabel and feedId already
// exists, this method replaces that entry. After inserting, updating, or
// deleting a CSS Product input, it may take several minutes before the
// processed CSS Product can be retrieved.
//
//   - parent: The account where this CSS Product will be inserted. Format:
//     accounts/{account}.
func (r *AccountsCssProductInputsService) Insert(parent string, cssproductinput *CssProductInput) *AccountsCssProductInputsInsertCall {
	c := &AccountsCssProductInputsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.cssproductinput = cssproductinput
	return c
}

// FeedId sets the optional parameter "feedId": Required. The primary or
// supplemental feed id. If CSS Product already exists and feed id provided is
// different, then the CSS Product will be moved to a new feed. Note: For now,
// CSSs do not need to provide feed ids as we create feeds on the fly. We do
// not have supplemental feed support for CSS Products yet.
func (c *AccountsCssProductInputsInsertCall) FeedId(feedId int64) *AccountsCssProductInputsInsertCall {
	c.urlParams_.Set("feedId", fmt.Sprint(feedId))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsCssProductInputsInsertCall) Fields(s ...googleapi.Field) *AccountsCssProductInputsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsCssProductInputsInsertCall) Context(ctx context.Context) *AccountsCssProductInputsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsCssProductInputsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsCssProductInputsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.cssproductinput)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/cssProductInputs:insert")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.cssProductInputs.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *CssProductInput.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *AccountsCssProductInputsInsertCall) Do(opts ...googleapi.CallOption) (*CssProductInput, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CssProductInput{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsCssProductsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves the processed CSS Product from your CSS Center account. After
// inserting, updating, or deleting a product input, it may take several
// minutes before the updated final product can be retrieved.
//
// - name: The name of the CSS product to retrieve.
func (r *AccountsCssProductsService) Get(name string) *AccountsCssProductsGetCall {
	c := &AccountsCssProductsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsCssProductsGetCall) Fields(s ...googleapi.Field) *AccountsCssProductsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsCssProductsGetCall) IfNoneMatch(entityTag string) *AccountsCssProductsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsCssProductsGetCall) Context(ctx context.Context) *AccountsCssProductsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsCssProductsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsCssProductsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.cssProducts.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *CssProduct.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsCssProductsGetCall) Do(opts ...googleapi.CallOption) (*CssProduct, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CssProduct{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsCssProductsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the processed CSS Products in your CSS Center account. The
// response might contain fewer items than specified by pageSize. Rely on
// pageToken to determine if there are more items to be requested. After
// inserting, updating, or deleting a CSS product input, it may take several
// minutes before the updated processed CSS product can be retrieved.
//
//   - parent: The account/domain to list processed CSS Products for. Format:
//     accounts/{account}.
func (r *AccountsCssProductsService) List(parent string) *AccountsCssProductsListCall {
	c := &AccountsCssProductsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of CSS
// Products to return. The service may return fewer than this value. The
// maximum value is 1000; values above 1000 will be coerced to 1000. If
// unspecified, the maximum number of CSS products will be returned.
func (c *AccountsCssProductsListCall) PageSize(pageSize int64) *AccountsCssProductsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListCssProducts` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListCssProducts` must match the call that provided the page token.
func (c *AccountsCssProductsListCall) PageToken(pageToken string) *AccountsCssProductsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsCssProductsListCall) Fields(s ...googleapi.Field) *AccountsCssProductsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsCssProductsListCall) IfNoneMatch(entityTag string) *AccountsCssProductsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsCssProductsListCall) Context(ctx context.Context) *AccountsCssProductsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsCssProductsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsCssProductsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/cssProducts")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.cssProducts.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListCssProductsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsCssProductsListCall) Do(opts ...googleapi.CallOption) (*ListCssProductsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListCssProductsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsCssProductsListCall) Pages(ctx context.Context, f func(*ListCssProductsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsLabelsCreateCall struct {
	s            *Service
	parent       string
	accountlabel *AccountLabel
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Create: Creates a new label, not assigned to any account.
//
// - parent: The parent account. Format: accounts/{account}.
func (r *AccountsLabelsService) Create(parent string, accountlabel *AccountLabel) *AccountsLabelsCreateCall {
	c := &AccountsLabelsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.accountlabel = accountlabel
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLabelsCreateCall) Fields(s ...googleapi.Field) *AccountsLabelsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLabelsCreateCall) Context(ctx context.Context) *AccountsLabelsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLabelsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLabelsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.accountlabel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/labels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.labels.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccountLabel.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsLabelsCreateCall) Do(opts ...googleapi.CallOption) (*AccountLabel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccountLabel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsLabelsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a label and removes it from all accounts to which it was
// assigned.
//
//   - name: The name of the label to delete. Format:
//     accounts/{account}/labels/{label}.
func (r *AccountsLabelsService) Delete(name string) *AccountsLabelsDeleteCall {
	c := &AccountsLabelsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLabelsDeleteCall) Fields(s ...googleapi.Field) *AccountsLabelsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLabelsDeleteCall) Context(ctx context.Context) *AccountsLabelsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLabelsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLabelsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.labels.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsLabelsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

type AccountsLabelsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the labels assigned to an account.
//
// - parent: The parent account. Format: accounts/{account}.
func (r *AccountsLabelsService) List(parent string) *AccountsLabelsListCall {
	c := &AccountsLabelsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// labels to return. The service may return fewer than this value. If
// unspecified, at most 50 labels will be returned. The maximum value is 1000;
// values above 1000 will be coerced to 1000.
func (c *AccountsLabelsListCall) PageSize(pageSize int64) *AccountsLabelsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListAccountLabels` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListAccountLabels` must match the call that provided the page token.
func (c *AccountsLabelsListCall) PageToken(pageToken string) *AccountsLabelsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLabelsListCall) Fields(s ...googleapi.Field) *AccountsLabelsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AccountsLabelsListCall) IfNoneMatch(entityTag string) *AccountsLabelsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLabelsListCall) Context(ctx context.Context) *AccountsLabelsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLabelsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLabelsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	var body io.Reader = nil
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/labels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.labels.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListAccountLabelsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *AccountsLabelsListCall) Do(opts ...googleapi.CallOption) (*ListAccountLabelsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListAccountLabelsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *AccountsLabelsListCall) Pages(ctx context.Context, f func(*ListAccountLabelsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type AccountsLabelsPatchCall struct {
	s            *Service
	name         string
	accountlabel *AccountLabel
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Patch: Updates a label.
//
//   - name: The resource name of the label. Format:
//     accounts/{account}/labels/{label}.
func (r *AccountsLabelsService) Patch(name string, accountlabel *AccountLabel) *AccountsLabelsPatchCall {
	c := &AccountsLabelsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.accountlabel = accountlabel
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AccountsLabelsPatchCall) Fields(s ...googleapi.Field) *AccountsLabelsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AccountsLabelsPatchCall) Context(ctx context.Context) *AccountsLabelsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AccountsLabelsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AccountsLabelsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	var body io.Reader = nil
	body, err := googleapi.WithoutDataWrapper.JSONReader(c.accountlabel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "css.accounts.labels.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccountLabel.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AccountsLabelsPatchCall) Do(opts ...googleapi.CallOption) (*AccountLabel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccountLabel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	if err := gensupport.DecodeResponse(target, res); err != nil {
		return nil, err
	}
	return ret, nil
}
