// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package addressvalidation provides access to the Address Validation API.
//
// For product documentation, see: https://developers.google.com/maps/documentation/addressvalidation
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/addressvalidation/v1"
//	...
//	ctx := context.Background()
//	addressvalidationService, err := addressvalidation.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	addressvalidationService, err := addressvalidation.NewService(ctx, option.WithScopes(addressvalidation.MapsPlatformAddressvalidationScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	addressvalidationService, err := addressvalidation.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	addressvalidationService, err := addressvalidation.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package addressvalidation // import "google.golang.org/api/addressvalidation/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "addressvalidation:v1"
const apiName = "addressvalidation"
const apiVersion = "v1"
const basePath = "https://addressvalidation.googleapis.com/"
const basePathTemplate = "https://addressvalidation.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://addressvalidation.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"

	// Private Service:
	// https://www.googleapis.com/auth/maps-platform.addressvalidation
	MapsPlatformAddressvalidationScope = "https://www.googleapis.com/auth/maps-platform.addressvalidation"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/cloud-platform",
		"https://www.googleapis.com/auth/maps-platform.addressvalidation",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.V1 = NewV1Service(s)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.Background(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	V1 *V1Service
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewV1Service(s *Service) *V1Service {
	rs := &V1Service{s: s}
	return rs
}

type V1Service struct {
	s *Service
}

// GoogleGeoTypeViewport: A latitude-longitude viewport, represented as two
// diagonally opposite `low` and `high` points. A viewport is considered a
// closed region, i.e. it includes its boundary. The latitude bounds must range
// between -90 to 90 degrees inclusive, and the longitude bounds must range
// between -180 to 180 degrees inclusive. Various cases include: - If `low` =
// `high`, the viewport consists of that single point. - If `low.longitude` >
// `high.longitude`, the longitude range is inverted (the viewport crosses the
// 180 degree longitude line). - If `low.longitude` = -180 degrees and
// `high.longitude` = 180 degrees, the viewport includes all longitudes. - If
// `low.longitude` = 180 degrees and `high.longitude` = -180 degrees, the
// longitude range is empty. - If `low.latitude` > `high.latitude`, the
// latitude range is empty. Both `low` and `high` must be populated, and the
// represented box cannot be empty (as specified by the definitions above). An
// empty viewport will result in an error. For example, this viewport fully
// encloses New York City: { "low": { "latitude": 40.477398, "longitude":
// -74.259087 }, "high": { "latitude": 40.91618, "longitude": -73.70018 } }
type GoogleGeoTypeViewport struct {
	// High: Required. The high point of the viewport.
	High *GoogleTypeLatLng `json:"high,omitempty"`
	// Low: Required. The low point of the viewport.
	Low *GoogleTypeLatLng `json:"low,omitempty"`
	// ForceSendFields is a list of field names (e.g. "High") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "High") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleGeoTypeViewport) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleGeoTypeViewport
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1Address: Details of the post-processed address.
// Post-processing includes correcting misspelled parts of the address,
// replacing incorrect parts, and inferring missing parts.
type GoogleMapsAddressvalidationV1Address struct {
	// AddressComponents: Unordered list. The individual address components of the
	// formatted and corrected address, along with validation information. This
	// provides information on the validation status of the individual components.
	// Address components are not ordered in a particular way. Do not make any
	// assumptions on the ordering of the address components in the list.
	AddressComponents []*GoogleMapsAddressvalidationV1AddressComponent `json:"addressComponents,omitempty"`
	// FormattedAddress: The post-processed address, formatted as a single-line
	// address following the address formatting rules of the region where the
	// address is located. Note: the format of this address may not match the
	// format of the address in the `postal_address` field. For example, the
	// `postal_address` always represents the country as a 2 letter `region_code`,
	// such as "US" or "NZ". By contrast, this field uses a longer form of the
	// country name, such as "USA" or "New Zealand".
	FormattedAddress string `json:"formattedAddress,omitempty"`
	// MissingComponentTypes: The types of components that were expected to be
	// present in a correctly formatted mailing address but were not found in the
	// input AND could not be inferred. Components of this type are not present in
	// `formatted_address`, `postal_address`, or `address_components`. An example
	// might be `['street_number', 'route']` for an input like "Boulder, Colorado,
	// 80301, USA". The list of possible types can be found here
	// (https://developers.google.com/maps/documentation/geocoding/requests-geocoding#Types).
	MissingComponentTypes []string `json:"missingComponentTypes,omitempty"`
	// PostalAddress: The post-processed address represented as a postal address.
	PostalAddress *GoogleTypePostalAddress `json:"postalAddress,omitempty"`
	// UnconfirmedComponentTypes: The types of the components that are present in
	// the `address_components` but could not be confirmed to be correct. This
	// field is provided for the sake of convenience: its contents are equivalent
	// to iterating through the `address_components` to find the types of all the
	// components where the confirmation_level is not CONFIRMED or the inferred
	// flag is not set to `true`. The list of possible types can be found here
	// (https://developers.google.com/maps/documentation/geocoding/requests-geocoding#Types).
	UnconfirmedComponentTypes []string `json:"unconfirmedComponentTypes,omitempty"`
	// UnresolvedTokens: Any tokens in the input that could not be resolved. This
	// might be an input that was not recognized as a valid part of an address. For
	// example, for an input such as "Parcel 0000123123 & 0000456456 Str # Guthrie
	// Center IA 50115 US", the unresolved tokens might look like `["Parcel",
	// "0000123123", "&", "0000456456"]`.
	UnresolvedTokens []string `json:"unresolvedTokens,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AddressComponents") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AddressComponents") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1Address) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1Address
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1AddressComponent: Represents an address
// component, such as a street, city, or state.
type GoogleMapsAddressvalidationV1AddressComponent struct {
	// ComponentName: The name for this component.
	ComponentName *GoogleMapsAddressvalidationV1ComponentName `json:"componentName,omitempty"`
	// ComponentType: The type of the address component. See Table 2: Additional
	// types returned by the Places service
	// (https://developers.google.com/places/web-service/supported_types#table2)
	// for a list of possible types.
	ComponentType string `json:"componentType,omitempty"`
	// ConfirmationLevel: Indicates the level of certainty that we have that the
	// component is correct.
	//
	// Possible values:
	//   "CONFIRMATION_LEVEL_UNSPECIFIED" - Default value. This value is unused.
	//   "CONFIRMED" - We were able to verify that this component exists and makes
	// sense in the context of the rest of the address.
	//   "UNCONFIRMED_BUT_PLAUSIBLE" - This component could not be confirmed, but
	// it is plausible that it exists. For example, a street number within a known
	// valid range of numbers on a street where specific house numbers are not
	// known.
	//   "UNCONFIRMED_AND_SUSPICIOUS" - This component was not confirmed and is
	// likely to be wrong. For example, a neighborhood that does not fit the rest
	// of the address.
	ConfirmationLevel string `json:"confirmationLevel,omitempty"`
	// Inferred: Indicates that the component was not part of the input, but we
	// inferred it for the address location and believe it should be provided for a
	// complete address.
	Inferred bool `json:"inferred,omitempty"`
	// Replaced: Indicates the name of the component was replaced with a completely
	// different one, for example a wrong postal code being replaced with one that
	// is correct for the address. This is not a cosmetic change, the input
	// component has been changed to a different one.
	Replaced bool `json:"replaced,omitempty"`
	// SpellCorrected: Indicates a correction to a misspelling in the component
	// name. The API does not always flag changes from one spelling variant to
	// another, such as when changing "centre" to "center". It also does not always
	// flag common misspellings, such as when changing "Amphitheater Pkwy" to
	// "Amphitheatre Pkwy".
	SpellCorrected bool `json:"spellCorrected,omitempty"`
	// Unexpected: Indicates an address component that is not expected to be
	// present in a postal address for the given region. We have retained it only
	// because it was part of the input.
	Unexpected bool `json:"unexpected,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ComponentName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ComponentName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1AddressComponent) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1AddressComponent
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1AddressMetadata: The metadata for the address.
// `metadata` is not guaranteed to be fully populated for every address sent to
// the Address Validation API.
type GoogleMapsAddressvalidationV1AddressMetadata struct {
	// Business: Indicates that this is the address of a business. If unset,
	// indicates that the value is unknown.
	Business bool `json:"business,omitempty"`
	// PoBox: Indicates that the address of a PO box. If unset, indicates that the
	// value is unknown.
	PoBox bool `json:"poBox,omitempty"`
	// Residential: Indicates that this is the address of a residence. If unset,
	// indicates that the value is unknown.
	Residential bool `json:"residential,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Business") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Business") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1AddressMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1AddressMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1ComponentName: A wrapper for the name of the
// component.
type GoogleMapsAddressvalidationV1ComponentName struct {
	// LanguageCode: The BCP-47 language code. This will not be present if the
	// component name is not associated with a language, such as a street number.
	LanguageCode string `json:"languageCode,omitempty"`
	// Text: The name text. For example, "5th Avenue" for a street name or "1253"
	// for a street number.
	Text string `json:"text,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LanguageCode") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LanguageCode") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1ComponentName) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1ComponentName
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1Geocode: Contains information about the place
// the input was geocoded to.
type GoogleMapsAddressvalidationV1Geocode struct {
	// Bounds: The bounds of the geocoded place.
	Bounds *GoogleGeoTypeViewport `json:"bounds,omitempty"`
	// FeatureSizeMeters: The size of the geocoded place, in meters. This is
	// another measure of the coarseness of the geocoded location, but in physical
	// size rather than in semantic meaning.
	FeatureSizeMeters float64 `json:"featureSizeMeters,omitempty"`
	// Location: The geocoded location of the input. Using place IDs is preferred
	// over using addresses, latitude/longitude coordinates, or plus codes. Using
	// coordinates when routing or calculating driving directions will always
	// result in the point being snapped to the road nearest to those coordinates.
	// This may not be a road that will quickly or safely lead to the destination
	// and may not be near an access point to the property. Additionally, when a
	// location is reverse geocoded, there is no guarantee that the returned
	// address will match the original.
	Location *GoogleTypeLatLng `json:"location,omitempty"`
	// PlaceId: The PlaceID of the place this input geocodes to. For more
	// information about Place IDs see here
	// (https://developers.google.com/maps/documentation/places/web-service/place-id).
	PlaceId string `json:"placeId,omitempty"`
	// PlaceTypes: The type(s) of place that the input geocoded to. For example,
	// `['locality', 'political']`. The full list of types can be found here
	// (https://developers.google.com/maps/documentation/geocoding/requests-geocoding#Types).
	PlaceTypes []string `json:"placeTypes,omitempty"`
	// PlusCode: The plus code corresponding to the `location`.
	PlusCode *GoogleMapsAddressvalidationV1PlusCode `json:"plusCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Bounds") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Bounds") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1Geocode) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1Geocode
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *GoogleMapsAddressvalidationV1Geocode) UnmarshalJSON(data []byte) error {
	type NoMethod GoogleMapsAddressvalidationV1Geocode
	var s1 struct {
		FeatureSizeMeters gensupport.JSONFloat64 `json:"featureSizeMeters"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.FeatureSizeMeters = float64(s1.FeatureSizeMeters)
	return nil
}

// GoogleMapsAddressvalidationV1LanguageOptions: Preview: This feature is in
// Preview (pre-GA). Pre-GA products and features might have limited support,
// and changes to pre-GA products and features might not be compatible with
// other pre-GA versions. Pre-GA Offerings are covered by the Google Maps
// Platform Service Specific Terms
// (https://cloud.google.com/maps-platform/terms/maps-service-terms). For more
// information, see the launch stage descriptions
// (https://developers.google.com/maps/launch-stages). Enables the Address
// Validation API to include additional information in the response.
type GoogleMapsAddressvalidationV1LanguageOptions struct {
	// ReturnEnglishLatinAddress: Preview: Return a
	// [google.maps.addressvalidation.v1.Address] in English. See
	// [google.maps.addressvalidation.v1.ValidationResult.english_latin_address]
	// for details.
	ReturnEnglishLatinAddress bool `json:"returnEnglishLatinAddress,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ReturnEnglishLatinAddress")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ReturnEnglishLatinAddress") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1LanguageOptions) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1LanguageOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1PlusCode: Plus code (http://plus.codes) is a
// location reference with two formats: global code defining a 14mx14m
// (1/8000th of a degree) or smaller rectangle, and compound code, replacing
// the prefix with a reference location.
type GoogleMapsAddressvalidationV1PlusCode struct {
	// CompoundCode: Place's compound code, such as "33GV+HQ, Ramberg, Norway",
	// containing the suffix of the global code and replacing the prefix with a
	// formatted name of a reference entity.
	CompoundCode string `json:"compoundCode,omitempty"`
	// GlobalCode: Place's global (full) code, such as "9FWM33GV+HQ", representing
	// an 1/8000 by 1/8000 degree area (~14 by 14 meters).
	GlobalCode string `json:"globalCode,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CompoundCode") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CompoundCode") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1PlusCode) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1PlusCode
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1ProvideValidationFeedbackRequest: The request
// for sending validation feedback.
type GoogleMapsAddressvalidationV1ProvideValidationFeedbackRequest struct {
	// Conclusion: Required. The outcome of the sequence of validation attempts. If
	// this field is set to `VALIDATION_CONCLUSION_UNSPECIFIED`, an
	// `INVALID_ARGUMENT` error will be returned.
	//
	// Possible values:
	//   "VALIDATION_CONCLUSION_UNSPECIFIED" - This value is unused. If the
	// `ProvideValidationFeedbackRequest.conclusion` field is set to
	// `VALIDATION_CONCLUSION_UNSPECIFIED`, an `INVALID_ARGUMENT` error will be
	// returned.
	//   "VALIDATED_VERSION_USED" - The version of the address returned by the
	// Address Validation API was used for the transaction.
	//   "USER_VERSION_USED" - The version of the address provided by the user was
	// used for the transaction
	//   "UNVALIDATED_VERSION_USED" - A version of the address that was entered
	// after the last validation attempt but that was not re-validated was used for
	// the transaction.
	//   "UNUSED" - The transaction was abandoned and the address was not used.
	Conclusion string `json:"conclusion,omitempty"`
	// ResponseId: Required. The ID of the response that this feedback is for. This
	// should be the response_id from the first response in a series of address
	// validation attempts.
	ResponseId string `json:"responseId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Conclusion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Conclusion") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1ProvideValidationFeedbackRequest) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1ProvideValidationFeedbackRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1ProvideValidationFeedbackResponse: The response
// for validation feedback. The response is empty if the feedback is sent
// successfully.
type GoogleMapsAddressvalidationV1ProvideValidationFeedbackResponse struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// GoogleMapsAddressvalidationV1UspsAddress: USPS representation of a US
// address.
type GoogleMapsAddressvalidationV1UspsAddress struct {
	// City: City name.
	City string `json:"city,omitempty"`
	// CityStateZipAddressLine: City + state + postal code.
	CityStateZipAddressLine string `json:"cityStateZipAddressLine,omitempty"`
	// Firm: Firm name.
	Firm string `json:"firm,omitempty"`
	// FirstAddressLine: First address line.
	FirstAddressLine string `json:"firstAddressLine,omitempty"`
	// SecondAddressLine: Second address line.
	SecondAddressLine string `json:"secondAddressLine,omitempty"`
	// State: 2 letter state code.
	State string `json:"state,omitempty"`
	// Urbanization: Puerto Rican urbanization name.
	Urbanization string `json:"urbanization,omitempty"`
	// ZipCode: Postal code e.g. 10009.
	ZipCode string `json:"zipCode,omitempty"`
	// ZipCodeExtension: 4-digit postal code extension e.g. 5023.
	ZipCodeExtension string `json:"zipCodeExtension,omitempty"`
	// ForceSendFields is a list of field names (e.g. "City") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "City") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1UspsAddress) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1UspsAddress
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1UspsData: The USPS data for the address.
// `uspsData` is not guaranteed to be fully populated for every US or PR
// address sent to the Address Validation API. It's recommended to integrate
// the backup address fields in the response if you utilize uspsData as the
// primary part of the response.
type GoogleMapsAddressvalidationV1UspsData struct {
	// AbbreviatedCity: Abbreviated city.
	AbbreviatedCity string `json:"abbreviatedCity,omitempty"`
	// AddressRecordType: Type of the address record that matches the input
	// address. * `F`: FIRM. This is a match to a Firm Record, which is the finest
	// level of match available for an address. * `G`: GENERAL DELIVERY. This is a
	// match to a General Delivery record. * `H`: BUILDING / APARTMENT. This is a
	// match to a Building or Apartment record. * `P`: POST OFFICE BOX. This is a
	// match to a Post Office Box. * `R`: RURAL ROUTE or HIGHWAY CONTRACT: This is
	// a match to either a Rural Route or a Highway Contract record, both of which
	// may have associated Box Number ranges. * `S`: STREET RECORD: This is a match
	// to a Street record containing a valid primary number range.
	AddressRecordType string `json:"addressRecordType,omitempty"`
	// CarrierRoute: The carrier route code. A four character code consisting of a
	// one letter prefix and a three digit route designator. Prefixes: * `C`:
	// Carrier route (or city route) * `R`: Rural route * `H`: Highway Contract
	// Route * `B`: Post Office Box Section * `G`: General delivery unit
	CarrierRoute string `json:"carrierRoute,omitempty"`
	// CarrierRouteIndicator: Carrier route rate sort indicator.
	CarrierRouteIndicator string `json:"carrierRouteIndicator,omitempty"`
	// CassProcessed: Indicator that the request has been CASS processed.
	CassProcessed bool `json:"cassProcessed,omitempty"`
	// County: County name.
	County string `json:"county,omitempty"`
	// DefaultAddress: Indicator that a default address was found, but more
	// specific addresses exists.
	DefaultAddress bool `json:"defaultAddress,omitempty"`
	// DeliveryPointCheckDigit: The delivery point check digit. This number is
	// added to the end of the delivery_point_barcode for mechanically scanned
	// mail. Adding all the digits of the delivery_point_barcode,
	// delivery_point_check_digit, postal code, and ZIP+4 together should yield a
	// number divisible by 10.
	DeliveryPointCheckDigit string `json:"deliveryPointCheckDigit,omitempty"`
	// DeliveryPointCode: 2 digit delivery point code
	DeliveryPointCode string `json:"deliveryPointCode,omitempty"`
	// DpvCmra: Indicates if the address is a CMRA (Commercial Mail Receiving
	// Agency)--a private business receiving mail for clients. Returns a single
	// character. * `Y`: The address is a CMRA * `N`: The address is not a CMRA
	DpvCmra string `json:"dpvCmra,omitempty"`
	// DpvConfirmation: The possible values for DPV confirmation. Returns a single
	// character or returns no value. * `N`: Primary and any secondary number
	// information failed to DPV confirm. * `D`: Address was DPV confirmed for the
	// primary number only, and the secondary number information was missing. *
	// `S`: Address was DPV confirmed for the primary number only, and the
	// secondary number information was present but not confirmed. * `Y`: Address
	// was DPV confirmed for primary and any secondary numbers. * Empty: If the
	// response does not contain a `dpv_confirmation` value, the address was not
	// submitted for DPV confirmation.
	DpvConfirmation string `json:"dpvConfirmation,omitempty"`
	// DpvDoorNotAccessible: Flag indicates addresses where USPS cannot knock on a
	// door to deliver mail. Returns a single character. * `Y`: The door is not
	// accessible. * `N`: No indication the door is not accessible.
	DpvDoorNotAccessible string `json:"dpvDoorNotAccessible,omitempty"`
	// DpvDrop: Flag indicates mail is delivered to a single receptable at a site.
	// Returns a single character. * `Y`: The mail is delivered to a single
	// receptable at a site. * `N`: The mail is not delivered to a single
	// receptable at a site.
	DpvDrop string `json:"dpvDrop,omitempty"`
	// DpvEnhancedDeliveryCode: Indicates that more than one DPV return code is
	// valid for the address. Returns a single character. * `Y`: Address was DPV
	// confirmed for primary and any secondary numbers. * `N`: Primary and any
	// secondary number information failed to DPV confirm. * `S`: Address was DPV
	// confirmed for the primary number only, and the secondary number information
	// was present but not confirmed, or a single trailing alpha on a primary
	// number was dropped to make a DPV match and secondary information required. *
	// `D`: Address was DPV confirmed for the primary number only, and the
	// secondary number information was missing. * `R`: Address confirmed but
	// assigned to phantom route R777 and R779 and USPS delivery is not provided.
	DpvEnhancedDeliveryCode string `json:"dpvEnhancedDeliveryCode,omitempty"`
	// DpvFootnote: The footnotes from delivery point validation. Multiple
	// footnotes may be strung together in the same string. * `AA`: Input address
	// matched to the ZIP+4 file * `A1`: Input address was not matched to the ZIP+4
	// file * `BB`: Matched to DPV (all components) * `CC`: Secondary number not
	// matched and not required * `C1`: Secondary number not matched but required *
	// `N1`: High-rise address missing secondary number * `M1`: Primary number
	// missing * `M3`: Primary number invalid * `P1`: Input address PO, RR or HC
	// box number missing * `P3`: Input address PO, RR, or HC Box number invalid *
	// `F1`: Input address matched to a military address * `G1`: Input address
	// matched to a general delivery address * `U1`: Input address matched to a
	// unique ZIP code * `PB`: Input address matched to PBSA record * `RR`: DPV
	// confirmed address with PMB information * `R1`: DPV confirmed address without
	// PMB information * `R7`: Carrier Route R777 or R779 record * `IA`: Informed
	// Address identified * `TA`: Primary number matched by dropping a trailing
	// alpha
	DpvFootnote string `json:"dpvFootnote,omitempty"`
	// DpvNoSecureLocation: Flag indicates door is accessible, but package will not
	// be left due to security concerns. Returns a single character. * `Y`: The
	// package will not be left due to security concerns. * `N`: No indication the
	// package will not be left due to security concerns.
	DpvNoSecureLocation string `json:"dpvNoSecureLocation,omitempty"`
	// DpvNoStat: Is this a no stat address or an active address? No stat addresses
	// are ones which are not continuously occupied or addresses that the USPS does
	// not service. Returns a single character. * `Y`: The address is not active *
	// `N`: The address is active
	DpvNoStat string `json:"dpvNoStat,omitempty"`
	// DpvNoStatReasonCode: Indicates the NoStat type. Returns a reason code as
	// int. * `1`: IDA (Internal Drop Address) – Addresses that do not receive
	// mail directly from the USPS but are delivered to a drop address that
	// services them. * `2`: CDS - Addresses that have not yet become deliverable.
	// For example, a new subdivision where lots and primary numbers have been
	// determined, but no structure exists yet for occupancy. * `3`: Collision -
	// Addresses that do not actually DPV confirm. * `4`: CMZ (College, Military
	// and Other Types) - ZIP + 4 records USPS has incorporated into the data. *
	// `5`: Regular - Indicates addresses not receiving delivery and the addresses
	// are not counted as possible deliveries. * `6`: Secondary Required - The
	// address requires secondary information.
	DpvNoStatReasonCode int64 `json:"dpvNoStatReasonCode,omitempty"`
	// DpvNonDeliveryDays: Flag indicates mail delivery is not performed every day
	// of the week. Returns a single character. * `Y`: The mail delivery is not
	// performed every day of the week. * `N`: No indication the mail delivery is
	// not performed every day of the week.
	DpvNonDeliveryDays string `json:"dpvNonDeliveryDays,omitempty"`
	// DpvNonDeliveryDaysValues: Integer identifying non-delivery days. It can be
	// interrogated using bit flags: 0x40 – Sunday is a non-delivery day 0x20 –
	// Monday is a non-delivery day 0x10 – Tuesday is a non-delivery day 0x08 –
	// Wednesday is a non-delivery day 0x04 – Thursday is a non-delivery day 0x02
	// – Friday is a non-delivery day 0x01 – Saturday is a non-delivery day
	DpvNonDeliveryDaysValues int64 `json:"dpvNonDeliveryDaysValues,omitempty"`
	// DpvPbsa: Indicates the address was matched to PBSA record. Returns a single
	// character. * `Y`: The address was matched to PBSA record. * `N`: The address
	// was not matched to PBSA record.
	DpvPbsa string `json:"dpvPbsa,omitempty"`
	// DpvThrowback: Indicates that mail is not delivered to the street address.
	// Returns a single character. * `Y`: The mail is not delivered to the street
	// address. * `N`: The mail is delivered to the street address.
	DpvThrowback string `json:"dpvThrowback,omitempty"`
	// DpvVacant: Is this place vacant? Returns a single character. * `Y`: The
	// address is vacant * `N`: The address is not vacant
	DpvVacant string `json:"dpvVacant,omitempty"`
	// ElotFlag: eLOT Ascending/Descending Flag (A/D).
	ElotFlag string `json:"elotFlag,omitempty"`
	// ElotNumber: Enhanced Line of Travel (eLOT) number.
	ElotNumber string `json:"elotNumber,omitempty"`
	// ErrorMessage: Error message for USPS data retrieval. This is populated when
	// USPS processing is suspended because of the detection of artificially
	// created addresses. The USPS data fields might not be populated when this
	// error is present.
	ErrorMessage string `json:"errorMessage,omitempty"`
	// EwsNoMatch: The delivery address is matchable, but the EWS file indicates
	// that an exact match will be available soon.
	EwsNoMatch bool `json:"ewsNoMatch,omitempty"`
	// FipsCountyCode: FIPS county code.
	FipsCountyCode string `json:"fipsCountyCode,omitempty"`
	// LacsLinkIndicator: LACSLink indicator.
	LacsLinkIndicator string `json:"lacsLinkIndicator,omitempty"`
	// LacsLinkReturnCode: LACSLink return code.
	LacsLinkReturnCode string `json:"lacsLinkReturnCode,omitempty"`
	// PmbDesignator: PMB (Private Mail Box) unit designator.
	PmbDesignator string `json:"pmbDesignator,omitempty"`
	// PmbNumber: PMB (Private Mail Box) number;
	PmbNumber string `json:"pmbNumber,omitempty"`
	// PoBoxOnlyPostalCode: PO Box only postal code.
	PoBoxOnlyPostalCode bool `json:"poBoxOnlyPostalCode,omitempty"`
	// PostOfficeCity: Main post office city.
	PostOfficeCity string `json:"postOfficeCity,omitempty"`
	// PostOfficeState: Main post office state.
	PostOfficeState string `json:"postOfficeState,omitempty"`
	// StandardizedAddress: USPS standardized address.
	StandardizedAddress *GoogleMapsAddressvalidationV1UspsAddress `json:"standardizedAddress,omitempty"`
	// SuitelinkFootnote: Footnotes from matching a street or highrise record to
	// suite information. If business name match is found, the secondary number is
	// returned. * `A`: SuiteLink record match, business address improved. * `00`:
	// No match, business address is not improved.
	SuitelinkFootnote string `json:"suitelinkFootnote,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AbbreviatedCity") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AbbreviatedCity") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1UspsData) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1UspsData
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1ValidateAddressRequest: The request for
// validating an address.
type GoogleMapsAddressvalidationV1ValidateAddressRequest struct {
	// Address: Required. The address being validated. Unformatted addresses should
	// be submitted via `address_lines`. The total length of the fields in this
	// input must not exceed 280 characters. Supported regions can be found here
	// (https://developers.google.com/maps/documentation/address-validation/coverage).
	// The language_code value in the input address is reserved for future uses and
	// is ignored today. The validated address result will be populated based on
	// the preferred language for the given address, as identified by the system.
	// The Address Validation API ignores the values in recipients and
	// organization. Any values in those fields will be discarded and not returned.
	// Please do not set them.
	Address *GoogleTypePostalAddress `json:"address,omitempty"`
	// EnableUspsCass: Enables USPS CASS compatible mode. This affects _only_ the
	// [google.maps.addressvalidation.v1.ValidationResult.usps_data] field of
	// [google.maps.addressvalidation.v1.ValidationResult]. Note: for USPS CASS
	// enabled requests for addresses in Puerto Rico, a
	// [google.type.PostalAddress.region_code] of the `address` must be provided as
	// "PR", or an [google.type.PostalAddress.administrative_area] of the `address`
	// must be provided as "Puerto Rico" (case-insensitive) or "PR". It's
	// recommended to use a componentized `address`, or alternatively specify at
	// least two [google.type.PostalAddress.address_lines] where the first line
	// contains the street number and name and the second line contains the city,
	// state, and zip code.
	EnableUspsCass bool `json:"enableUspsCass,omitempty"`
	// LanguageOptions: Optional. Preview: This feature is in Preview (pre-GA).
	// Pre-GA products and features might have limited support, and changes to
	// pre-GA products and features might not be compatible with other pre-GA
	// versions. Pre-GA Offerings are covered by the Google Maps Platform Service
	// Specific Terms
	// (https://cloud.google.com/maps-platform/terms/maps-service-terms). For more
	// information, see the launch stage descriptions
	// (https://developers.google.com/maps/launch-stages). Enables the Address
	// Validation API to include additional information in the response.
	LanguageOptions *GoogleMapsAddressvalidationV1LanguageOptions `json:"languageOptions,omitempty"`
	// PreviousResponseId: This field must be empty for the first address
	// validation request. If more requests are necessary to fully validate a
	// single address (for example if the changes the user makes after the initial
	// validation need to be re-validated), then each followup request must
	// populate this field with the response_id from the very first response in the
	// validation sequence.
	PreviousResponseId string `json:"previousResponseId,omitempty"`
	// SessionToken: Optional. A string which identifies an Autocomplete session
	// for billing purposes. Must be a URL and filename safe base64 string with at
	// most 36 ASCII characters in length. Otherwise an INVALID_ARGUMENT error is
	// returned. The session begins when the user makes an Autocomplete query, and
	// concludes when they select a place and a call to Place Details or Address
	// Validation is made. Each session can have multiple Autocomplete queries,
	// followed by one Place Details or Address Validation request. The credentials
	// used for each request within a session must belong to the same Google Cloud
	// Console project. Once a session has concluded, the token is no longer valid;
	// your app must generate a fresh token for each session. If the `sessionToken`
	// parameter is omitted, or if you reuse a session token, the session is
	// charged as if no session token was provided (each request is billed
	// separately). Note: Address Validation can only be used in sessions with the
	// Autocomplete (New) API, not the Autocomplete API. See
	// https://developers.google.com/maps/documentation/places/web-service/session-pricing
	// for more details.
	SessionToken string `json:"sessionToken,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Address") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Address") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1ValidateAddressRequest) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1ValidateAddressRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1ValidateAddressResponse: The response to an
// address validation request.
type GoogleMapsAddressvalidationV1ValidateAddressResponse struct {
	// ResponseId: The UUID that identifies this response. If the address needs to
	// be re-validated, this UUID *must* accompany the new request.
	ResponseId string `json:"responseId,omitempty"`
	// Result: The result of the address validation.
	Result *GoogleMapsAddressvalidationV1ValidationResult `json:"result,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ResponseId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ResponseId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1ValidateAddressResponse) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1ValidateAddressResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1ValidationResult: The result of validating an
// address.
type GoogleMapsAddressvalidationV1ValidationResult struct {
	// Address: Information about the address itself as opposed to the geocode.
	Address *GoogleMapsAddressvalidationV1Address `json:"address,omitempty"`
	// EnglishLatinAddress: Preview: This feature is in Preview (pre-GA). Pre-GA
	// products and features might have limited support, and changes to pre-GA
	// products and features might not be compatible with other pre-GA versions.
	// Pre-GA Offerings are covered by the Google Maps Platform Service Specific
	// Terms (https://cloud.google.com/maps-platform/terms/maps-service-terms). For
	// more information, see the launch stage descriptions
	// (https://developers.google.com/maps/launch-stages). The address translated
	// to English. Translated addresses are not reusable as API input. The service
	// provides them so that the user can use their native language to confirm or
	// deny the validation of the originally-provided address. If part of the
	// address doesn't have an English translation, the service returns that part
	// in an alternate language that uses a Latin script. See here
	// (https://developers.google.com/maps/documentation/address-validation/convert-addresses-english)
	// for an explanation of how the alternate language is selected. If part of the
	// address doesn't have any translations or transliterations in a language that
	// uses a Latin script, the service returns that part in the local language
	// associated with the address. Enable this output by using the
	// [google.maps.addressvalidation.v1.LanguageOptions.return_english_latin_addres
	// s] flag. Note: the
	// [google.maps.addressvalidation.v1.Address.unconfirmed_component_types] field
	// in the `english_latin_address` and the
	// [google.maps.addressvalidation.v1.AddressComponent.confirmation_level]
	// fields in `english_latin_address.address_components` are not populated.
	EnglishLatinAddress *GoogleMapsAddressvalidationV1Address `json:"englishLatinAddress,omitempty"`
	// Geocode: Information about the location and place that the address geocoded
	// to.
	Geocode *GoogleMapsAddressvalidationV1Geocode `json:"geocode,omitempty"`
	// Metadata: Other information relevant to deliverability. `metadata` is not
	// guaranteed to be fully populated for every address sent to the Address
	// Validation API.
	Metadata *GoogleMapsAddressvalidationV1AddressMetadata `json:"metadata,omitempty"`
	// UspsData: Extra deliverability flags provided by USPS. Only provided in
	// region `US` and `PR`.
	UspsData *GoogleMapsAddressvalidationV1UspsData `json:"uspsData,omitempty"`
	// Verdict: Overall verdict flags
	Verdict *GoogleMapsAddressvalidationV1Verdict `json:"verdict,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Address") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Address") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1ValidationResult) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1ValidationResult
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleMapsAddressvalidationV1Verdict: High level overview of the address
// validation result and geocode.
type GoogleMapsAddressvalidationV1Verdict struct {
	// AddressComplete: The address is considered complete if there are no
	// unresolved tokens, no unexpected or missing address components. If unset,
	// indicates that the value is `false`. See `missing_component_types`,
	// `unresolved_tokens` or `unexpected` fields for more details.
	AddressComplete bool `json:"addressComplete,omitempty"`
	// GeocodeGranularity: Information about the granularity of the `geocode`. This
	// can be understood as the semantic meaning of how coarse or fine the geocoded
	// location is. This can differ from the `validation_granularity` above
	// occasionally. For example, our database might record the existence of an
	// apartment number but do not have a precise location for the apartment within
	// a big apartment complex. In that case, the `validation_granularity` will be
	// `SUB_PREMISE` but the `geocode_granularity` will be `PREMISE`.
	//
	// Possible values:
	//   "GRANULARITY_UNSPECIFIED" - Default value. This value is unused.
	//   "SUB_PREMISE" - Below-building level result, such as an apartment.
	//   "PREMISE" - Building-level result.
	//   "PREMISE_PROXIMITY" - A geocode that approximates the building-level
	// location of the address.
	//   "BLOCK" - The address or geocode indicates a block. Only used in regions
	// which have block-level addressing, such as Japan.
	//   "ROUTE" - The geocode or address is granular to route, such as a street,
	// road, or highway.
	//   "OTHER" - All other granularities, which are bucketed together since they
	// are not deliverable.
	GeocodeGranularity string `json:"geocodeGranularity,omitempty"`
	// HasInferredComponents: At least one address component was inferred (added)
	// that wasn't in the input, see
	// [google.maps.addressvalidation.v1.Address.address_components] for details.
	HasInferredComponents bool `json:"hasInferredComponents,omitempty"`
	// HasReplacedComponents: At least one address component was replaced, see
	// [google.maps.addressvalidation.v1.Address.address_components] for details.
	HasReplacedComponents bool `json:"hasReplacedComponents,omitempty"`
	// HasUnconfirmedComponents: At least one address component cannot be
	// categorized or validated, see
	// [google.maps.addressvalidation.v1.Address.address_components] for details.
	HasUnconfirmedComponents bool `json:"hasUnconfirmedComponents,omitempty"`
	// InputGranularity: The granularity of the **input** address. This is the
	// result of parsing the input address and does not give any validation
	// signals. For validation signals, refer to `validation_granularity` below.
	// For example, if the input address includes a specific apartment number, then
	// the `input_granularity` here will be `SUB_PREMISE`. If we cannot match the
	// apartment number in the databases or the apartment number is invalid, the
	// `validation_granularity` will likely be `PREMISE` or below.
	//
	// Possible values:
	//   "GRANULARITY_UNSPECIFIED" - Default value. This value is unused.
	//   "SUB_PREMISE" - Below-building level result, such as an apartment.
	//   "PREMISE" - Building-level result.
	//   "PREMISE_PROXIMITY" - A geocode that approximates the building-level
	// location of the address.
	//   "BLOCK" - The address or geocode indicates a block. Only used in regions
	// which have block-level addressing, such as Japan.
	//   "ROUTE" - The geocode or address is granular to route, such as a street,
	// road, or highway.
	//   "OTHER" - All other granularities, which are bucketed together since they
	// are not deliverable.
	InputGranularity string `json:"inputGranularity,omitempty"`
	// ValidationGranularity: The granularity level that the API can fully
	// **validate** the address to. For example, an `validation_granularity` of
	// `PREMISE` indicates all address components at the level of `PREMISE` or more
	// coarse can be validated. Per address component validation result can be
	// found in [google.maps.addressvalidation.v1.Address.address_components].
	//
	// Possible values:
	//   "GRANULARITY_UNSPECIFIED" - Default value. This value is unused.
	//   "SUB_PREMISE" - Below-building level result, such as an apartment.
	//   "PREMISE" - Building-level result.
	//   "PREMISE_PROXIMITY" - A geocode that approximates the building-level
	// location of the address.
	//   "BLOCK" - The address or geocode indicates a block. Only used in regions
	// which have block-level addressing, such as Japan.
	//   "ROUTE" - The geocode or address is granular to route, such as a street,
	// road, or highway.
	//   "OTHER" - All other granularities, which are bucketed together since they
	// are not deliverable.
	ValidationGranularity string `json:"validationGranularity,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AddressComplete") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AddressComplete") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleMapsAddressvalidationV1Verdict) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleMapsAddressvalidationV1Verdict
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GoogleTypeLatLng: An object that represents a latitude/longitude pair. This
// is expressed as a pair of doubles to represent degrees latitude and degrees
// longitude. Unless specified otherwise, this object must conform to the WGS84
// standard. Values must be within normalized ranges.
type GoogleTypeLatLng struct {
	// Latitude: The latitude in degrees. It must be in the range [-90.0, +90.0].
	Latitude float64 `json:"latitude,omitempty"`
	// Longitude: The longitude in degrees. It must be in the range [-180.0,
	// +180.0].
	Longitude float64 `json:"longitude,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Latitude") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Latitude") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleTypeLatLng) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleTypeLatLng
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *GoogleTypeLatLng) UnmarshalJSON(data []byte) error {
	type NoMethod GoogleTypeLatLng
	var s1 struct {
		Latitude  gensupport.JSONFloat64 `json:"latitude"`
		Longitude gensupport.JSONFloat64 `json:"longitude"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Latitude = float64(s1.Latitude)
	s.Longitude = float64(s1.Longitude)
	return nil
}

// GoogleTypePostalAddress: Represents a postal address. For example for postal
// delivery or payments addresses. Given a postal address, a postal service can
// deliver items to a premise, P.O. Box or similar. It is not intended to model
// geographical locations (roads, towns, mountains). In typical usage an
// address would be created by user input or from importing existing data,
// depending on the type of process. Advice on address input / editing: - Use
// an internationalization-ready address widget such as
// https://github.com/google/libaddressinput) - Users should not be presented
// with UI elements for input or editing of fields outside countries where that
// field is used. For more guidance on how to use this schema, see:
// https://support.google.com/business/answer/6397478
type GoogleTypePostalAddress struct {
	// AddressLines: Unstructured address lines describing the lower levels of an
	// address. Because values in address_lines do not have type information and
	// may sometimes contain multiple values in a single field (For example
	// "Austin, TX"), it is important that the line order is clear. The order of
	// address lines should be "envelope order" for the country/region of the
	// address. In places where this can vary (For example Japan), address_language
	// is used to make it explicit (For example "ja" for large-to-small ordering
	// and "ja-Latn" or "en" for small-to-large). This way, the most specific line
	// of an address can be selected based on the language. The minimum permitted
	// structural representation of an address consists of a region_code with all
	// remaining information placed in the address_lines. It would be possible to
	// format such an address very approximately without geocoding, but no semantic
	// reasoning could be made about any of the address components until it was at
	// least partially resolved. Creating an address only containing a region_code
	// and address_lines, and then geocoding is the recommended way to handle
	// completely unstructured addresses (as opposed to guessing which parts of the
	// address should be localities or administrative areas).
	AddressLines []string `json:"addressLines,omitempty"`
	// AdministrativeArea: Optional. Highest administrative subdivision which is
	// used for postal addresses of a country or region. For example, this can be a
	// state, a province, an oblast, or a prefecture. Specifically, for Spain this
	// is the province and not the autonomous community (For example "Barcelona"
	// and not "Catalonia"). Many countries don't use an administrative area in
	// postal addresses. For example in Switzerland this should be left
	// unpopulated.
	AdministrativeArea string `json:"administrativeArea,omitempty"`
	// LanguageCode: Optional. BCP-47 language code of the contents of this address
	// (if known). This is often the UI language of the input form or is expected
	// to match one of the languages used in the address' country/region, or their
	// transliterated equivalents. This can affect formatting in certain countries,
	// but is not critical to the correctness of the data and will never affect any
	// validation or other non-formatting related operations. If this value is not
	// known, it should be omitted (rather than specifying a possibly incorrect
	// default). Examples: "zh-Hant", "ja", "ja-Latn", "en".
	LanguageCode string `json:"languageCode,omitempty"`
	// Locality: Optional. Generally refers to the city/town portion of the
	// address. Examples: US city, IT comune, UK post town. In regions of the world
	// where localities are not well defined or do not fit into this structure
	// well, leave locality empty and use address_lines.
	Locality string `json:"locality,omitempty"`
	// Organization: Optional. The name of the organization at the address.
	Organization string `json:"organization,omitempty"`
	// PostalCode: Optional. Postal code of the address. Not all countries use or
	// require postal codes to be present, but where they are used, they may
	// trigger additional validation with other parts of the address (For example
	// state/zip validation in the U.S.A.).
	PostalCode string `json:"postalCode,omitempty"`
	// Recipients: Optional. The recipient at the address. This field may, under
	// certain circumstances, contain multiline information. For example, it might
	// contain "care of" information.
	Recipients []string `json:"recipients,omitempty"`
	// RegionCode: Required. CLDR region code of the country/region of the address.
	// This is never inferred and it is up to the user to ensure the value is
	// correct. See https://cldr.unicode.org/ and
	// https://www.unicode.org/cldr/charts/30/supplemental/territory_information.html
	// for details. Example: "CH" for Switzerland.
	RegionCode string `json:"regionCode,omitempty"`
	// Revision: The schema revision of the `PostalAddress`. This must be set to 0,
	// which is the latest revision. All new revisions **must** be backward
	// compatible with old revisions.
	Revision int64 `json:"revision,omitempty"`
	// SortingCode: Optional. Additional, country-specific, sorting code. This is
	// not used in most regions. Where it is used, the value is either a string
	// like "CEDEX", optionally followed by a number (For example "CEDEX 7"), or
	// just a number alone, representing the "sector code" (Jamaica), "delivery
	// area indicator" (Malawi) or "post office indicator" (For example Côte
	// d'Ivoire).
	SortingCode string `json:"sortingCode,omitempty"`
	// Sublocality: Optional. Sublocality of the address. For example, this can be
	// neighborhoods, boroughs, districts.
	Sublocality string `json:"sublocality,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AddressLines") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AddressLines") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GoogleTypePostalAddress) MarshalJSON() ([]byte, error) {
	type NoMethod GoogleTypePostalAddress
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type V1ProvideValidationFeedbackCall struct {
	s                                                             *Service
	googlemapsaddressvalidationv1providevalidationfeedbackrequest *GoogleMapsAddressvalidationV1ProvideValidationFeedbackRequest
	urlParams_                                                    gensupport.URLParams
	ctx_                                                          context.Context
	header_                                                       http.Header
}

// ProvideValidationFeedback: Feedback about the outcome of the sequence of
// validation attempts. This should be the last call made after a sequence of
// validation calls for the same address, and should be called once the
// transaction is concluded. This should only be sent once for the sequence of
// `ValidateAddress` requests needed to validate an address fully.
func (r *V1Service) ProvideValidationFeedback(googlemapsaddressvalidationv1providevalidationfeedbackrequest *GoogleMapsAddressvalidationV1ProvideValidationFeedbackRequest) *V1ProvideValidationFeedbackCall {
	c := &V1ProvideValidationFeedbackCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.googlemapsaddressvalidationv1providevalidationfeedbackrequest = googlemapsaddressvalidationv1providevalidationfeedbackrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *V1ProvideValidationFeedbackCall) Fields(s ...googleapi.Field) *V1ProvideValidationFeedbackCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *V1ProvideValidationFeedbackCall) Context(ctx context.Context) *V1ProvideValidationFeedbackCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *V1ProvideValidationFeedbackCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *V1ProvideValidationFeedbackCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.googlemapsaddressvalidationv1providevalidationfeedbackrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1:provideValidationFeedback")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "addressvalidation.provideValidationFeedback", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "addressvalidation.provideValidationFeedback" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleMapsAddressvalidationV1ProvideValidationFeedbackResponse.ServerRespons
// e.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *V1ProvideValidationFeedbackCall) Do(opts ...googleapi.CallOption) (*GoogleMapsAddressvalidationV1ProvideValidationFeedbackResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleMapsAddressvalidationV1ProvideValidationFeedbackResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "addressvalidation.provideValidationFeedback", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type V1ValidateAddressCall struct {
	s                                                   *Service
	googlemapsaddressvalidationv1validateaddressrequest *GoogleMapsAddressvalidationV1ValidateAddressRequest
	urlParams_                                          gensupport.URLParams
	ctx_                                                context.Context
	header_                                             http.Header
}

// ValidateAddress: Validates an address.
func (r *V1Service) ValidateAddress(googlemapsaddressvalidationv1validateaddressrequest *GoogleMapsAddressvalidationV1ValidateAddressRequest) *V1ValidateAddressCall {
	c := &V1ValidateAddressCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.googlemapsaddressvalidationv1validateaddressrequest = googlemapsaddressvalidationv1validateaddressrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *V1ValidateAddressCall) Fields(s ...googleapi.Field) *V1ValidateAddressCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *V1ValidateAddressCall) Context(ctx context.Context) *V1ValidateAddressCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *V1ValidateAddressCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *V1ValidateAddressCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.googlemapsaddressvalidationv1validateaddressrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1:validateAddress")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "addressvalidation.validateAddress", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "addressvalidation.validateAddress" call.
// Any non-2xx status code is an error. Response headers are in either
// *GoogleMapsAddressvalidationV1ValidateAddressResponse.ServerResponse.Header
// or (if a response was returned at all) in error.(*googleapi.Error).Header.
// Use googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *V1ValidateAddressCall) Do(opts ...googleapi.CallOption) (*GoogleMapsAddressvalidationV1ValidateAddressResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GoogleMapsAddressvalidationV1ValidateAddressResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "addressvalidation.validateAddress", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
