//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
// Read the configuration list from the Weather Station software and 
// create a TAG based file.
//
// Version 2.00
//

#include "msginc.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


#define MAX_DATABUF            1024
#define MAX_FILEBUF            5000
#define MAX_FILEBUF            5000
#define COUNT_ROM              64
#define SWITCH_ROM             72
#define TEMP_ROM               80

extern int AddTAGGroup(ushort, TAGType *);
extern int EndTAGGroup(TAGType *);
extern int AddTAGData(ushort, void *, int, TAGType *);
extern int TAGToString(TAGType *, char *, int);
extern int ParseTAG(TAGType *,int,int);
extern int ToHex(char);

int iButtonFound(void);
int ParseData(char *, int, uchar *, int);


//----------------------------------------------------------------------
//  This is the Main routine for list2tag.
//
int main(short argc, char **argv)
{
   int fllen,buflen,i,j,mcnt,ln;
   FILE *infp,*outfp;
   TAGType OutTAG;
   char filebuf[MAX_FILEBUF];
   uchar databuf[MAX_DATABUF];
   uchar rombuf[88];
   char msg[80],buf[2048];
   ulong ul;

   // check arguments to see if request instruction with '?' or too many
   if (argc < 2)
   {
       printf("\nusage: list2obm input_filename <output_filename>\n"
              "  - Create an TAG Tag based on the ROM numbers\n"
              "    in the Weather Station config file (usually ini.txt)\n"
              "  - argument 1, input_filename: input filename\n"
              "  - optional argument 2, output_filename: output filename\n"
              "  - version 1.00\n");
       exit(1);
   }

   // open the input file
   infp = fopen(argv[1],"r+");
   if(infp == NULL)
   {
      printf("ERROR, Could not open input file!\n");
      exit(1);
   }
   else
      printf("File '%s' opened to read.\n",
              argv[1]);

   // open the output file
   if (argc >= 3)
   {
      outfp = fopen(argv[2],"wb");
      if(outfp == NULL)
      {
         printf("ERROR, Could not open output file!\n");
         exit(1);
      }
      else
         printf("File '%s' opened to write ouput TAG Tag.\n",
                 argv[2]);
   }
   else
   {
      // use standard out
      printf("Output will only be sent to standard out\n");      
      outfp = NULL;
   }

   
   // read the input file 
   fllen = fread(filebuf,1,MAX_FILEBUF,infp);

   // clean up opened files
   fclose(infp);
   printf("File '%s' closed.\n",
           argv[1]);

   // attempt to convert the text into an array of uchars 
   buflen = ParseData(filebuf, fllen, databuf, MAX_DATABUF);

   // verify correct length
   if (buflen != 88)
   {
      printf("ERROR, input file is the incorrect length!\n");
      exit(1);
   }

   // reverse order of ROM's for ease of use
   for (i = 0; i < 11; i++)
   {
      for (j = 0; j < 8; j++)
         rombuf[7 - j + 8 * i] = databuf[j + 8 * i];
   }

   // create a TAG using the ROM info

   // {G: ParseData} [
   AddTAGGroup(GD_ParseData,&OutTAG);
      // {D: Description ()}
      mcnt = sprintf(msg,"1-Wire Weather Station");
      AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
      // {D: Cluster ID ()}
      for (i = 6; i >= 0; i--)
         if (rombuf[SWITCH_ROM + i] != 0)
            break;
      ln = i + 1; j = 0;
      for (; i >= 0; i--)
         databuf[j++] = rombuf[SWITCH_ROM + i];
      AddTAGData(DD_ClusterID,databuf,ln,&OutTAG);
      // {D: Cluster Revision ()}
      ul = 0x00;
      AddTAGData(DD_ClusterRev,&ul,1,&OutTAG);
      // {G: OWCluster} [
      AddTAGGroup(GD_OWCluster,&OutTAG);
         // {D: Description ()}
         sprintf(msg,"Isolation Branch");  
         AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
         // {D: ROM Array (8)}
         AddTAGData(DD_RomData,&rombuf[SWITCH_ROM],8,&OutTAG);
         // {D: ChannelMask (1) }
         ul = 0x02;
         AddTAGData(DD_ChannelMask,&ul,1,&OutTAG);
         // {D: ChannelState (1) }
         ul = 0x02;
         AddTAGData(DD_ChannelState,&ul,1,&OutTAG);
         // {D: AccessMethod () }
         ul = AM_SWITCH_2406;
         AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
         // {G: OWBranch} [
         AddTAGGroup(GD_OWBranch,&OutTAG);
            // {G: OWSensor} [
            AddTAGGroup(GD_OWSensor,&OutTAG);
               // {D: Description ()}
               sprintf(msg,"Wind Direction");  
               AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
               // {D: AccessMethod () }
               ul = AM_ORDER_ROM_SWITCH_LIST;
               AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
               // {D: Init State ()}
               AddTAGData(DD_InitState,&rombuf[0],64,&OutTAG);
            // ]
            EndTAGGroup(&OutTAG);
         // ]
         EndTAGGroup(&OutTAG);

         // {G: OWSensor} [
         AddTAGGroup(GD_OWSensor,&OutTAG);
            // {D: Description ()}
            sprintf(msg,"Enclosure");  
            AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
            // {D: ROM Array (8)}
            AddTAGData(DD_RomData,&rombuf[TEMP_ROM],8,&OutTAG);
            // {D: AccessMethod () }
            ul = AM_TEMPERATURE_1820;
            AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
         // ]
         EndTAGGroup(&OutTAG);

         // {G: OWSensor} [
         AddTAGGroup(GD_OWSensor,&OutTAG);
            // {D: Description ()}
            sprintf(msg,"Wind Speed");  
            AddTAGData(DD_Description,&msg[0],strlen(msg),&OutTAG);
            // {D: ROM Array (8)}
            AddTAGData(DD_RomData,&rombuf[COUNT_ROM],8,&OutTAG);
            // {D: Unit Name ()}
            sprintf(msg,"MPH");  
            AddTAGData(DD_UnitName,&msg[0],strlen(msg),&OutTAG);
            // {D: Scale FactorM (1) }
            ul = 0x2FE9;
            AddTAGData(DD_ScaleFactorM,&ul,2,&OutTAG);
            // {D: Scale FactorD (1) }
            ul = 0x2710;
            AddTAGData(DD_ScaleFactorD,&ul,2,&OutTAG);
            // {D: Init State ()}
            msg[0] = 15;
            AddTAGData(DD_InitState,&msg[0],1,&OutTAG);
            // {D: AccessMethod () }
            ul = AM_COUNT_VELOCITY_2423;
            AddTAGData(DD_AccessMethod,&ul,1,&OutTAG);
         // ]
         EndTAGGroup(&OutTAG);
      // ]
      EndTAGGroup(&OutTAG);
   // ]
   EndTAGGroup(&OutTAG);

   // print out the TAG Tag in human readable form
   TAGToString(&OutTAG,buf,sizeof(buf));
   printf("\n%s\n",buf);

   // write the output file
   if (outfp != NULL)
   {   
      // write to the output file
      fwrite(&OutTAG.msg[0], 1, OutTAG.msglen, outfp);
      // close output file
      printf("File '%s' closed.\n",
              argv[2]);
      fclose(outfp);
   }
   
   // normal exit
   exit(0);

   return 0;
}


//----------------------------------------------------------------------
// Parse the raw file data in to an array of uchar's.  The data must
// be hex characters possible seperated by white space
//     12 34 456789ABCD EF
//     FE DC BA 98 7654 32 10
// would be converted to an array of 16 uchars.
// return the array length.  If an invalid hex pair is found then the
// return will be 0.
//
int ParseData(char *inbuf, int insize, uchar *outbuf, int maxsize)
{
   int ps, outlen=0, gotmnib=0;
   uchar mnib;

   // loop until end of data
   for (ps = 0; ps < insize; ps++)
   {
      // check for white space 
      if (isspace(inbuf[ps]))
         continue;
      // not white space, make sure hex
      else if (isxdigit(inbuf[ps]))
      {
         // check if have first nibble yet
         if (gotmnib)
         {  
            // this is the second nibble
            outbuf[outlen++] = (mnib << 4) | ToHex(inbuf[ps]);
            gotmnib = 0;
         }
         else
         {
            // this is the first nibble
            mnib = ToHex(inbuf[ps]);
            gotmnib = 1;
         }
      }
      else
         return 0;

      // if getting to the max return what we have
      if ((outlen + 1) >= maxsize)
         return outlen;
   }

   return outlen;
}


