//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  mibvect.c   - Vector functions for MIB groups.  These functions are 
//                called during TAG parsing.  These functions currently call
//                the 1-Wire script functions to actually perform the 
//                sensor operations.  The could instead just log the 
//                sensor information to a database.
//
//  Version: 2.00
//  Revisions:
//

// Includes
#include "1wsensor.h"
#include "msginc.h"
#include "mib.h"

// external prototypes
extern int RunScript(int, Script script[], void *, int);
extern void output_status(int, char *);
extern int ReadFromMIB(ushort, uchar *, int);
extern int WriteToMIB(ushort, void *, int);
extern void *MIBPayload(ushort);
extern int ROMToStr(char *, uchar *);
extern int TAGDataToString(ushort, uchar *, int, int, char *, int);
extern int TAGGroupToString(ushort, int, int, char *, int);
extern int ZeroMIBVariable(ushort);
extern int MIBZero(ushort);

// local functions
int indent(char *, int);
int rom(char *);
void DisplayBranchResults(int, int, int);
void DisplayVelocityResults(ulong *, int);
int DisplayOrderedSwitchResults(int, int);
int OutputDescriptorAndData(ushort, int);
void DisplayLevelResults(uchar *, int);
void DisplayActivityResults(uchar *, int);
int BranchStack(int);
void DisplayCounterResults(ulong *, int);

//--------------------------------------------------------------------------
// MIB Vector functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Vector function called when a Branch group is read.
// It will be called at the begining (PRE_EX) and at the end
// (POST_EX) of the group parsing.
//
// Required objects:
//   DD_RomData,DD_AccessMethod,DD_ChannelMask,DD_ChannelState
//
// Optional objects:  
//   DD_Description
//
//  Returns:  TRUE - required MIB objects present and 1-Wire script run 
//                   without error
//            FALSE - required MIB objects NOT present or error occured 
//                    while running 1-Wire script
//
int VectOWBranch(int portnum, uchar attrib, int depth)
{
   int rslt,rt;
   uchar buf[10];
   char msg[256]; 
   uchar am;
   int msgcnt = 0;

   // check if pre or post call
   if (attrib  == PRE_EX)
   {
      // pre call so push this ROM on the branch stack
      if (!BranchStack(attrib))
         output_status(LV_ALWAYS,"ERROR, exceeded 10 deep on branch stack\n");

      // print group
      OutputDescriptorAndData(GD_OWBranch,depth-1);
      // print description
      OutputDescriptorAndData(DD_Description,depth);
      // print rom
      OutputDescriptorAndData(DD_RomData,depth);
   }
   // post call 
   else
   {
      // pop off the branch values
      if (!BranchStack(attrib))
         output_status(LV_ALWAYS,"ERROR, branch post call without branch on stack\n");
      // Set state to zero, to turn off branches
      buf[0] = 0x00;
      WriteToMIB(DD_ChannelState, &buf[0], 1);
   }

   // verify required mib variables are present
   if (MIBZero(DD_RomData) || MIBZero(DD_ChannelState) ||
       MIBZero(DD_ChannelMask) || MIBZero(DD_AccessMethod) )
   {
      msgcnt = indent(msg + msgcnt,depth);
      sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
      // output the status
      output_status(LV_ALWAYS,msg);
      return FALSE;
   }

   // read the access method
   if (ReadFromMIB(DD_AccessMethod,&am,1))
   {
      // switch to decide how to access this sensor
      switch (am)
      {
         case AM_SWITCH_2406:
            // run the script to set the Branch state
            rt = RunScript(portnum, SetBranch2406,&rslt,sizeof(rslt));
            break;
         case AM_SWITCH_2409:
            // run the script to set the Branch state
            rt = RunScript(portnum, SetBranch2409,&rslt,sizeof(rslt));
            break;
         default:
            msgcnt = indent(msg + msgcnt,depth);
            sprintf(msg + msgcnt,"RESULT: (Access method not known)\n");
            // output the status
            output_status(LV_ALWAYS,msg);
            rt = FALSE;
            break;
      }
   }
   else
      rt = FALSE;

   // print result if script ended corretly
   if (rt == TRUE)
   {
      // print the results   
      msgcnt += indent(msg + msgcnt,depth);
      sprintf(msg + msgcnt,"RESULT: (set branch) %d\n",rslt);
      // output the status
      output_status(LV_ALWAYS,msg);
   }

   // zero the MIB for all branch variables
   ZeroMIBVariable(DD_RomData);
   ZeroMIBVariable(DD_AccessMethod);
   ZeroMIBVariable(DD_Description);
   ZeroMIBVariable(DD_ChannelMask);
   ZeroMIBVariable(DD_ChannelState);

   return rt;
}

//--------------------------------------------------------------------------
// Vector function called when a 1-Wire Cluster group is read.
// It will be called at the begining (PRE_EX) and at the end 
// (POST_EX) of the group parsing.
//
// Required objects:
//   DD_ClusterID, DD_ClusterRev
//
// Optional objects:  
//   DD_Description
//
//  Returns:  TRUE - required MIB objects present and 1-Wire script run 
//                   without error
//            FALSE - required MIB objects NOT present or error occured
//                    while running 1-Wire script
//
int VectOWCluster(int portnum, uchar attrib, int depth)
{
   char msg[256]; 
   int msgcnt = 0;

   // check if pre or post call
   if (attrib  == PRE_EX)
   {  
      // verify required mib variables are present
      if (MIBZero(DD_ClusterID) || MIBZero(DD_ClusterRev))
      {
         msgcnt = indent(msg + msgcnt,depth);
         sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
         // output the status
         output_status(LV_ALWAYS,msg);
         return FALSE;
      }

      // cluster start marker
      output_status(LV_ALWAYS,"=============================================================================\n");
      // print the description of the cluster
      TAGDataToString(DD_Description, 
                MIBPayload(DD_Description), depth - 1, TRUE,  
                msg, sizeof(msg));
      output_status(LV_ALWAYS,msg);
      // print the cluster ID
      TAGDataToString(DD_ClusterID, 
                MIBPayload(DD_ClusterID), depth - 1, TRUE, 
                msg, sizeof(msg));
      output_status(LV_ALWAYS,msg);
      // print the cluster Rev      
      TAGDataToString(DD_ClusterRev, 
                MIBPayload(DD_ClusterRev), depth - 1, TRUE,  
                msg, sizeof(msg));
      output_status(LV_ALWAYS,msg);
      // print the Manufacturer (string)
      TAGDataToString(DD_Manufacturer, 
                MIBPayload(DD_Manufacturer), depth - 1, TRUE,  
                msg, sizeof(msg));
      output_status(LV_ALWAYS,msg);
      // print the Manufacturer (byte array)
      TAGDataToString(DD_ManufacturerCode, 
                MIBPayload(DD_ManufacturerCode), depth - 1, TRUE,  
                msg, sizeof(msg));
      output_status(LV_ALWAYS,msg);
      // print the Enumeration number 
      TAGDataToString(DD_Enum, 
                MIBPayload(DD_Enum), depth - 1, TRUE,  
                msg, sizeof(msg));
      output_status(LV_ALWAYS,msg);
      // print the time/date 
      TAGDataToString(DD_SecondsSince1970, 
                MIBPayload(DD_SecondsSince1970), depth - 1, TRUE,  
                msg, sizeof(msg));
      output_status(LV_ALWAYS,msg);
   }
   else
      // cluster end marker
      output_status(LV_ALWAYS,"-----------------------------------------------------------------------------\n\n");

   // zero the MIB for all cluster variables
   ZeroMIBVariable(DD_Description);
   ZeroMIBVariable(DD_ClusterID);
   ZeroMIBVariable(DD_ClusterRev);
   ZeroMIBVariable(DD_Manufacturer);
   ZeroMIBVariable(DD_ManufacturerCode);
   ZeroMIBVariable(DD_Enum);
   ZeroMIBVariable(DD_SecondsSince1970);

   return TRUE;
}

//--------------------------------------------------------------------------
// Vector function called when a 1-Wire Sensor group is read.
// It will be called at the (POST_EX) of the group parsing.
//
// Required objects:
//   DD_AccessMethod
//
// Required/Optional(depends on access method) objects:
//   DD_ChannelMask,DD_ChannelState
//   DD_Description,DD_RomData,DD_InitState,DD_ScaleFactorM,
//   DD_ScaleFactorD,DD_UnitName
//
//  Returns:  TRUE - required MIB objects present and 1-Wire script run 
//                   without error
//            FALSE - required MIB objects NOT present or error occured 
//                    while running 1-Wire script
//
int VectOWSensor(int portnum, uchar attrib, int depth)
{
   uchar am;
   float tfl;
   char msg[512]; 
   int msgcnt = 0,rt=TRUE;
   uchar buf[50];
   ulong values[4];

   // print group
   OutputDescriptorAndData(GD_OWSensor,depth-1);

   // read the access method
   if (ReadFromMIB(DD_AccessMethod,&am,1))
   {
      // print description
      OutputDescriptorAndData(DD_Description,depth);

      // switch to decide how to access this sensor
      switch (am)
      {
         // DS1820/DS1920 temperature reading
         case AM_TEMPERATURE_1820:
            // verify required mib variables are present
            if (MIBZero(DD_RomData))
            {
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
               // output the status
               output_status(LV_ALWAYS,msg);
               break;
            }

            // print rom
            OutputDescriptorAndData(DD_RomData,depth);
            // run the script to read this access method device
            if (RunScript(portnum,ReadDS1820,&tfl,sizeof(tfl)))
            {
               // id and result
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: (temperature) %5.1f Celcius, "
                         " %5.1f Fahrenhiet\n",tfl,tfl * 9 / 5 + 32);
               // output the status
               output_status(LV_ALWAYS,msg);
            }
            break;
         // ordered list of rom switches
         case AM_ORDER_ROM_SWITCH_LIST:
            // verify required mib variables are present
            if (MIBZero(DD_InitState))
            {
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
               // output the status
               output_status(LV_ALWAYS,msg);
               break;
            }
            // run the script to check all of the rom switches
            if (RunScript(portnum,ReadOrderedSwitch,&buf[0], sizeof(buf)))
               DisplayOrderedSwitchResults(buf[0], depth);
            break;
         // velocity using a counter over time
         case AM_COUNT_VELOCITY_2423:
            // verify required mib variables are present
            if (MIBZero(DD_RomData) || MIBZero(DD_InitState))
            {
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
               // output the status
               output_status(LV_ALWAYS,msg);
               break;
            }
            // print rom
            OutputDescriptorAndData(DD_RomData,depth);
            if (RunScript(portnum,ReadCounterVelocity2423,&values[0], sizeof(values)))
               DisplayVelocityResults(&values[0],depth);
            break;
         // velocity using a counter over time
         case AM_COUNT_TOTAL_2423:
            // verify required mib variables are present
            if (MIBZero(DD_RomData) || MIBZero(DD_InitState))
            {
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
               // output the status
               output_status(LV_ALWAYS,msg);
               break;
            }
            // print rom
            OutputDescriptorAndData(DD_RomData,depth);
            if (RunScript(portnum,ReadCounter2423,&values[0], sizeof(values)))
               DisplayCounterResults(&values[0],depth);
            break;

         // level (HIGH(1) or LOW(0)) using sensor such as switch
         case AM_LEVEL_2406:
            // verify required mib variables are present
            if (MIBZero(DD_RomData) || MIBZero(DD_ChannelState) || 
                MIBZero(DD_ChannelMask))
            {
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
               // output the status
               output_status(LV_ALWAYS,msg);
               break;
            }
            // print rom
            OutputDescriptorAndData(DD_RomData,depth);
            // run the script to read the level
            if (RunScript(portnum,ReadLevel2406,&buf[0], sizeof(buf)))
               DisplayLevelResults(&buf[0],depth);
            break;

         // level (HIGH(1) or LOW(0)) using sensor such as switch
         case AM_LEVEL_2409:
            // verify required mib variables are present
            if (MIBZero(DD_RomData) || MIBZero(DD_ChannelState) || 
                MIBZero(DD_ChannelMask))
            {
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
               // output the status
               output_status(LV_ALWAYS,msg);
               break;
            }
            // print rom
            OutputDescriptorAndData(DD_RomData,depth);
            // run the script to read the level
            if (RunScript(portnum,ReadLevel2409,&buf[0], sizeof(buf)))
               DisplayLevelResults(&buf[0],depth);
            break;

         // level (HIGH(1) or LOW(0)) using sensor such as switch
         case AM_ACTIVITY_2406:
            // verify required mib variables are present
            if (MIBZero(DD_RomData) || MIBZero(DD_ChannelMask))
            {
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
               // output the status
               output_status(LV_ALWAYS,msg);
               break;
            }
            // print rom
            OutputDescriptorAndData(DD_RomData,depth);
            // run the script to read the level
            if (RunScript(portnum,ReadActivity2406,&buf[0], sizeof(buf)))
               DisplayActivityResults(&buf[0],depth);
            break;

         // level (HIGH(1) or LOW(0)) using sensor such as switch
         case AM_ACTIVITY_2409:
            // verify required mib variables are present
            if (MIBZero(DD_RomData) || MIBZero(DD_ChannelMask))
            {
               msgcnt = indent(msg + msgcnt,depth);
               sprintf(msg + msgcnt,"RESULT: all required MIB variables not present\n");
               // output the status
               output_status(LV_ALWAYS,msg);
               break;
            }
            // print rom
            OutputDescriptorAndData(DD_RomData,depth);
            // run the script to read the level
            if (RunScript(portnum,ReadActivity2409,&buf[0], sizeof(buf)))
               DisplayActivityResults(&buf[0],depth);
            break;

         default:
            msgcnt = indent(msg + msgcnt,depth);
            sprintf(msg + msgcnt,"RESULT: (Access method not known)\n");
            // output the status
            output_status(LV_ALWAYS,msg);
            rt = FALSE;
            break;
      }
   }
   else
      rt = FALSE;

   // zero the MIB for all Sensor variables
   ZeroMIBVariable(DD_RomData);
   ZeroMIBVariable(DD_AccessMethod);
   ZeroMIBVariable(DD_Description);
   ZeroMIBVariable(DD_InitState);
   ZeroMIBVariable(DD_ChannelMask);
   ZeroMIBVariable(DD_ChannelState);
   ZeroMIBVariable(DD_ScaleFactorM);
   ZeroMIBVariable(DD_ScaleFactorD);
   ZeroMIBVariable(DD_UnitName);

   return rt;
}

//--------------------------------------------------------------------------
// Vector function called when a 1-Wire Actuator group is read.
// It will be called at the end (POST_EX) of the group parsing.
//
// Required objects:
//   ??
//
// Optional objects:  
//   ??
//
//  Returns:  TRUE - required MIB objects present and 1-Wire script run 
//                   without error
//            FALSE - required MIB objects NOT present or error occured 
//                    while running 1-Wire script
//
int VectOWActuator(int portnum, uchar attrib, int depth)
{
   char msg[512];
   int msgcnt=0;
   //??? Not supported yet
   msgcnt = indent(msg + msgcnt,depth);
   sprintf(msg + msgcnt,"RESULT: (Actuators not yet supported)\n");
   // output the status
   output_status(LV_ALWAYS,msg);
   return FALSE;
}

//--------------------------------------------------------------------------
// Display functions
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Output the string representation of the specified object descriptor 
// 'ob_desc'. Indent with 'depth'.  Use the 'output_status' function
// to sent output.
//
//  Returns:  number of characters printed not including the zero
//            termination.
//
int OutputDescriptorAndData(ushort ob_desc, int depth)
{
   char msg[512];
   int msgcnt=0;

   // print out as a group or data
   if (ob_desc & GROUP)
      msgcnt = TAGGroupToString(ob_desc, depth, TRUE,
                msg + msgcnt, sizeof(msg) - msgcnt);
   else
      msgcnt = TAGDataToString(ob_desc,
                MIBPayload(ob_desc), depth, TRUE,
                msg + msgcnt, sizeof(msg) - msgcnt);

   // output the status
   output_status(LV_ALWAYS,msg);

   return msgcnt;
}

//--------------------------------------------------------------------------
// Display the results from reading a 'level' 1-Wire sensor with up to 8
// channels.  The data is passed in a uchar array 'buf'.  The data is
// in the following format:
//   Byte number / value
//   0 / bit field of switch level values (0-LOW, 1-HIGH)
//   1 / mask of channels read (0-CHANNEL NOT READ, 1-CHANNEL READ)
//
//  Returns:  number of characters printed not including the zero
//            termination.
//
void DisplayLevelResults(uchar *buf, int depth)
{
   char msg[512];
   int msgcnt,i,mask,value;

   value = buf[0];
   mask = buf[1];

   msgcnt = indent(msg,depth);
   msgcnt += sprintf(msg + msgcnt,"RESULT: (level) ");

   // loop through each channel
   for (i = 0; i < 8; i++)
   {
      // if this level desired
      if (mask & 0x01)
      {
         msgcnt += sprintf(msg + msgcnt,"Channel(%d)-",i);
         if (value & 0x01)
            msgcnt += sprintf(msg + msgcnt,"HIGH ");
         else
            msgcnt += sprintf(msg + msgcnt,"LOW ");
      }
      mask >>= 1;
      value >>= 1;
   }

   sprintf(msg + msgcnt,"\n");
   // output the status
   output_status(LV_ALWAYS,msg);
}

//--------------------------------------------------------------------------
// Display the results from reading an 'Activity' 1-Wire sensor with up to 
// 8 channels.  The data is passed in a uchar array 'buf'.  The data is 
// in the following format:
//   Byte number / value
//   0 / bit field of activity values (0-NO ACTIVITY, 1-ACTIVITY DETECTED)
//   1 / mask of channels read (0-CHANNEL NOT READ, 1-CHANNEL READ)
//
//  Returns:  number of characters printed not including the zero
//            termination.
//
void DisplayActivityResults(uchar *buf, int depth)
{
   char msg[512];
   int msgcnt,i,mask,value;

   value = buf[0];
   mask = buf[1];

   msgcnt = indent(msg,depth);
   msgcnt += sprintf(msg + msgcnt,"RESULT: (activity) ");

   // loop through each channel 
   for (i = 0; i < 8; i++)
   {
      // if this level desired
      if (mask & 0x01)
      {
         msgcnt += sprintf(msg + msgcnt,"Channel(%d)-",i);
         if (value & 0x01)
            msgcnt += sprintf(msg + msgcnt,"YES ");
         else
            msgcnt += sprintf(msg + msgcnt,"NO ");
      }
      mask >>= 1;
      value >>= 1;
   }

   sprintf(msg + msgcnt,"\n");
   // output the status
   output_status(LV_ALWAYS,msg);
}

//--------------------------------------------------------------------------
// Display the results from setting a 1-Wire 'Branch'.  The result of the
// operation is passed in 'rslt' (1 success, 0 failure).  The 'attrib'
// paramter indicates if this was a pre (PRE_EX) or post (POST_EX) call.
//
void DisplayBranchResults(int attrib, int rslt, int depth)
{
   char msg[256]; 
   int msgcnt;

   // check if pre or post call
   if (attrib  == PRE_EX)
   {  
      // print group
      OutputDescriptorAndData(GD_OWBranch,depth-1);
      // print description
      OutputDescriptorAndData(DD_Description,depth);
      // print rom
      OutputDescriptorAndData(DD_RomData,depth);
      // channel mask
      OutputDescriptorAndData(DD_ChannelMask,depth);
      // channel state
      OutputDescriptorAndData(DD_ChannelState,depth);
   }
   else
   {
      // print only result on post call
      msgcnt = indent(msg,depth);
      msgcnt += sprintf(msg + msgcnt,"BRANCH: Close ");
      msgcnt += rom(msg + msgcnt);
      sprintf(msg + msgcnt," result: %d\n",rslt);
   }

   // output the status
   output_status(LV_ALWAYS,msg);
}

//--------------------------------------------------------------------------
// Display the results from reading an ordered list of ROM-switch 1-Wire
// sensors with up to 8 switches.  The data is passed in 'onmask' where
// an ON ROM-switch (1) and OFF ROM-switch (0).  The number of ROM-switches
// is found from the length of the DD_InitState / 8.
//
// Returns:  number of characters printed not including the zero
//           termination.
//
int DisplayOrderedSwitchResults(int onmask, int depth)
{
   uchar tbuf[256];
   char msg[512];
   int msgcnt,ln,i;

   // print title
   msgcnt = indent(msg,depth);
   msgcnt += sprintf(msg + msgcnt,"RESULT: (ordered 'on' list) ");

   // read the MIB for the array or ROM switches
   ln = ReadFromMIB(DD_InitState, &tbuf[0], sizeof(tbuf));

   // loop though each rom and display the ones that
   // have a cooresponding bit set in the on-mask
   for (i = 0; i < ln; i += 8)
   {
      if ((onmask >> (i / 8)) & 0x01)
      {
         msgcnt += sprintf(msg + msgcnt,"(%d) ",i/8);
         msgcnt += ROMToStr(msg + msgcnt, &tbuf[i]);
      }
   }
   msgcnt += sprintf(msg + msgcnt,"\n");

   // output the status
   output_status(LV_ALWAYS,msg);

   return msgcnt;
}

//--------------------------------------------------------------------------
// Display the results from reading a Velocity 1-Wire Sensor.  The data
// is passed in an array of unsigned longs in the following format.
//
//   unsigned long number / value
//   0 / start time in millisecond (t1)  
//   1 / start counter value (c1)
//   2 / end time in millisecond  (t2) 
//   3 / end counter value (c2)
//
// Velocity is calculated as:
//
// ((c2 - c1) * 1000)/(t2 - t1) * DD_ScaleFactorM / DD_ScaleFactorD 
//
void DisplayVelocityResults(ulong *values, int depth)
{
   uchar tbuf[256];
   char msg[256];
   int msgcnt,ln;
   double cps;
   ulong um,ud;

   // print title 
   msgcnt = indent(msg,depth);
   msgcnt += sprintf(msg + msgcnt,"RESULT: (velocity) ");

   // caluculate counts per second
   if (values[2] != values[0])
      cps = (((values[3] - values[1]) * 1000.0) / (values[2] - values[0]));
   else
      cps = 0;

   // read the scale factor
   ln = ReadFromMIB(DD_ScaleFactorM, (uchar *)&um, sizeof(um));
   if ((ln == 0) || (um == 0))
      um = 1;
   ln = ReadFromMIB(DD_ScaleFactorD, (uchar *)&ud, sizeof(ud));
   if ((ln == 0) || (ud == 0))
      ud = 1;

   // scale and print
   msgcnt += sprintf(msg + msgcnt,"%5.1f ", cps * um / ud);

   // add the unit name 
   ln = ReadFromMIB(DD_UnitName, &tbuf[0], sizeof(tbuf));
   tbuf[ln] = 0;
   sprintf(msg + msgcnt,"%s\n",tbuf);

   // output the status
   output_status(LV_ALWAYS,msg);
}

//--------------------------------------------------------------------------
// Display the results from reading a Counter 1-Wire Sensor.  The data
// is passed in an array of unsigned longs in the following format.
//
//   unsigned long number / value
//   0 / start time in millisecond (t)  
//   1 / start counter value (c)
//
// Count is calculated as:
//
// c * DD_ScaleFactorM / DD_ScaleFactorD
//
void DisplayCounterResults(ulong *values, int depth)
{
   uchar tbuf[256];
   char msg[256];
   int msgcnt,ln;
   ulong um,ud;

   // print title
   msgcnt = indent(msg,depth);
   msgcnt += sprintf(msg + msgcnt,"RESULT: (count) ");

   // read the scale factor
   ln = ReadFromMIB(DD_ScaleFactorM, (uchar *)&um, sizeof(um));
   if ((ln == 0) || (um == 0))
      um = 1;
   ln = ReadFromMIB(DD_ScaleFactorD, (uchar *)&ud, sizeof(ud));
   if ((ln == 0) || (ud == 0))
      ud = 1;

   // scale and print
   msgcnt += sprintf(msg + msgcnt,"%5.1f ", (double)values[1] * um / ud);

   // add the unit name
   ln = ReadFromMIB(DD_UnitName, &tbuf[0], sizeof(tbuf));
   tbuf[ln] = 0;
   sprintf(msg + msgcnt,"%s\n",tbuf);

   // output the status
   output_status(LV_ALWAYS,msg);
}

//--------------------------------------------------------------------------
// Function to store (push) and recover (pop) Branch information. This
// will allow a pre and post called the 1-Wire Branch Vector to get
// of the information need to control the branch.  'attrib' PRE_EX
// will push and POST_EX will pop the branch info.  The objects stored
// are DD_RomData, DD_AccessMethod, and DD_ChannelMask.
//
// Returns:  TRUE - branch info successfully pushed or popped
//           FALSE - could not push due to too many branches or
//                   could not pop because no branch info on stack
//
int BranchStack(int attrib)
{
   static uchar branch_stack[10][8];
   static uchar access_stack[10], mask_stack[10];
   static int stack_ptr=0;

   // check if pre (push) or post (pop) call
   if (attrib  == PRE_EX)
   {
      if (stack_ptr < 9) 
      {
         // read rom
         ReadFromMIB(DD_RomData, &branch_stack[stack_ptr][0], 8);
         // read access method
         ReadFromMIB(DD_AccessMethod, &access_stack[stack_ptr], 1);
         // read mask method         
         ReadFromMIB(DD_ChannelMask, &mask_stack[stack_ptr], 1);
         // increment stack pointer
         stack_ptr++;
         return TRUE;
      }
      else
         return FALSE;
   }
   else
   {
      if (stack_ptr >= 1)
      {
         stack_ptr--;
         // post call, so close the branch on the branch stack
         // set the ROM in the MIB from the stack   
         WriteToMIB(DD_RomData, &branch_stack[stack_ptr][0], 8);
         // set the channel mask to both channels
         WriteToMIB(DD_ChannelMask, &mask_stack[stack_ptr], 1);
         // set access method 
         WriteToMIB(DD_AccessMethod, &access_stack[stack_ptr], 1);
         return TRUE;
      }
      else
         return FALSE;
   }
}

//--------------------------------------------------------------------------
// Add spaces to the provided character string.  Add 3 * depth spaces.
//
// Returns:  number of characters added not including zero termination
//
int indent(char *msg, int depth)
{
   int msgcnt = 0, i;

   // indent number of depth
   for (i = 0; i < depth; i++)
      msgcnt += sprintf(msg + msgcnt,"   ");

   return msgcnt;
}

//--------------------------------------------------------------------------
// Read the DD_RomData MIB object and append to the 'msg' buffer in
// normal ROM print method (ls byte first).
//
// Returns:  number of characters added not including zero termination
//
int rom(char *msg)
{
   int msgcnt = 0, i;
   uchar buf[8];

   if (ReadFromMIB(DD_RomData, buf, 8) == 8)
   {
      msgcnt += sprintf(msg + msgcnt,"(");
      // loop to print the rom in reverse order
      for (i = 7; i >= 0; i--)
         msgcnt += sprintf(msg + msgcnt,"%02X",buf[i]);
      msgcnt += sprintf(msg + msgcnt,")");
   }

   return msgcnt;
}

